#include "TeDecoderMrSID.h"
#include "TeMrSID.h"
#include "TeDecoderMemory.h"
#include "TeVectorRemap.h"
#include "TeAsciiFile.h"
#include "TeGeometryAlgorithms.h"


TeDecoderMrSID::TeDecoderMrSID(const TeRasterParams& par) :
	mrsid_(0),
	nSelectedBlocks_(0),
	data_(0),
	dX_(0),
	dY_(0)
{
	params_ = par; 
	params_.decoderIdentifier_= "MrSID";
	try
	{
		mrsid_ = new TeMrSIDReader(params_.fileName_.c_str());
	}
	catch(/*exception& e*/...)
	{
		return;
	}
	fillRasterParams();
	originalLevel_ = currentLevel_;
}

TeDecoderMrSID::TeDecoderMrSID(const std::string& fname):
	mrsid_(0),
	nSelectedBlocks_(0),
	data_(0),
	dX_(0),
	dY_(0)
{
	try
	{
		mrsid_ = new TeMrSIDReader(params_.fileName_.c_str());
	}
	catch(...)
	{
		return;
	}
	fillRasterParams();
}

TeProjection* 
TeDecoderMrSID::readProjection()
{
	TeProjection* proj = 0;
	try
	{
		string projFile = TeGetName(params_.fileName_.c_str()) + ".prj";
		TeAsciiFile	pFile (projFile);
		string name;
		string pname, dname;
		int zone;
		TeProjectionParams projPar;
		while (pFile.isNotAtEOF())
		{
			name = pFile.readString();
			if (name == "Projection")
			{
				pname = pFile.readString();
			}	
			else if (name == "Datum")
			{
				name = pFile.readString();
				if (name == "SOUTH_AMERICAN_1969")
					dname = "SAD69";
				if (name == "WGS84")
					dname = "WGS84";

			}
			else if (name == "Zone")
			{
				zone = pFile.readInt();
			}
		}
		if (pname == "UTM" && (dname == "SAD69" || "WGS84"))
		{
			TeDatum d = TeDatumFactory::make(dname);
			proj = new TeUtm(d,zone*TeCDR);
		}
	}
	catch(...)
	{
	}
	if (!proj)
		proj = new TeNoProjection();
	return proj;
}

void TeDecoderMrSID::fillRasterParams()
{
	params_.projection(readProjection());
	params_.nBands(mrsid_->nBands());
	params_.setDataType(TeUNSIGNEDCHAR);
	if(mrsid_->getColorModel() == TeMrSIDReader::ColorModelRGB)
		params_.setPhotometric(TeRASTERRGB);
	else
		params_.setPhotometric(TeRASTERMULTIBAND);

	// retrieves the original resolution (level 0) of MrSID
	double oResx_ = mrsid_->resX();
	double oResy_ = -mrsid_->resY();

	// calculates the proper MrSID level accordingly to the asked TerraLib resolution
	// lowest level where resolution is equal or greater than the asked resolution
	double dResx_ = oResx_* params_.resolution_;
	double dResy_ = oResy_* params_.resolution_;
	currentLevel_ = 0;

	while (oResx_ < dResx_ && oResy_ < dResy_ && currentLevel_ < mrsid_->nlev())
	{
		currentLevel_++;
		mrsid_->zoomTo(currentLevel_);
		mrsid_->getCurrentLevelResolution(oResx_, oResy_);
	}
	zoomToLevel(currentLevel_);
	if (params_.blockHeight_ == 0 || params_.blockWidth_ != params_.ncols_)
	{
		params_.blockWidth_ = params_.ncols_;
		params_.blockHeight_ = nLinesInBlock();
	}
}


void TeDecoderMrSID::init()
{
	params_.status_= TeNOTREADY;
	if (params_.mode_ != 'r')	
		return;

	if (mrsid_)
	{
		string curFile(mrsid_->fileName());
		if (curFile.empty() || curFile != params_.fileName_)
		{
			delete mrsid_;
			mrsid_ = 0;
		}
	}

	if (!mrsid_)
	{
		try
		{
			mrsid_ = new TeMrSIDReader(params_.fileName_.c_str());
		}
		catch(...)
		{
			return;
		}
	}
	
	if (data_)
	{
		delete data_;
		data_ = 0;
	}
	fillRasterParams();
	originalLevel_ = currentLevel_;
	if (!allocateMemory())
		return;
	params_.status_= TeREADYTOREAD;
	return;
}

bool 
TeDecoderMrSID::allocateMemory()
{
	curStrip_ = -1;
	if (data_)
		delete []data_;
	data_ = 0;
	data_ = new unsigned char[params_.blockWidth_ * params_.blockHeight_ * params_.nBands()];
	return (data_ != 0);
}

bool 
TeDecoderMrSID::clear()
{
	if (mrsid_)
		mrsid_->clear();
	if (data_)
		delete data_;
	data_ = 0;
	dX_ = dY_ = 0;
	curStrip_ = -1;
	nSelectedBlocks_ = 0;
	params_.status_= TeNOTREADY;
	return true;
}

bool
TeDecoderMrSID::getElement(int col, int lin, double& val, int band)
{
	if ((unsigned int)(lin/params_.blockHeight_) != (unsigned int)(curStrip_/params_.blockHeight_))
	{
		curStrip_ = ((unsigned int)(lin/params_.blockHeight_))*params_.blockHeight_;
		if (!loadStrip())
			return false;
	}
	unsigned int position  = params_.nBands()*(params_.ncols_*(lin-curStrip_)+col)+band;
	val = (unsigned char) data_[position];
	return true;
}

bool TeDecoderMrSID::loadLine(int row, unsigned char* buf, int nlines)
{
	try
	{	
		if (!mrsid_->selectArea(dX_,row+dY_,params_.blockWidth_,nlines))
			return false;
		mrsid_->getSelectedArea(buf);
	}
	catch(...)
	{
		return false;
	}
	return true;
}


bool
TeDecoderMrSID::zoomToLevel(int level)
{
	if (level < 0 || level >= mrsid_->nlev())
		return false;

	mrsid_->zoomTo(level);
	int ncols, nlines;
	mrsid_->getDimensionAtLevel(level, ncols, nlines);
	double x0,y0,x1,y1;
	mrsid_->getCurrentBoundingBox(x0,y0,x1,y1);
	params_.boundingBoxLinesColumns(x0,y0,x1,y1,nlines,ncols);
	params_.resolution_ = (int)(params_.resx_/mrsid_->resX());
	return true;
}

int 
TeDecoderMrSID::bestResolution(TeBox& bb, int ncols, int nlines, TeProjection* proj)
{
	if (!mrsid_)
		return -1;

	TeBox box = bb;
	if (proj && !(*proj==*params_.projection()))
		box = TeRemapBox(bb, proj, params_.projection());
	
	double resx = box.width()/ncols;		// desired resolution
	double resy = box.height()/nlines;
	double res = MIN(resx,resy);

	int prevLevel = mrsid_->getCurrentLevel();			// save current level

	int nlev = mrsid_->nlev();
	int  sidscale_ = 0;
	double mrsidresx, mrsidresy, mrsidres;
	do 
	{
		mrsid_->zoomTo(sidscale_);
		mrsid_->getCurrentLevelResolution(mrsidresx, mrsidresy);
		mrsidres = std::min(mrsidresx, mrsidresy);
		sidscale_++;
	}while (res >= mrsidres && sidscale_< nlev);
	
	mrsid_->zoomTo(prevLevel);
	return sidscale_-1;
}

bool TeDecoderMrSID::selectBlocks(TeBox& bbox, int resLevel, TeRasterParams& parBlock) 
{
	if (!mrsid_)
		return false;

	// check if desired resolution level is between the range of available levels
	if (resLevel < 0 || resLevel >= mrsid_->nlev())
		return false;

	nSelectedBlocks_ = 0;
	int prevLevel =  mrsid_->getCurrentLevel();				// save current level

	// zoom to level and defines clipping area
	zoomToLevel(resLevel);

	// calculates width and height of the selected area
	double bw = bbox.width();
	double bh = bbox.height();
	int w = TeRound(bbox.width()/params_.resx_);
	int h = TeRound(bbox.height()/params_.resy_);

	TeCoord2D ul = coord2Index(TeCoord2D(bbox.x1_,bbox.y2_));
	int ulx = TeRoundRasterIndex(ul.x_);
	int uly = TeRoundRasterIndex(ul.y_);

	if (!mrsid_->selectArea(ulx,uly,w,h))
	{
		zoomToLevel(prevLevel);
		return false;
	}
	params_.blockWidth_ = w;
	params_.blockHeight_ = h;
	
	// defines parameters of the selected block
	nSelectedBlocks_ = 1;			// always select only one block
	parBlock = params_;
	parBlock.boundingBoxResolution(bbox.x1_,bbox.y1_,bbox.x2_,bbox.y2_,parBlock.resx_,parBlock.resy_);
	parBlock.interleaving_ = TePerPixel;
	return true;
}

bool TeDecoderMrSID::getSelectedRasterBlock(TeDecoderMemory* memDec) 
{
  if (!mrsid_ || nSelectedBlocks_ <= 0)
	  return false;
  bool res = mrsid_->getSelectedArea((unsigned char*) memDec->data(0));
  nSelectedBlocks_--;
  return res;
}

void 
TeDecoderMrSID::clearBlockSelection()
{	
	zoomToLevel(originalLevel_);
	nSelectedBlocks_ = 0;	
}

int 
TeDecoderMrSID::nLinesInBlock()
{
	int nlines = 1;
	while (nlines*params_.ncols_*params_.nBands() < 200*1024*1024 && nlines < params_.nlines_)
		nlines *= 2;
	nlines /= 2;
	return nlines;
}

void
TeDecoderMrSID::setClippingArea(TeBox clipBox)
{
	zoomToLevel(originalLevel_);
	TeBox clipBoxInter;
	if (TeIntersection(params_.boundingBox(),clipBox,clipBoxInter))		//if there where a given box set in parameters
	{
		int nlines = params_.nlines_;
		clipBox = params_.boundingBox();
		params_.boundingBoxResolution(clipBoxInter.x1_,clipBoxInter.y1_,
								      clipBoxInter.x2_,clipBoxInter.y2_,
								      params_.resx_,params_.resy_);	
	
		dX_ = static_cast<int>((params_.boundingBox().x1_ - clipBox.x1_)/params_.resx_);
		dY_ = nlines - static_cast<int>((params_.boundingBox().y1_ - clipBox.y1_)/params_.resy_)
			  - params_.nlines_;
	}
	params_.blockWidth_ = params_.ncols_;
	params_.blockHeight_ = nLinesInBlock();
	allocateMemory();
}

void 
TeDecoderMrSID::resetClippingArea()
{
	zoomToLevel(originalLevel_);
	params_.blockWidth_ = params_.ncols_;
	params_.blockHeight_ = nLinesInBlock();
	allocateMemory();
}

bool
TeDecoderMrSID::loadStrip()
{
	try
	{	
		//(int leftColumn, int topRow, int width, int height)
		if (!mrsid_->selectArea(dX_,curStrip_+dY_,params_.blockWidth_, params_.blockHeight_))
			return false;
		mrsid_->getSelectedArea(data_);
	}
	catch(...)
	{
		return false;
	}
	return true;
}
