/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/
/*! \file TePlotTheme.h
    This file contains support to plot a theme on a canvas
*/
#ifndef  __TERRALIB_INTERNAL_PLOTTHEME_H
#define  __TERRALIB_INTERNAL_PLOTTHEME_H

#include "TeThemeApplication.h"
#include "TeDatabase.h"
#include "TeGeometryAlgorithms.h"
#include "TeVectorRemap.h"
#include "TeDecoderMemory.h"
#include <time.h>
#include <set>
#include "TeApplicationUtils.h"

template<class Canvas, class Progress>
void TePlotTextWV(TeThemeApplication* theme, Canvas* canvas, Progress * progress, TeBox box = TeBox())
{
	if ((theme->visibleRep() & TeTEXT) == 0)
		return;

	if(theme->textTable().empty())
		return;
	TeDatabase* db = theme->layer()->database();
	if(db->tableExist(theme->textTable()) == false)
		return;
	canvas->setDataProjection (theme->layer()->projection());
	TeDatabasePortal* plot = db->getPortal();

	string	tabtx = theme->textTable();
	string	tabs = theme->collectionTable();

	string wherebox;
	if(box.isValid())
	{
		wherebox = "NOT (x >= " + Te2String(box.x2()+4*box.width());
		wherebox += " OR y >= " + Te2String(box.y2()+2*box.height());
		wherebox += " OR x <= " + Te2String(box.x1()-4*box.width());
		wherebox += " OR y <= " + Te2String(box.y1()-2*box.height()) + ")";
	}
	else
	{
		wherebox = getWhereBox(plot, theme, canvas, TeTEXT, box);
		if(wherebox.empty())
		{
			TeBox box = canvas->getDataWorld();
			wherebox = " x >= " + Te2String(box.x1());
			wherebox += " AND x <= " + Te2String(box.x2());
			wherebox += " AND y >= " + Te2String(box.y1());
			wherebox += " AND y <= " + Te2String(box.y2());
		}
	}

	// calculate initial filed number for collection and legenda (sind and vind)
	int sind=0, vind=0;
	string ind = "SELECT * FROM " + tabtx + " WHERE object_id < ' '";
	plot->freeResult();
	if (plot->query (ind))
		sind = plot->AttributeList().size();

	ind = "SELECT * FROM " + tabs + " WHERE c_object_id < ' '";
	plot->freeResult();
	if (plot->query (ind))
		vind = plot->AttributeList().size() + sind;

	int numrows = 0;
	string conta = "SELECT COUNT(*) FROM " + tabs;
	plot->freeResult();
	if (plot->query (conta))
	{
		if(plot->fetchRow())
			numrows = atoi(plot->getData(0));
	}
	delete plot;

	string tabv = tabtx + "_txvisual";
	string sa = "SELECT " + tabtx + ".*, " + tabv + ".*, " + tabs + ".* ";
	sa += "FROM " + tabtx + ", " + tabv + ", " + tabs;
	sa += " WHERE (" + tabs + ".c_object_id = " + tabtx + ".object_id";
	sa += " AND " + tabtx + ".geom_id = " + tabv + ".geom_id)";
	string squery;
	squery = sa + " AND (" + wherebox + ")";
	plotTextWV(theme, canvas, squery, progress);

	string othersQuery = "SELECT " + tabtx + ".*, " + tabv + ".* ";
	othersQuery += "FROM " + tabtx + ", " + tabv;
	othersQuery += " WHERE " + tabtx + ".geom_id = " + tabv + ".geom_id";
	othersQuery += " AND " + tabtx + ".object_id = 'TE_NONE_ID'";
	plotTextWV(theme, canvas, othersQuery, progress);
}

template<class Canvas, class Progress>
void plotTextWV(TeThemeApplication *theme, Canvas *canvas, string query, Progress *progress)
{
	TeText	text;
	int	np = 0;
	int dt = CLOCKS_PER_SEC/4;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;
	TeDatabasePortal* plot = theme->layer()->database()->getPortal();

	if(progress == 0)
	{
		if(plot->query(query) && plot->fetchRow())
		{
			t1 = clock();
			t2 = t1;
			bool flag = true;
			do
			{
				bool bold = false, italic = false;
				int	size;
				int dot_height = atoi(plot->getData(10));
				bool fixedSize = atoi(plot->getData(11));
				int rgb = atoi(plot->getData(12));
				TeColor	cor((rgb >> 16) & 0xff, (rgb >> 8) & 0xff, rgb & 0xff);
				string	family = plot->getData(13);
				if(atoi(plot->getData(14)))
					bold = true;
				if(atoi(plot->getData(15)))
					italic = true;

				flag = plot->fetchGeometry(text);
				if(fixedSize == false && text.height() > 0.)
				{
					TeBox wbox = canvas->getDataWorld();
					TeCoord2D p1(wbox.x1_, wbox.y1_);
					TeCoord2D p2(p1.x() + text.height(), p1.y() + text.height());
					TeBox box(p1, p2);
					canvas->mapWtoV(box);
					size = int(box.height());
				}
				else
					size = dot_height;

				canvas->setTextColor(cor.red_, cor.green_, cor.blue_);
				canvas->setTextStyle (family, size, bold, italic );
				TeVisual visual;
				visual.size(size);
				visual.family(family);
				visual.bold(bold);
				visual.italic(italic);
				visual.fixedSize(fixedSize);
				visual.ptAngle((int)text.angle());

				canvas->plotText (text, visual);
				np++;
				t2 = clock();
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					canvas->copyPixmapToWindow();
				}
			} while (flag);
			canvas->copyPixmapToWindow();
		}
	}
	else
	{
		int steps = 0;
		string conta = query;
		int ff;
		if((ff=conta.find(" FROM ")) >= 0)
			conta.replace(0, ff, "SELECT COUNT(*)");
		if((ff=conta.find("ORDER BY")) >= 0)
			conta.erase(ff, conta.size()-ff);
		plot->freeResult();
		if(plot->query(conta) && plot->fetchRow())
			steps = atoi(plot->getData(0));

		plot->freeResult();
		if(plot->query(query) && plot->fetchRow())
		{
			string caption = "Plotando tema: " + theme->name();
			progress->setCaption(caption.c_str());
			progress->setLabelText("      Plotagem de textos em andamento, aguarde...      ");
			progress->setTotalSteps(steps);
			t2 = clock();
			t0 = t1 = t2;
			bool flag = true;
			do
			{
				bool bold = false, italic = false;
				int	size;
				int dot_height = atoi(plot->getData(10));
				bool fixedSize = atoi(plot->getData(11));
				int rgb = atoi(plot->getData(12));
				TeColor	cor((rgb >> 16) & 0xff, (rgb >> 8) & 0xff, rgb & 0xff);
				string	family = plot->getData(13);
				if(atoi(plot->getData(14)))
					bold = true;
				if(atoi(plot->getData(15)))
					italic = true;

				flag = plot->fetchGeometry(text);
				if(fixedSize == false && text.height() > 0.)
				{
					TeBox wbox = canvas->getDataWorld();
					TeCoord2D p1(wbox.x1_, wbox.y1_);
					TeCoord2D p2(p1.x() + text.height(), p1.y() + text.height());
					TeBox box(p1, p2);
					canvas->mapWtoV(box);
					size = int(box.height());
				}
				else
					size = dot_height;

				canvas->setTextColor(cor.red_, cor.green_, cor.blue_);
				canvas->setTextStyle (family, size, bold, italic );
				TeVisual visual;
				visual.size(size);
				visual.family(family);
				visual.bold(bold);
				visual.italic(italic);
				visual.fixedSize(fixedSize);
				visual.ptAngle((int)text.angle());

				canvas->plotText (text, visual);
				np++;
				t2 = clock();
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					canvas->copyPixmapToWindow();
					if(progress->wasCancelled())
					{
						progress->reset();
						break;
					}
					if((int)(t2-t0) > dt2)
						progress->setProgress(np);
				}
			} while (flag);
			canvas->copyPixmapToWindow();
		}
		progress->reset();
	}
	delete plot;
}

template<class Canvas, class Progress>
void TePlotTexts(TeThemeApplication* theme, Canvas* canvas, Progress * progress)
{
	if ((theme->visibleRep() & TeTEXT) == 0)
		return;

	if(theme->textTable().empty())
		return;
	TeDatabase* db = theme->layer()->database();
	if(db->tableExist(theme->textTable()) == false)
		return;
	canvas->setDataProjection (theme->layer()->projection());
	TeDatabasePortal* plot = db->getPortal();

	string	tabtx = theme->textTable();
	string	tabs = theme->collectionTable();

	string wherebox = getWhereBox(plot, theme, canvas, TeTEXT);
	if(wherebox.empty())
	{
		TeBox box = canvas->getDataWorld();
		wherebox = " x >= " + Te2String(box.x1());
		wherebox += " AND x <= " + Te2String(box.x2());
		wherebox += " AND y >= " + Te2String(box.y1());
		wherebox += " AND y <= " + Te2String(box.y2());
	}

	// calculate initial filed number for collection and legenda (sind and vind)
	int sind=0, vind=0;
	string ind = "SELECT * FROM " + tabtx + " WHERE object_id < ' '";
	plot->freeResult();
	if (plot->query (ind))
		sind = plot->AttributeList().size();

	ind = "SELECT * FROM " + tabs + " WHERE c_object_id < ' '";
	plot->freeResult();
	if (plot->query (ind))
		vind = plot->AttributeList().size() + sind;

	int numrows = 0;
	string conta = "SELECT COUNT(*) FROM " + tabs;
	plot->freeResult();
	if (plot->query (conta))
	{
		if(plot->fetchRow())
			numrows = atoi(plot->getData(0));
	}
	string sa;
	string sel;
	if(plot->getDatabase()->dbmsName() == "OracleSpatial")	
		sel = " SELECT /*+ ordered */ * "; 
	else
		sel = " SELECT * ";

	delete plot;
	if(numrows > 0)
		sa = sel + " FROM (" + tabtx + " INNER JOIN " + tabs;
	else
		sa = sel + " FROM (" + tabtx + " LEFT JOIN " + tabs;

	sa += " ON " + tabtx + ".object_id = " + tabs + ".c_object_id) WHERE ";

	string squery;
	squery = sa + wherebox;
	plotTexts(theme, canvas, squery, progress);

	string othersQuery = "SELECT * FROM " + tabtx + " WHERE object_id = 'TE_NONE_ID'";
	plotTexts(theme, canvas, othersQuery, progress);
}

template<class Canvas, class Progress>
void plotTexts(TeThemeApplication *theme, Canvas *canvas, string query, Progress *progress)
{
	TeText	text;
	int	np = 0;
	int dt = CLOCKS_PER_SEC/4;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;
	TeDatabasePortal* plot = theme->layer()->database()->getPortal();

	if(progress == 0)
	{
		if(plot->query(query) && plot->fetchRow())
		{
			t1 = clock();
			t2 = t1;
			bool flag = true;
			do
			{
				flag = plot->fetchGeometry(text);
				setTextAttributes(theme, canvas, text);
				canvas->plotText (text, theme->defaultLegend().visual(TeTEXT));
				np++;
				t2 = clock();
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					canvas->copyPixmapToWindow();
				}
			} while (flag);
			canvas->copyPixmapToWindow();
		}
	}
	else
	{
		int steps = 0;
		string conta = query;
		conta.replace(0, strlen("SELECT *")+1, "SELECT COUNT(*)");
		int ff;
		if((ff=conta.find("ORDER BY")) >= 0)
			conta.erase(ff, conta.size()-ff);
		plot->freeResult();
		if(plot->query(conta) && plot->fetchRow())
			steps = atoi(plot->getData(0));

		plot->freeResult();
		if(plot->query(query) && plot->fetchRow())
		{
			string caption = "Plotando tema: " + theme->name();
			progress->setCaption(caption.c_str());
			progress->setLabelText("      Plotagem de textos em andamento, aguarde...      ");
			progress->setTotalSteps(steps);
			t2 = clock();
			t0 = t1 = t2;
			bool flag = true;
			do
			{
				flag = plot->fetchGeometry(text);
				setTextAttributes(theme, canvas, text);
				canvas->plotText (text, theme->defaultLegend().visual(TeTEXT));
				np++;
				t2 = clock();
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					canvas->copyPixmapToWindow();
					if(progress->wasCancelled())
					{
						progress->reset();
						break;
					}
					if((int)(t2-t0) > dt2)
						progress->setProgress(np);
				}
			} while (flag);
			canvas->copyPixmapToWindow();
		}
		progress->reset();
	}
	delete plot;
}

template<class Canvas>
void setTextAttributes(TeThemeApplication* theme, Canvas* canvas, TeText tx)
{
	TeColor	cor;
	static	TeVisual snc;
	string	family;
	static int		ssize;
	int		size = 0;
	bool	bold, italic, fixedSize;

	if(theme == 0)
	{
		ssize = 0;
		snc.family("notFamily");
		return;
	}

	fixedSize = theme->defaultLegend().visual(TeTEXT).fixedSize(); // font size is fixed
	if(snc == theme->defaultLegend().visual(TeTEXT))
	{
		if(fixedSize == false && tx.height() > 0.)
		{
			TeBox wbox = canvas->getDataWorld();
			TeCoord2D p1(wbox.x1_, wbox.y1_);
			TeCoord2D p2(p1.x() + tx.height(), p1.y() + tx.height());
			TeBox box(p1, p2);
			canvas->mapWtoV(box);
			size = int(box.height());
		}
		else
			size = theme->defaultLegend().visual(TeTEXT).size(); // font size
	}

	if(size == 0)
	{
		if(fixedSize == false && tx.height() > 0.)
		{
			TeBox wbox = canvas->getDataWorld();
			TeCoord2D p1(wbox.x1_, wbox.y1_);
			TeCoord2D p2(p1.x() + tx.height(), p1.y() + tx.height());
			TeBox box(p1, p2);
			canvas->mapWtoV(box);
			size = int(box.height());
		}
		else
			size = theme->defaultLegend().visual(TeTEXT).size(); // font size
	}

	cor = theme->defaultLegend().visual(TeTEXT).color();
	family = theme->defaultLegend().visual(TeTEXT).family();
	bold = theme->defaultLegend().visual(TeTEXT).bold();
	italic = theme->defaultLegend().visual(TeTEXT).italic();
	fixedSize = theme->defaultLegend().visual(TeTEXT).fixedSize(); // font size is fixed

	canvas->setTextColor(cor.red_, cor.green_, cor.blue_);
	canvas->setTextStyle (family, size, bold, italic );
	snc = theme->defaultLegend().visual(TeTEXT);
	ssize = size;
}

template<class Canvas, class Progress>
void plotPieBars(TeThemeApplication *theme, Canvas *canvas, Progress *progress, TeBox ebox = TeBox())
{
	if((theme->visibleRep() & 0x80000000) == 0)
		return;

	int steps = 0;
	int	np = 0;
	int dt = CLOCKS_PER_SEC/4;
	int dt2 = CLOCKS_PER_SEC * 2;
	clock_t	t0, t1, t2;
	TeDatabasePortal* plot = theme->layer()->database()->getPortal();
	canvas->setDataProjection (theme->layer()->projection());

	if(theme->chartAttributes_.size() == 0)
	{
		delete plot;
		return;
	}
	double	width = theme->barWidth();
	double	maxh = theme->barMaxHeight();
	double	minh = theme->barMinHeight();
	double	maxd = theme->pieMaxDiameter();
	double	mind = theme->pieMinDiameter();
	double	diameter = theme->pieDiameter();
	string FUNC = theme->chartFunction();

	if(theme->keepDimension() != 0)
	{
		width = canvas->mapVtoData((int)width);
		maxh = canvas->mapVtoData((int)maxh);
		minh = canvas->mapVtoData((int)minh);
		maxd = canvas->mapVtoData((int)maxd);
		mind = canvas->mapVtoData((int)mind);
		diameter = canvas->mapVtoData((int)diameter);
	}

	unsigned int i;
	string	input, cinput;
	TeSelectedObjects  selectedObjects = theme->chartObjects();
	
	if(selectedObjects == TeSelectedByPointing)
	{
		input = "(grid_status = 1 OR grid_status = 3";
		input += " OR (grid_status is null AND (c_object_status = 1 OR c_object_status = 3)))";
		cinput = "(c_object_status = 1 OR c_object_status = 3)";
	}
	else if(selectedObjects == TeNotSelectedByPointing)
	{
		input = "(grid_status = 0 OR grid_status = 2";
		input += " OR (grid_status is null AND (c_object_status = 0 OR c_object_status = 2)))";
		cinput = "(c_object_status = 0 OR c_object_status = 2)";
	}
	else if(selectedObjects == TeSelectedByQuery)
	{
		input = "(grid_status = 2 OR grid_status = 3";
		input += " OR (grid_status is null AND (c_object_status = 2 OR c_object_status = 3)))";
		cinput = "(c_object_status = 2 OR c_object_status = 3)";
	}
	else if(selectedObjects == TeNotSelectedByQuery)
	{
		input = "(grid_status = 0 OR grid_status = 1";
		input += " OR (grid_status is null AND (c_object_status = 0 OR c_object_status = 1)))";
		cinput = "(c_object_status = 0 OR c_object_status = 1)";
	}
	else if(selectedObjects == TeGrouped)
	{
		input = "c_legend_id <> 0";
		cinput = "c_legend_id <> 0";
	}
	else if(selectedObjects == TeNotGrouped)
	{
		input = "c_legend_id = 0";
		cinput = "c_legend_id = 0";
	}

	canvas->setLineStyle(TeLnTypeContinuous, 0);
	canvas->setLineColor(0, 0, 0);
	TeChartType chartType = (TeChartType)theme->chartType();
	if(chartType == TePieChart)
	{
		string wherebox;
		double delta = diameter / 2.;
		if(!(theme->pieDimAttribute() == "NONE"))
			delta = maxd / 2.;

		TeBox box;
		if(ebox.isValid())
			box = ebox;
		else
			box = canvas->getDataWorld();

		box.x1_ = box.x1_ - delta;
		box.x2_ = box.x2_ + delta;
		box.y1_ = box.y1_ - delta;
		box.y2_ = box.y2_ + delta;

		wherebox = "NOT(label_x > " + Te2String(box.x2_, 10) + " OR ";
		wherebox = wherebox + "label_x < " + Te2String(box.x1_, 10) + " OR ";
		wherebox = wherebox + "label_y > " + Te2String(box.y2_, 10) + " OR ";
		wherebox = wherebox + "label_y < " + Te2String(box.y1_, 10) + ")";

		string attrs, q;
		for(i=0; i<theme->chartAttributes_.size(); i++)
			attrs += FUNC + "(" + theme->chartAttributes_[i] + "),";
		attrs += "AVG(label_x),AVG(label_y)";
		if(!(theme->pieDimAttribute() == "NONE"))
			attrs += "," + FUNC + "(" + theme->pieDimAttribute() + ")";

		if(wherebox.empty() && input.empty())
			q = "SELECT " + attrs + theme->sqlGridFrom();
		else if(wherebox.empty() && input.empty()==false)
			q = "SELECT " + attrs + theme->sqlGridFrom() + " WHERE " + input;
		else if(wherebox.empty()==false && input.empty())
			q = "SELECT " + attrs + theme->sqlGridFrom() + " WHERE " + wherebox;
		else if(wherebox.empty()==false && input.empty()==false)
			q = "SELECT " + attrs + theme->sqlGridFrom() + " WHERE " + input + " AND " + wherebox;
		q += " GROUP BY " + theme->collectionTable() + ".c_object_id";

		string conta = "SELECT COUNT(*) FROM " + theme->collectionTable();
		if(wherebox.empty() && cinput.empty()==false)
			conta += " WHERE " + cinput;
		else if(wherebox.empty()==false && cinput.empty())
			conta += " WHERE " + wherebox;
		else if(wherebox.empty()==false && cinput.empty()==false)
			conta += " WHERE " + cinput + " AND " + wherebox;
		
		plot->freeResult();
		if(plot->query(conta) && plot->fetchRow())
			steps = atoi(plot->getData(0));

		plot->freeResult();
		if(plot->query(q))
		{
			if(progress)
			{
				string caption = "Plotando tema: " + theme->name();
				progress->setCaption(caption.c_str());
				progress->setLabelText("      Plotagem de tortas em andamento, aguarde...      ");
				progress->setTotalSteps(steps);
			}
			t2 = clock();
			t0 = t1 = t2;
			bool b = plot->fetchRow(0);
			while(b)
			{
				double tot = 0.;
				for(i=0; i<theme->chartAttributes_.size(); i++)
				{
					string v = plot->getData(i);
					if(v.empty())
					{
						tot = 0.;
						break;
					}
					tot += atof(v.c_str());
				}
				if(tot == 0.)
				{
					b = plot->fetchRow();
					np++;
					continue;
				}

				if(!(theme->pieDimAttribute() == "NONE"))
				{
					if(theme->chartMaxVal() - theme->chartMinVal() == 0)
						diameter = 0.;
					else
					{
						double	adim = atof(plot->getData(theme->chartAttributes_.size()+2));
						double a = (adim - theme->chartMinVal()) / (theme->chartMaxVal() - theme->chartMinVal());
						diameter = a * (maxd - mind) + mind;
					}
				}
				double	x = atof(plot->getData(theme->chartAttributes_.size())) - diameter / 2.;
				double	y = atof(plot->getData(theme->chartAttributes_.size()+1)) - diameter / 2.;

				double	a = 0.;
				for(i=0; i<theme->chartAttributes_.size(); i++)
				{
					double b = atof(plot->getData(i)) / tot * 360.;
					TeColor	cor = theme->chartColors_[i];
					canvas->setPieColor(cor.red_, cor.green_, cor.blue_);
					canvas->plotPie(x, y, diameter, diameter, a, b);
					a += b;
				}
				np++;
				t2 = clock();
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					canvas->copyPixmapToWindow();
					if(progress)
					{
						if(progress->wasCancelled())
						{
							progress->reset();
							break;
						}
						if((int)(t2-t0) > dt2)
							progress->setProgress(np);
					}
				}
				b = plot->fetchRow();
			}
			canvas->copyPixmapToWindow();
		}
	}
	else
	{
		double	n = theme->chartAttributes_.size();
		double	maxv = theme->chartMaxVal();
		double	minv = theme->chartMinVal();
		double delta = maxh / 2.;

		string wherebox;

		TeBox box;
		if(ebox.isValid())
			box = ebox;
		else
			box = canvas->getDataWorld();

		box.x1_ = box.x1_ - delta;
		box.x2_ = box.x2_ + delta;
		box.y1_ = box.y1_ - delta;
		box.y2_ = box.y2_ + delta;

		wherebox = "NOT(label_x > " + Te2String(box.x2_, 10) + " OR ";
		wherebox = wherebox + "label_x < " + Te2String(box.x1_, 10) + " OR ";
		wherebox = wherebox + "label_y > " + Te2String(box.y2_, 10) + " OR ";
		wherebox = wherebox + "label_y < " + Te2String(box.y1_, 10) + ")";

		string attrs, q;
		for(i=0; i<theme->chartAttributes_.size(); i++)
			attrs += FUNC + "(" + theme->chartAttributes_[i] + "),";
		attrs += "AVG(label_x),AVG(label_y)";

		if(wherebox.empty() && input.empty())
			q = "SELECT " + attrs + theme->sqlGridFrom();
		else if (wherebox.empty() == false && input.empty() == false)
			q = "SELECT " + attrs + theme->sqlGridFrom() + " WHERE " + input + " AND " + wherebox;
		else if (wherebox.empty() == false && input.empty())
			q = "SELECT " + attrs + theme->sqlGridFrom() + " WHERE " + wherebox;
		else if (wherebox.empty() && input.empty() == false)
			q = "SELECT " + attrs + theme->sqlGridFrom() + " WHERE " + input;
		q += " GROUP BY " + theme->collectionTable() + ".c_object_id";

		string conta = "SELECT COUNT(*) FROM " + theme->collectionTable();
		if(wherebox.empty() && cinput.empty()==false)
			conta += " WHERE " + cinput;
		else if(wherebox.empty()==false && cinput.empty())
			conta += " WHERE " + wherebox;
		else if(wherebox.empty()==false && cinput.empty()==false)
			conta += " WHERE " + cinput + " AND " + wherebox;

		plot->freeResult();
		if(plot->query(conta) && plot->fetchRow())
			steps = atoi(plot->getData(0));

		vector<double> dvec;
		plot->freeResult();
		if(plot->query(q))
		{
			if(progress)
			{
				string caption = "Plotando tema: " + theme->name();
				progress->setCaption(caption.c_str());
				progress->setLabelText("      Plotagem de barras em andamento, aguarde...      ");
				progress->setTotalSteps(steps);
			}
			t2 = clock();
			t0 = t1 = t2;
			bool b = plot->fetchRow(0);
			while(b)
			{
				double	x = atof(plot->getData(theme->chartAttributes_.size())) - n * width / 2.;
				double	y = atof(plot->getData(theme->chartAttributes_.size()+1));

				dvec.clear();
				for(i=0; i<theme->chartAttributes_.size(); i++)
				{
					string val = plot->getData(i);
					if(val.empty())
					{
						dvec.clear();
						break;
					}
					dvec.push_back(atof(val.c_str()));
				}

				for(i=0; i<dvec.size(); i++)
				{
					double height;
					double v = dvec[i];
					if(maxv - minv == 0)
						height = 0.;
					else
						height = (v - minv) * (maxh - minh) / (maxv - minv) + minh;
					TeColor	cor = theme->chartColors_[i];
					canvas->setRectColor(cor.red_, cor.green_, cor.blue_);
					canvas->plotRect(x, y, width, height);
					x += width;
				}
				np++;
				t2 = clock();
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					canvas->copyPixmapToWindow();
					if(progress)
					{
						if(progress->wasCancelled())
						{
							progress->reset();
							break;
						}
						if((int)(t2-t0) > dt2)
							progress->setProgress(np);
					}
				}
				b = plot->fetchRow();
			}
			canvas->copyPixmapToWindow();
		}
	}
	if(progress)
		progress->reset();
	delete plot;
}


template<class Portal, class Canvas>
string getWhereBox(Portal *plot, TeThemeApplication *theme, Canvas *canvas, TeGeomRep rep, TeBox bbox = TeBox())
{
	string wherebox;
	if(theme->chartSelected_.empty() == true)
	{
		TeBox box;
		if(bbox.isValid())
			box = bbox;
		else
			box = canvas->getDataWorld ();
		wherebox = plot->getDatabase()->getSQLBoxWhere(box, rep);
	}
	else
	{
		if (rep == TePOINTS)
		{
			double	width = theme->barWidth();
			double	maxh = theme->barMaxHeight();
			double	minh = theme->barMinHeight();
			double	maxd = theme->pieMaxDiameter();
			double	mind = theme->pieMinDiameter();
			double	diameter = theme->pieDiameter();
			double	delta;

			if(theme->keepDimension() != 0)
			{
				width = canvas->mapVtoData((int)width);
				maxh = canvas->mapVtoData((int)maxh);
				minh = canvas->mapVtoData((int)minh);
				maxd = canvas->mapVtoData((int)maxd);
				mind = canvas->mapVtoData((int)mind);
				diameter = canvas->mapVtoData((int)diameter);
			}

			TeChartType chartType = (TeChartType)theme->chartType();
			if(chartType == TePieChart)
			{
				delta = diameter / 2.;
				if(!(theme->pieDimAttribute() == "NONE"))
					delta = maxd / 2.;
			}
			else
				delta = maxh / 2.;

			string sel = "SELECT label_x, label_y FROM " + theme->collectionTable();
			sel += " WHERE c_object_id = '" + theme->chartSelected_ + "'";
			plot->freeResult();
			if(plot->query(sel))
			{
				bool b = plot->fetchRow(0);
				if(b == false)
					return wherebox;
				if(b)
				{
					double x = atof(plot->getData(0));
					double y = atof(plot->getData(1));
					string xmin = Te2String(x - delta);
					string xmax = Te2String(x + delta);
					string ymin = Te2String(y - delta);
					string ymax = Te2String(y + delta);
					wherebox = "label_x > " + xmin + " AND ";
					wherebox += "label_x < " + xmax + " AND ";
					wherebox += "label_y > " + ymin + " AND ";
					wherebox += "label_y < " + ymax;
				}
			}
		}
		else if(rep == TePOLYGONS || rep == TeLINES || rep == TeCELLS)
		{
			string tab;
			if(rep == TePOLYGONS)
				tab = theme->layer()->tableName(TePOLYGONS);
			else if (rep == TeLINES)
				tab = theme->layer()->tableName(TeLINES);
			else if (rep == TeCELLS)
				tab = theme->layer()->tableName(TeCELLS);
			
			plot->freeResult();

			TeBox b;
			bool result = plot->getDatabase()->getMBRGeom(tab, theme->chartSelected_, b, "spatial_data");
			if(!result)
				return wherebox;
			else
				wherebox = plot->getDatabase()->getSQLBoxWhere(b, rep);
		}
	}
	return wherebox;
}

template<class Portal>
TeVisual getVisual(Portal* portal, TeThemeApplication* theme, TeGeomRep GEOMREP, int resPos, int groPos, int ownPos)
{
	TeVisual visual;
	TeDatabase* db = theme->layer()->database();
	TeLegendEntry* legendEntry;

	int status = atoi(portal->getData(resPos));
	int groId = atoi(portal->getData(groPos));
	int ownId = atoi(portal->getData(ownPos));

	if(groId && (theme->visibleRep()&0x40000000))
		legendEntry = db->legendMap()[groId];
	else if(ownId != 0)
		legendEntry = db->legendMap()[ownId];
	else
		legendEntry = &(theme->defaultLegend());
	
//	if(legendEntry)
//		visual = legendEntry->visual(GEOMREP);
//	else
//		visual = TeVisual(GEOMREP);
/////////////////////////////////////////////////////////
	TeGeomRepVisualMap vm = legendEntry->getVisualMap();
	if(legendEntry && vm.find(GEOMREP) != vm.end())
		visual = legendEntry->visual(GEOMREP);
	else
	{
		if(ownId != 0)
			legendEntry = db->legendMap()[ownId];
		else
			legendEntry = &(theme->defaultLegend());

		visual = legendEntry->visual(GEOMREP);
	}
/////////////////////////////////////////////////////////

	if(status == 1)
		visual.color(theme->pointingLegend().visual(TePOLYGONS).color());
	else if(status == 2)
		visual.color(theme->queryLegend().visual(TePOLYGONS).color());
	else if(status == 3)
		visual.color(theme->queryAndPointingLegend().visual(TePOLYGONS).color());

	return visual;
}

template<class Canvas , class Progress>
void TePlotObjects (TeThemeApplication *theme, Canvas *canvas, map<int, RepMap>& layerRepMap, Progress *progress = 0, TeBox box = TeBox())
{
	string tableName, objectId;
	TeColor color;
	TeVisual visual;
	int dt = CLOCKS_PER_SEC/4;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;
	int nSteps, nLoops;


	TeLayer *layer = theme->layer();
	TeDatabase *db = layer->database();
	TeDatabasePortal *portal = db->getPortal();
	canvas->setDataProjection (layer->projection());
	map<int, RepMap>::iterator it = layerRepMap.find(layer->id());

	TeBox vb = box;
	if(box.isValid())
	{
		canvas->mapDataWorldToCanvasWorld(vb);
		canvas->mapCanvasWorldToViewport(vb);
	}

	if (layer->hasGeometry(TeRASTER) && (theme->visibleRep() & TeRASTER))
	{
		TeBox bIn;
		if (box.isValid())
		{
			canvas->setClipRegion((int)vb.x1(), (int)vb.y1(), (int)vb.width(), (int)vb.height());
			bIn = vb;	
		}
	
		TeRaster* raster = theme->layer()->raster();
		if (!theme->rasterVisual())
			theme->createRasterVisual();
	
		canvas->plotRaster(raster,theme->rasterVisual(),progress);
	}	
	if(layer->hasGeometry(TePOLYGONS) && (theme->visibleRep() & TePOLYGONS))
	{
		nLoops = 0;
		// query the database table containing the polygons geometry
		string wherebox = getWhereBox(portal, theme, canvas, TePOLYGONS, box);
		tableName = layer->tableName(TePOLYGONS);

		string queryPolygons  = "SELECT " + tableName + ".*, c_legend_id, c_legend_own, c_object_status";
		queryPolygons += " FROM " + tableName + ", " + theme->collectionTable();
		queryPolygons += " WHERE object_id = c_object_id";
		if(wherebox.empty() == false)
			queryPolygons += " AND " + wherebox;

		if(db->dbmsName() != "OracleSpatial" && db->dbmsName() != "PostGIS")
			queryPolygons += " ORDER BY object_id ASC, parent_id ASC, num_holes DESC";
		else
			queryPolygons += " ORDER BY object_id ASC";

		//Calculate the number of polygons to be drawn
		string selectPolyCount; 
		if(db->dbmsName()=="SqlServerAdo")
			selectPolyCount="SELECT COUNT(DISTINCT(parent_id)) FROM ";
		else if(db->dbmsName() != "OracleSpatial" && db->dbmsName() != "PostGIS")
			selectPolyCount = "SELECT DISTINCT parent_id FROM ";
		else
			selectPolyCount = "SELECT DISTINCT object_id FROM ";

		selectPolyCount += tableName + ", " + theme->collectionTable() + " WHERE c_object_id = object_id";
		if(wherebox.empty() == false)
			selectPolyCount += " AND " + wherebox;

		string polyCount;
		if(db->dbmsName() =="SqlServerAdo")
			polyCount=selectPolyCount;
		else if(db->dbmsName() == "PostgreSQL" || db->dbmsName() == "PostGIS")
			polyCount = "SELECT COUNT(*) FROM (" + selectPolyCount + ") AS r";		
		else if(db->dbmsName() == "MySQL")
		{
			string createTempTable = "CREATE TEMPORARY TABLE count_table " + selectPolyCount;
			db->execute(createTempTable);
			polyCount = "SELECT COUNT(*) FROM count_table";
		}
		else 
			polyCount = "SELECT COUNT(*) FROM (" + selectPolyCount + ")";
		
		if (portal->query(polyCount))
		{
			if (portal->fetchRow())
				nSteps = atoi(portal->getData(0));
		}
		else
		{
			delete portal;
			return;
		}

		if (db->dbmsName() == "MySQL")
			db->execute("DROP TABLE count_table");

		portal->freeResult();

		//Plot the polygons whose number was placed in nSteps
		if(db->dbmsName() == "PostgreSQL" || db->dbmsName() == "PostGIS")
		{
			 if(portal->query(queryPolygons, TeSERVERSIDE, TeUNIDIRECTIONAL, TeREADONLY, TeBINARYCURSOR) == false)
			 {
				delete portal;
				return;
			 } 
		}
		else
		{
			if (portal->query(queryPolygons) == false)
			{
				delete portal;
				return;
			}
		}

		int	groPos = portal->AttributeList().size() - 3;
		int	ownPos = portal->AttributeList().size() - 2;
		int	resPos = portal->AttributeList().size() - 1;
		
		progress->setTotalSteps(nSteps);
		t2 = clock();
		t0 = t1 = t2;
		if(portal->fetchRow())
		{
			bool flag;
			flag = true;
			do
			{
				int geomId = atoi(portal->getData(0));					
				TePolygon poly;
				visual = getVisual(portal, theme, TePOLYGONS, resPos, groPos, ownPos);					
				color = visual.color();
				canvas->setPolygonColor(color.red_, color.green_, color.blue_);
				canvas->setPolygonStyle((int)visual.style(),visual.transparency());
				color = visual.contourColor();
				canvas->setPolygonLineColor (color.red_, color.green_, color.blue_);
				canvas->setPolygonLineStyle((int)visual.contourStyle(),visual.contourWidth());
				if(box.isValid())
					canvas->setClipRegion((int)vb.x1(), (int)vb.y1(), (int)vb.width(), (int)vb.height());
				if(it == layerRepMap.end())
				{
					flag = portal->fetchGeometry(poly);
					canvas->plotPolygon(poly);
				}
				else
				{
					canvas->plotPolygon(layerRepMap[layer->id()].polygonMap_[geomId]);
					int parent_id = atoi(portal->getData(4));
					int parent = parent_id;
					while(parent == parent_id)
					{
						flag = portal->fetchRow();
						if(flag)
							parent = atoi(portal->getData(4));
						else
							parent = -1;
					}
				}
				t2 = clock();
				nLoops++;
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					canvas->copyPixmapToWindow();
					if(progress->wasCancelled())
					{
						progress->reset();
						break;
					}
					if((int)(t2-t0) > dt2)
						progress->setProgress(nLoops);
				}
			}
			while (flag);
		}
		progress->reset();
	}

	if(box.isValid())
		canvas->setClipRegion((int)vb.x1(), (int)vb.y1(), (int)vb.width(), (int)vb.height());
	if(layer->hasGeometry(TeCELLS) && (theme->visibleRep() & TeCELLS))
	{
		nLoops = 0;
		// query the database table containing the polygons geometry
		string wherebox = getWhereBox(portal, theme, canvas, TeCELLS, box);
		tableName = layer->tableName(TeCELLS);

		string queryCells  = "SELECT " + tableName + ".*, c_legend_id, c_legend_own, c_object_status";
		queryCells += " FROM " + tableName + ", " + theme->collectionTable();
		queryCells += " WHERE object_id = c_object_id";
		if(wherebox.empty() == false)
			queryCells += " AND " + wherebox;

		string cellsCount = "SELECT COUNT(*) FROM " + tableName + ", " + theme->collectionTable();
		cellsCount += " WHERE object_id = c_object_id";
		if(wherebox.empty() == false)
			cellsCount += " AND " + wherebox;
		
		portal->freeResult();
		if (portal->query(cellsCount))
		{
			if (portal->fetchRow())
				nSteps = atoi(portal->getData(0));
		}
		else
		{
			delete portal;
			return;
		}

		portal->freeResult();
		if (portal->query(queryCells) == false)
		{
			delete portal;
			return;
		}
		int	groPos = portal->AttributeList().size() - 3;
		int	ownPos = portal->AttributeList().size() - 2;
		int	resPos = portal->AttributeList().size() - 1;

		progress->setTotalSteps(nSteps);
		t2 = clock();
		t0 = t1 = t2;
		if(portal->fetchRow())
		{
			bool flag;
			flag = true;
			do
			{
				TeCell cell;
				visual = getVisual(portal, theme, TePOLYGONS, resPos, groPos, ownPos);	//use TePOLYGONS				
				color = visual.color();
				canvas->setPolygonColor(color.red_, color.green_, color.blue_);
				canvas->setPolygonStyle((int)visual.style(),visual.transparency());
				color = visual.contourColor();
				canvas->setPolygonLineColor (color.red_, color.green_, color.blue_);
				canvas->setPolygonLineStyle((int)visual.contourStyle(),visual.contourWidth());
				flag = portal->fetchGeometry(cell);
				canvas->plotCell(cell);
				t2 = clock();
				nLoops++;
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					canvas->copyPixmapToWindow();
					if(progress->wasCancelled())
					{
						progress->reset();
						break;
					}
					if((int)(t2-t0) > dt2)
						progress->setProgress(nLoops);
				}
			}
			while (flag);
		}
		progress->reset();
	}

	if (layer->hasGeometry(TeLINES) && (theme->visibleRep() & TeLINES))
	{
		nLoops = 0;
		string wherebox = getWhereBox(portal, theme, canvas, TeLINES, box);
		// query the database table containing the lines geometry
		tableName = layer->tableName(TeLINES);

		string queryLines  = "SELECT " + tableName + ".*, c_legend_id, c_legend_own, c_object_status";
		queryLines += " FROM " + tableName + ", " + theme->collectionTable();
		queryLines += " WHERE object_id = c_object_id";
		if(wherebox.empty() == false)
			queryLines += " AND " + wherebox;

		string linesCount = "SELECT COUNT(*) FROM " + tableName + ", " + theme->collectionTable();
		linesCount += " WHERE object_id = c_object_id";
		if(wherebox.empty() == false)
			linesCount += " AND " + wherebox;

		portal->freeResult();
		if (portal->query(linesCount))
		{
			if (portal->fetchRow())
				nSteps = atoi(portal->getData(0));
		}
		else
		{
			delete portal;
			return;
		}

		portal->freeResult();
		if (portal->query(queryLines) == false)
		{
			delete portal;
			return;
		}
		int	groPos = portal->AttributeList().size() - 3;
		int	ownPos = portal->AttributeList().size() - 2;
		int	resPos = portal->AttributeList().size() - 1;

		progress->setTotalSteps(nSteps);
		t2 = clock();
		t0 = t1 = t2;
		if(portal->fetchRow())
		{
			bool flag = true;
			do
			{
				int geomId = atoi(portal->getData(0));					
				TeLine2D line;
				visual = getVisual(portal, theme, TeLINES, resPos, groPos, ownPos);					
				color = visual.color();
				canvas->setLineColor(color.red_, color.green_, color.blue_);
				canvas->setLineStyle((int)visual.style(),visual.width());
				if(it == layerRepMap.end())
				{
					flag = portal->fetchGeometry(line);
					canvas->plotLine(line);
				}
				else
				{
					canvas->plotLine(layerRepMap[layer->id()].lineMap_[geomId]);
					flag = portal->fetchRow();
				}
				t2 = clock();
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					canvas->copyPixmapToWindow();
					if(progress->wasCancelled())
					{
						progress->reset();
						break;
					}
					if((int)(t2-t0) > dt2)
						progress->setProgress(int(t2));
				}
			}
			while (flag);
		}
		progress->reset();
	}

	if (layer->hasGeometry(TePOINTS) && (theme->visibleRep() & TePOINTS))
	{
		nLoops = 0;
		string wherebox = getWhereBox(portal, theme, canvas, TePOINTS, box);
		// query the database table containing the points geometry
		tableName = layer->tableName(TePOINTS);

		string queryPoints  = "SELECT " + tableName + ".*, c_legend_id, c_legend_own, c_object_status";
		queryPoints += " FROM " + tableName + ", " + theme->collectionTable();
		queryPoints += " WHERE object_id = c_object_id";
		if(wherebox.empty() == false)
			queryPoints += " AND " + wherebox;

		string pointsCount = "SELECT COUNT(*) FROM " + tableName + ", " + theme->collectionTable();
		pointsCount += " WHERE object_id = c_object_id";
		if(wherebox.empty() == false)
			pointsCount += " AND " + wherebox;

		portal->freeResult();
		if (portal->query(pointsCount))
		{
			if (portal->fetchRow())
				nSteps = atoi(portal->getData(0));
		}
		else
		{
			delete portal;
			return;
		}

		portal->freeResult();
		if (portal->query(queryPoints) == false)
		{
			delete portal;
			return;
		}
		int	groPos = portal->AttributeList().size() - 3;
		int	ownPos = portal->AttributeList().size() - 2;
		int	resPos = portal->AttributeList().size() - 1;

		progress->setTotalSteps(nSteps);
		t2 = clock();
		t0 = t1 = t2;
		if(portal->fetchRow())
		{
			bool flag = true;
			do
			{
				TePoint point;
				visual = getVisual(portal, theme, TePOINTS, resPos, groPos, ownPos);					
				color = visual.color();
				canvas->setPointColor(color.red_, color.green_, color.blue_);
				canvas->setPointStyle((int)visual.style(),visual.size());
				flag = portal->fetchGeometry(point);
				canvas->plotPoint(point);
				t2 = clock();
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					canvas->copyPixmapToWindow();
					if(progress->wasCancelled())
					{
						progress->reset();
						break;
					}
					if((int)(t2-t0) > dt2)
						progress->setProgress(int(t2));
				}
			}
			while (flag);
		}
		progress->reset();
	}
	delete portal;

	if(box.isValid())
		canvas->setClipping(false);
}



template<class Canvas , class Progress>
TeBox TePlotSelectedObjects(TeThemeApplication *theme, Canvas *canvas, set<string>& drawSet, map<int, RepMap>& layerRepMap, Progress * /* progress */ = 0, TeGeomRepVisualMap* visualMap = 0)
{
	TeBox	boxRet;
	int i, nLoops;
	int dt = CLOCKS_PER_SEC/4;
	clock_t	t0, t1, t2;
	vector<string> inClauseVector;

	TeLayer *layer = theme->layer();
	TeDatabase *db = layer->database();
	TeDatabasePortal *portal = db->getPortal();
	canvas->setDataProjection (layer->projection());
	string TC = theme->collectionTable();

	string tableName, objectId;
	TeColor color;
	TeVisual	visual;

	set<string>::iterator itB = drawSet.begin();
	set<string>::iterator itE = drawSet.end();
	inClauseVector = generateInClauses(itB,itE, db);

	if (layer->hasGeometry(TePOLYGONS) && (theme->visibleRep() & TePOLYGONS))
	{
		map<int, RepMap>::iterator it = layerRepMap.find(layer->id());
//		nLoops = 0;
//		progress->setTotalSteps(drawSet.size());
		t2 = clock();
		t0 = t1 = t2;

		bool hasVisual = false;

		if(visualMap)
			hasVisual = (visualMap->find(TePOLYGONS) != visualMap->end());

		TeVisual visualPol;

		if(hasVisual)
			visualPol = (*visualMap)[TePOLYGONS];


		for (i = 0; i < (int)inClauseVector.size(); ++i)
		{
			// query the database table containing the polygons geometry
			tableName = layer->tableName(TePOLYGONS);
			string queryPolygons  = "SELECT " + tableName + ".*, c_legend_id, c_legend_own, c_object_status";
			queryPolygons += " FROM " + tableName + ", " + theme->collectionTable();
			queryPolygons += " WHERE object_id = c_object_id AND object_id IN " + inClauseVector[i];
			
			if(db->dbmsName() != "OracleSpatial" && db->dbmsName() != "PostGIS")
				queryPolygons += " ORDER BY object_id ASC, parent_id ASC, num_holes DESC";
			else
				queryPolygons += " ORDER BY object_id ASC";

			portal->freeResult();
			if (portal->query(queryPolygons) == false)
			{
				delete portal;
				return boxRet;
			}
			int	groPos = portal->AttributeList().size() - 3;
			int	ownPos = portal->AttributeList().size() - 2;
			int	resPos = portal->AttributeList().size() - 1;

			if(portal->fetchRow())
			{
				bool flag = true;
				do
				{
					TePolygon poly;

					if(hasVisual)
					{
						color = visualPol.color();
						canvas->setPolygonColor(color.red_, color.green_, color.blue_);
						canvas->setPolygonStyle((int)visualPol.style(),visualPol.transparency());
						color = visualPol.contourColor();
						canvas->setPolygonLineColor (color.red_, color.green_, color.blue_);
						canvas->setPolygonLineStyle((int)visualPol.contourStyle(),visualPol.contourWidth());
					}
					else
					{
						visual = getVisual(portal, theme, TePOLYGONS, resPos, groPos, ownPos);					
						color = visual.color();
						canvas->setPolygonColor(color.red_, color.green_, color.blue_);
						canvas->setPolygonStyle((int)visual.style(),visual.transparency());
						color = visual.contourColor();
						canvas->setPolygonLineColor (color.red_, color.green_, color.blue_);
						canvas->setPolygonLineStyle((int)visual.contourStyle(),visual.contourWidth());
					}

					if(it == layerRepMap.end())
					{
						flag = portal->fetchGeometry(poly);
						boxRet = TeUnion(boxRet, poly.box());
						canvas->plotPolygon(poly);
					}
					else
					{
						int geomId = atoi(portal->getData(0));					
						int parent_id = atoi(portal->getData(4));
						int parent = parent_id;
						while(parent == parent_id)
						{
							flag = portal->fetchRow();
							if(flag)
								parent = atoi(portal->getData(4));
							else
								parent = -1;
						}
						boxRet = TeUnion(boxRet, layerRepMap[layer->id()].polygonMap_[geomId].box());
						canvas->plotPolygon(layerRepMap[layer->id()].polygonMap_[geomId]);
					}
					t2 = clock();
//					nLoops++;
					if (int(t2-t1) > dt)
					{
						t1 = t2;
						canvas->copyPixmap1ToWindow();
//						if(progress->wasCancelled())
//							break;
//						progress->setProgress(nLoops);
					}
				}
				while (flag);
			}
		}
//		progress->reset();
	}

	if (layer->hasGeometry(TeCELLS) && (theme->visibleRep() & TeCELLS))
	{
//		nLoops = 0;
//		progress->setTotalSteps(drawSet.size());
		t2 = clock();
		t0 = t1 = t2;

		bool hasVisual = false;

		if(visualMap)
			hasVisual = (visualMap->find(TeCELLS) != visualMap->end());

		TeVisual visualCell;

		if(hasVisual)
			visualCell = (*visualMap)[TeCELLS];


		for (i = 0; i < (int)inClauseVector.size(); ++i)
		{
			// query the database table containing the lines geometry
			tableName = layer->tableName(TeCELLS);
			string queryCells  = "SELECT " + tableName + ".*, c_legend_id, c_legend_own, c_object_status";
			queryCells += " FROM " + tableName + ", " + theme->collectionTable();
			queryCells += " WHERE object_id = c_object_id AND object_id IN " + inClauseVector[i];

			portal->freeResult();
			if (portal->query(queryCells) == false)
			{
				delete portal;
				return boxRet;
			}
			int	groPos = portal->AttributeList().size() - 3;
			int	ownPos = portal->AttributeList().size() - 2;
			int	resPos = portal->AttributeList().size() - 1;
			
			if(portal->fetchRow())
			{
				bool flag = true;
				do
				{
					TeCell cell;

					if(hasVisual)
					{
						color = visualCell.color();
						canvas->setPolygonColor(color.red_, color.green_, color.blue_);
						canvas->setPolygonStyle((int)visualCell.style(),visualCell.transparency());
						color = visualCell.contourColor();
						canvas->setPolygonLineColor (color.red_, color.green_, color.blue_);
						canvas->setPolygonLineStyle((int)visualCell.contourStyle(),visualCell.contourWidth());
					}
					else
					{

						visual = getVisual(portal, theme, TePOLYGONS, resPos, groPos, ownPos);	//use TePOLYGONS				
						color = visual.color();
						canvas->setPolygonColor(color.red_, color.green_, color.blue_);
						canvas->setPolygonStyle((int)visual.style(),visual.transparency());
						color = visual.contourColor();
						canvas->setPolygonLineColor (color.red_, color.green_, color.blue_);
						canvas->setPolygonLineStyle((int)visual.contourStyle(),visual.contourWidth());
					}

					flag = portal->fetchGeometry(cell);
					boxRet = TeUnion(boxRet, cell.box());
					canvas->plotCell(cell);
					t2 = clock();
//					nLoops++;
					if (int(t2-t1) > dt)
					{
						t1 = t2;
						canvas->copyPixmap1ToWindow();
//						if(progress->wasCancelled())
//							break;
//						progress->setProgress(nLoops);
					}
				}
				while (flag);
			}
		}
//		progress->reset();
	}

	if (layer->hasGeometry(TeLINES) && (theme->visibleRep() & TeLINES))
	{
//		nLoops = 0;
//		progress->setTotalSteps(drawSet.size());
		t2 = clock();
		t0 = t1 = t2;

		bool hasVisual = false;

		if(visualMap)
			hasVisual = (visualMap->find(TeLINES) != visualMap->end());

		TeVisual visualLin;

		if(hasVisual)
			visualLin = (*visualMap)[TeLINES];


		for (i = 0; i < (int)inClauseVector.size(); ++i)
		{
			// query the database table containing the lines geometry
			tableName = layer->tableName(TeLINES);
			string queryLines  = "SELECT " + tableName + ".*, c_legend_id, c_legend_own, c_object_status";
			queryLines += " FROM " + tableName + ", " + theme->collectionTable();
			queryLines += " WHERE object_id = c_object_id AND object_id IN " + inClauseVector[i];

			portal->freeResult();
			if (portal->query(queryLines) == false)
			{
				delete portal;
				return boxRet;
			}
			int	groPos = portal->AttributeList().size() - 3;
			int	ownPos = portal->AttributeList().size() - 2;
			int	resPos = portal->AttributeList().size() - 1;

			if(portal->fetchRow())
			{
				bool flag = true;
				do
				{
					TeLine2D line;

					if(hasVisual)
					{
						color = visualLin.color();
						canvas->setLineColor(color.red_, color.green_, color.blue_);
						canvas->setLineStyle((int)visualLin.style(),visualLin.width());
					}
					else
					{
						visual = getVisual(portal, theme, TeLINES, resPos, groPos, ownPos);					
						color = visual.color();
						canvas->setLineColor(color.red_, color.green_, color.blue_);
						canvas->setLineStyle((int)visual.style(),visual.width());
					}

					flag = portal->fetchGeometry(line);
					boxRet = TeUnion(boxRet, line.box());
					canvas->plotLine(line);
					t2 = clock();
//					nLoops++;
					if (int(t2-t1) > dt)
					{
						t1 = t2;
						canvas->copyPixmap1ToWindow();
//						if(progress->wasCancelled())
//							break;
//						progress->setProgress(nLoops);
					}
				}
				while (flag);
			}
		}
//		progress->reset();
	}

	if (layer->hasGeometry(TePOINTS) && (theme->visibleRep() & TePOINTS))
	{
		nLoops = 0;
//		progress->setTotalSteps(drawSet.size());
		t2 = clock();
		t0 = t1 = t2;

		bool hasVisual = false;

		if(visualMap)
			hasVisual = (visualMap->find(TePOINTS) != visualMap->end());

		TeVisual visualPt;

		if(hasVisual)
			visualPt = (*visualMap)[TePOINTS];



		for (i = 0; i < (int)inClauseVector.size(); ++i)
		{
			// query the database table containing the points geometry
			tableName = layer->tableName(TePOINTS);
			string queryLines  = "SELECT " + tableName + ".*, c_legend_id, c_legend_own, c_object_status";
			queryLines += " FROM " + tableName + ", " + theme->collectionTable();
			queryLines += " WHERE object_id = c_object_id AND object_id IN " + inClauseVector[i];

			portal->freeResult();
			if (portal->query(queryLines) == false)
			{
				delete portal;
				return boxRet;
			}
			int	groPos = portal->AttributeList().size() - 3;
			int	ownPos = portal->AttributeList().size() - 2;
			int	resPos = portal->AttributeList().size() - 1;

			if(portal->fetchRow())
			{
				bool flag = true;
				do
				{
					TePoint point;

					if(hasVisual)
					{
						color = visualPt.color();
						canvas->setPointColor(color.red_, color.green_, color.blue_);
						canvas->setPointStyle((int)visualPt.style(),visualPt.size());
					}
					else
					{
						visual = getVisual(portal, theme, TePOINTS, resPos, groPos, ownPos);					
						color = visual.color();
						canvas->setPointColor(color.red_, color.green_, color.blue_);
						canvas->setPointStyle((int)visual.style(),visual.size());
					}

					flag = portal->fetchGeometry(point);
					double dw = canvas->mapVtoDW(visual.size());
					TeCoord2D p = point.location();
					TeBox b(p.x()-dw, p.y()-dw, p.x()+dw, p.y()+dw);
					boxRet = TeUnion(boxRet, b);
					canvas->plotPoint(point);
					t2 = clock();
					nLoops++;
					if (int(t2-t1) > dt)
					{
						t1 = t2;
						canvas->copyPixmap1ToWindow();
//						if(progress->wasCancelled())
//							break;
//						progress->setProgress(nLoops);
					}
				}
				while (flag);
			}
		}
//		progress->reset();
	}
	delete portal;
	return boxRet;
}

template<class Canvas>
bool objBoxInCanvas(TeThemeApplication *theme, Canvas *canvas, set<string>& drawSet, TeGeomRep repType, TeBox& bout, double tol)
{
	//calculate the box that contains all the objects in objectIdSet.
	//If this box is inside the canvas box, return true to indicate that
	//is enough to paint the objects selected. If this box is not inside
	//the canvas box, center this box in the canvas, and return false
	//to indicate that a plot in the whole canvas area is necessary.

	TeLayer *layer = theme->layer();
	TeDatabase *db = layer->database();
	TeDatabasePortal *portal = db->getPortal();

	string tableName;
	if (repType & TePOLYGONS)
		tableName = theme->layer()->tableName(TePOLYGONS);
	else if (repType & TeLINES)
		tableName = theme->layer()->tableName(TeLINES);
	else if (repType & TePOINTS)
		tableName = theme->layer()->tableName(TePOINTS);
	else if (repType & TeCELLS)
		tableName = theme->layer()->tableName(TeCELLS);

	set<string>::iterator it, beginIt;
	int i, chunk = 200, size = drawSet.size();
	string inClause, plic, fields, queryString;

	beginIt = drawSet.begin();
	while (size > 0)
	{
		portal->freeResult();
		inClause = "(";
		plic = "'";
		i = 0;
		for (it = beginIt; it != drawSet.end(); ++it)
		{
			if (i >= chunk)
				break;
			inClause += plic + *it + "',";
			++i;
		}
		inClause[inClause.size() - 1] = ')';
		beginIt = it;
		size -= chunk;

		if((repType & TePOLYGONS) || (repType & TeLINES) || (repType & TeCELLS))
		{
			if(db->dbmsName() == "PostGIS")
				fields = "MIN(xmin(spatial_data::box3d)), MIN(ymin(spatial_data::box3d)), MAX(xmax(spatial_data::box3d)), MAX(ymax(spatial_data::box3d))";
			else
				fields = "MIN(lower_x), MIN(lower_y), MAX(upper_x), MAX(upper_y)";

			queryString =  "SELECT " + fields;
			queryString += " FROM " + tableName; 
			queryString += " WHERE object_id IN " + inClause;
			
			if (portal->query(queryString))
			{
				while(portal->fetchRow())
				{
					string vxmin = portal->getData(0);
					string vymin = portal->getData(1);
					string vxmax = portal->getData(2);
					string vymax = portal->getData(3);
					if(vxmin.empty() || vymin.empty() || vxmax.empty() || vymax.empty())
						continue;
					double xmin = atof(vxmin.c_str());
					double ymin = atof(vymin.c_str());
					double xmax = atof(vxmax.c_str());
					double ymax = atof(vymax.c_str());
					TeBox ibox(xmin+tol, ymin+tol, xmax-tol, ymax-tol);
					updateBox (bout, ibox);
				}
			}
		}
		else if(repType & TePOINTS)
		{
			if(db->dbmsName() == "PostGIS")
				fields = "MIN(xmin(spatial_data::box3d)), MIN(ymin(spatial_data::box3d)), MAX(xmax(spatial_data::box3d)), MAX(ymax(spatial_data::box3d))";
			else
				fields = "MIN(x), MIN(y), MAX(x), MAX(y)";

			queryString =  "SELECT " + fields;
			queryString += " FROM " + tableName; 
			queryString += " WHERE object_id IN " + inClause;

			if (portal->query(queryString))
			{
				while(portal->fetchRow())
				{
					string vx = portal->getData(0);
					string vy = portal->getData(1);
					if(vx.empty() || vy.empty())
						continue;
					double x = atof(vx.c_str());
					double y = atof(vy.c_str()); 

					TeBox ibox(x-tol, y-tol, x+tol, y+tol);
					updateBox (bout, ibox);
				}
			}
		}
	}

	TeBox cbox = canvas->getDataWorld();
	if(TeWithin(bout, cbox) == true)
	{
		delete portal;
		return true;
	}

	//box of the objects selected is not inside the canvas box
	if(bout.width() < cbox.width())
	{
		double w = cbox.width();
		double tw = bout.width();
		double dw = (w - tw) / 2.;
		bout.x1_ = bout.x1_ - dw;
		bout.x2_ = bout.x2_ + dw;
	}

	if(bout.height() < cbox.height())
	{
		double h = cbox.height();
		double th = bout.height();
		double dh = (h - th) / 2.;
		bout.y1_ = bout.y1_ - dh;
		bout.y2_ = bout.y2_ + dh;
	}

	delete portal;

	// layer to canvas projection
	bout = TeRemapBox(bout, theme->layer()->projection(), canvas->projection());
	return false;
}


template<class Canvas>
void TePlotFrame(TeThemeApplication* theme, Canvas* canvas, map<int, RepMap>& layerRepMap, map<string, TeGeomRepVisualMap*>& objVisualMap)
{
	string tableName, objectId;
	TeColor color;
	TeVisual visual;

	TeLayer *layer = theme->layer();
	TeDatabase *db = layer->database();
	TeDatabasePortal *portal = db->getPortal();
	canvas->setDataProjection (layer->projection());
	map<int, RepMap>::iterator it = layerRepMap.find(layer->id());
	TeGeomRepVisualMap* vmap;

	TeBox box;
	TeBox vb = box;
	if(box.isValid())
	{
		canvas->mapDataWorldToCanvasWorld(vb);
		canvas->mapCanvasWorldToViewport(vb);
	}

	if (layer->hasGeometry(TeRASTER) && (theme->visibleRep() & TeRASTER))
	{
		TeBox bIn;
		if (box.isValid())
		{
			canvas->setClipRegion((int)vb.x1(), (int)vb.y1(), (int)vb.width(), (int)vb.height());
			bIn = vb;	
		}
	
		TeRaster* raster = theme->layer()->raster();
		if (!theme->rasterVisual())
			theme->createRasterVisual();
	
		canvas->plotRaster(raster,theme->rasterVisual(), 0);
	}	
	if(layer->hasGeometry(TePOLYGONS) && (theme->visibleRep() & TePOLYGONS))
	{
		// query the database table containing the polygons geometry
		string wherebox = getWhereBox(portal, theme, canvas, TePOLYGONS, box);
		tableName = layer->tableName(TePOLYGONS);

		string queryPolygons  = "SELECT " + tableName + ".*";
		queryPolygons += " FROM " + tableName + ", " + theme->collectionTable();
		queryPolygons += " WHERE object_id = c_object_id";
		if(wherebox.empty() == false)
			queryPolygons += " AND " + wherebox;

		if(db->dbmsName() != "OracleSpatial" && db->dbmsName() != "PostGIS")
			queryPolygons += " ORDER BY object_id ASC, parent_id ASC, num_holes DESC";
		else
			queryPolygons += " ORDER BY object_id ASC";

		portal->freeResult();

		//Plot the polygons whose number was placed in nSteps
		if (portal->query(queryPolygons) == false)
		{
			delete portal;
			return;
		}
		
		if(portal->fetchRow())
		{
			bool flag;
			flag = true;
			do
			{
				TePolygon poly;
				int geomId = atoi(portal->getData(0));
				string oid = portal->getData(1);
				if (objVisualMap.find(oid) != objVisualMap.end())
					vmap = objVisualMap[oid];
				else
				{
					flag = portal->fetchGeometry(poly);
					continue;
				}

				visual = vmap->operator[](TePOLYGONS);					
				color = visual.color();
				canvas->setPolygonColor(color.red_, color.green_, color.blue_);
				canvas->setPolygonStyle((int)visual.style(),visual.transparency());
				color = visual.contourColor();
				canvas->setPolygonLineColor (color.red_, color.green_, color.blue_);
				canvas->setPolygonLineStyle((int)visual.contourStyle(),visual.contourWidth());
				if(box.isValid())
					canvas->setClipRegion((int)vb.x1(), (int)vb.y1(), (int)vb.width(), (int)vb.height());
				if(it == layerRepMap.end())
				{
					flag = portal->fetchGeometry(poly);
					canvas->plotPolygon(poly);
				}
				else
				{
					canvas->plotPolygon(layerRepMap[layer->id()].polygonMap_[geomId]);
					int parent_id = atoi(portal->getData(4));
					int parent = parent_id;
					while(parent == parent_id)
					{
						flag = portal->fetchRow();
						if(flag)
							parent = atoi(portal->getData(4));
						else
							parent = -1;
					}
				}
			}
			while (flag);
		}
	}

	if(box.isValid())
		canvas->setClipRegion((int)vb.x1(), (int)vb.y1(), (int)vb.width(), (int)vb.height());
	if(layer->hasGeometry(TeCELLS) && (theme->visibleRep() & TeCELLS))
	{
		// query the database table containing the polygons geometry
		string wherebox = getWhereBox(portal, theme, canvas, TeCELLS, box);
		tableName = layer->tableName(TeCELLS);

		string queryCells  = "SELECT " + tableName + ".*";
		queryCells += " FROM " + tableName + ", " + theme->collectionTable();
		queryCells += " WHERE object_id = c_object_id";
		if(wherebox.empty() == false)
			queryCells += " AND " + wherebox;

		portal->freeResult();
		if (portal->query(queryCells) == false)
		{
			delete portal;
			return;
		}

		if(portal->fetchRow())
		{
			bool flag;
			flag = true;
			do
			{
				TeCell cell;
				string oid = portal->getData(1);
				if (objVisualMap.find(oid) != objVisualMap.end())
					vmap = objVisualMap[oid];
				else
				{
					flag = portal->fetchGeometry(cell);
					continue;
				}
				visual = vmap->operator[](TePOLYGONS);					
				color = visual.color();
				canvas->setPolygonColor(color.red_, color.green_, color.blue_);
				canvas->setPolygonStyle((int)visual.style(),visual.transparency());
				color = visual.contourColor();
				canvas->setPolygonLineColor (color.red_, color.green_, color.blue_);
				canvas->setPolygonLineStyle((int)visual.contourStyle(),visual.contourWidth());
				flag = portal->fetchGeometry(cell);
				canvas->plotCell(cell);
			}
			while (flag);
		}
	}

	if (layer->hasGeometry(TeLINES) && (theme->visibleRep() & TeLINES))
	{
		string wherebox = getWhereBox(portal, theme, canvas, TeLINES, box);
		// query the database table containing the lines geometry
		tableName = layer->tableName(TeLINES);

		string queryLines  = "SELECT " + tableName + ".*";
		queryLines += " FROM " + tableName + ", " + theme->collectionTable();
		queryLines += " WHERE object_id = c_object_id";
		if(wherebox.empty() == false)
			queryLines += " AND " + wherebox;

		portal->freeResult();
		if (portal->query(queryLines) == false)
		{
			delete portal;
			return;
		}

		if(portal->fetchRow())
		{
			bool flag = true;
			do
			{
				TeLine2D line;
				int geomId = atoi(portal->getData(0));					
				string oid = portal->getData(1);
				if (objVisualMap.find(oid) != objVisualMap.end())
					vmap = objVisualMap[oid];
				else
				{
					flag = portal->fetchGeometry(line);
					continue;
				}

				visual = vmap->operator[](TeLINES);					
				color = visual.color();
				canvas->setLineColor(color.red_, color.green_, color.blue_);
				canvas->setLineStyle((int)visual.style(),visual.width());
				if(it == layerRepMap.end())
				{
					flag = portal->fetchGeometry(line);
					canvas->plotLine(line);
				}
				else
				{
					canvas->plotLine(layerRepMap[layer->id()].lineMap_[geomId]);
					flag = portal->fetchRow();
				}
			}
			while (flag);
		}
	}

	if (layer->hasGeometry(TePOINTS) && (theme->visibleRep() & TePOINTS))
	{
		string wherebox = getWhereBox(portal, theme, canvas, TePOINTS, box);
		// query the database table containing the points geometry
		tableName = layer->tableName(TePOINTS);

		string queryPoints  = "SELECT " + tableName + ".*";
		queryPoints += " FROM " + tableName + ", " + theme->collectionTable();
		queryPoints += " WHERE object_id = c_object_id";
		if(wherebox.empty() == false)
			queryPoints += " AND " + wherebox;

		portal->freeResult();
		if (portal->query(queryPoints) == false)
		{
			delete portal;
			return;
		}

		if(portal->fetchRow())
		{
			bool flag = true;
			do
			{
				TePoint point;
				string oid = portal->getData(1);
				if (objVisualMap.find(oid) != objVisualMap.end())
					vmap = objVisualMap[oid];
				else
				{
					flag = portal->fetchGeometry(point);
					continue;
				}

				visual = vmap->operator[](TePOINTS);					
				color = visual.color();
				canvas->setPointColor(color.red_, color.green_, color.blue_);
				canvas->setPointStyle((int)visual.style(),visual.size());
				flag = portal->fetchGeometry(point);
				canvas->plotPoint(point);
			}
			while (flag);
		}
	}
	delete portal;

	if(box.isValid())
		canvas->setClipping(false);
}

#endif

