/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "syncml.h"

#include "syncml_internals.h"
#include "sml_parse_internals.h"
#include "sml_elements_internals.h"
#include "sml_command_internals.h"

/**
 * @defgroup GroupIDPrivate Group Description Internals
 * @ingroup ParentGroupID
 * @brief The private part
 * 
 */
/*@{*/

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/

SmlLocation *smlLocationNew(const char *locURI, const char *locName, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%s, %s, %p)", __func__, locURI, locName, error);
	
	if (!locURI) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No locURI was given");
		goto error;
	}
	
	SmlLocation *location = smlTryMalloc0(sizeof(SmlLocation), error);
	if (!location)
		goto error;
	
	location->refCount = 1;
	location->locURI = g_strdup(locURI);
	location->locName = g_strdup(locName);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, location);
	return location;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;	
}

SmlLocation *smlLocationRef(SmlLocation *loc)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, loc);
	smlAssert(loc);
	
	g_atomic_int_inc(&(loc->refCount));
	
	smlTrace(TRACE_EXIT, "%s: New refcount: %i", __func__, loc->refCount);
	return loc;
}

void smlLocationUnref(SmlLocation *loc)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, loc);
	smlAssert(loc);
	
	if (g_atomic_int_dec_and_test(&(loc->refCount))) {
		smlTrace(TRACE_INTERNAL, "Refcount == 0!");
		
		if (loc->locURI)
			g_free(loc->locURI);
			
		if (loc->locName)
			g_free(loc->locName);
			
		g_free(loc);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

const char *smlLocationGetURI(SmlLocation *loc)
{
	smlAssert(loc);
	return loc->locURI;
}

const char *smlLocationGetName(SmlLocation *loc)
{
	smlAssert(loc);
	return loc->locName;
}

void smlLocationSetName(SmlLocation *loc, const char *name)
{
	smlAssert(loc);
	smlAssert(name);
	if (loc->locName)
		g_free(loc->locName);
	loc->locName = g_strdup(name);
}

SmlLocation *smlLocationClone(SmlLocation *source, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, source, error);
	smlAssert(source);
	
	SmlLocation *location = smlTryMalloc0(sizeof(SmlLocation), error);
	if (!location)
		goto error;
	location->refCount = 1;
	
	smlLocationCopy(source, location);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, location);
	return location;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;	
}

void smlLocationCopy(SmlLocation *source, SmlLocation *target)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, source, target);
	smlAssert(source);
	smlAssert(target);
	
	if (target->locURI)
		g_free(target->locURI);
		
	if (target->locName)
		g_free(target->locName);
		
	target->locURI = g_strdup(source->locURI);
	target->locName = g_strdup(source->locName);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

#define URL_DELIMITER "/"

char *strreplace(const char *input, const char *needle, const char *replacement)
{
	char *output = g_strdup(input);
	
	while (g_strrstr(output, needle)) {
		char *buffer = g_strndup(output, g_strrstr(output, needle) - output);
		char *buffer2 = g_strconcat(buffer, replacement ? replacement : "", g_strrstr(output, needle) + strlen(needle), NULL);
		g_free(output);
		output = buffer2;
		g_free(buffer);
	}
	
	return output;
}

char *normalizeUrl(const char *url)
{
	smlTrace(TRACE_ENTRY, "%s(%s)", __func__, url);
	
	if (!url)
		return NULL;
	
	char *buffer = strreplace(url, "./", "");
	char *buffer2 = strreplace(buffer, "//", "/");
	g_free(buffer);
	
	if (buffer2[strlen(buffer2) - 1] == '/')
		buffer2[strlen(buffer2) - 1] = 0;
	
	smlTrace(TRACE_EXIT, "%s: %s", __func__, buffer2);
	return buffer2;
}

SmlBool smlLocationCompare(SmlLocation *objectroot, SmlLocation *object, SmlLocation *urlroot, SmlLocation *url)
{
	smlTrace(TRACE_ENTRY, "%s(%p(%s), %p(%s), %p(%s), %p(%s))", __func__, objectroot, objectroot ? objectroot->locURI : "null", object, object ? object->locURI : "null", urlroot, urlroot ? urlroot->locURI : "null", url, url ? url->locURI : "null");
	char *cmpobjurl = NULL;
	char *cmpurl = NULL;
	char *buffer = NULL;
	char *buffer2 = NULL;
	SmlBool ret = FALSE;
	
	if (object && !url) {
		smlTrace(TRACE_EXIT, "%s: url not given but object: FALSE", __func__);
		return FALSE;
	}
	
	if (!object) {
		smlTrace(TRACE_EXIT, "%s: No object given: TRUE", __func__);
		return TRUE;
	}
	
	if (g_path_is_absolute(url->locURI) || !urlroot)
		cmpurl = normalizeUrl(url->locURI);
	else {
		buffer2 = normalizeUrl(url->locURI);
		buffer = g_strconcat(urlroot->locURI, "/", buffer2, NULL);
		cmpurl = normalizeUrl(buffer);
		g_free(buffer);
		g_free(buffer2);
	}
	
	if (g_path_is_absolute(object->locURI) || !objectroot)
		cmpobjurl = normalizeUrl(object->locURI);
	else {
		buffer2 = normalizeUrl(object->locURI);
		buffer = g_strconcat(objectroot->locURI, "/", buffer2, NULL);
		cmpobjurl = normalizeUrl(buffer);
		g_free(buffer);
		g_free(buffer2);
	}
	
	
	smlTrace(TRACE_INTERNAL, "Comparing %s and %s", cmpobjurl, cmpurl);
	
	if (strcmp(cmpobjurl, cmpurl))
		ret = FALSE;
	else
		ret = TRUE;
	
	g_free(cmpobjurl);
	g_free(cmpurl);
	
	smlTrace(TRACE_EXIT, "%s: %i", __func__, ret);
	return ret;
}

SmlBool smlLocationIsRelative(SmlLocation *location)
{
	smlAssert(location);
	
	return g_path_is_absolute(location->locURI) ? FALSE : TRUE;
}

SmlAnchor *smlAnchorNew(const char *last, const char *next, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%s, %s, %p)", __func__, last, next, error);
	
	SmlAnchor *anchor = smlTryMalloc0(sizeof(SmlAnchor), error);
	if (!anchor)
		goto error;
	
	anchor->last = g_strdup(last);
	anchor->next = g_strdup(next);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, anchor);
	return anchor;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

void smlAnchorFree(SmlAnchor *anchor)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, anchor);
	smlAssert(anchor);
	if (anchor->last)
		g_free(anchor->last);
		
	if (anchor->next)
		g_free(anchor->next);
		
	g_free(anchor);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

void smlHeaderFree(SmlHeader *header)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, header);
	
	if (header->sessionID)
		g_free(header->sessionID);

	if (header->emi)
		g_free(header->emi);

	if (header->source)
		smlLocationUnref(header->source);
		
	if (header->target)
		smlLocationUnref(header->target);
	
	g_free(header);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
};

SmlItem *smlItemNew(unsigned int size, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%i, %p)", __func__, size, error);
		
	SmlItem *item = smlTryMalloc0(sizeof(SmlItem), error);
	if (!item)
		goto error;
	
	item->refCount = 1;
	item->size = size;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, item);
	return item;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

/* If data is NULL, this call is the same if smlItemNew */
SmlItem *smlItemNewForData(const char *data, unsigned int size, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p)", __func__, data, size, error);
	
	SmlItem *item = smlItemNew(size, error);
	if (!item)
		goto error;
	
	if (data) {
		if (!smlItemAddData(item, data, size, error))
			goto error_free_item;
	}
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, item);
	return item;

error_free_item:
	smlItemUnref(item);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlItem *smlItemRef(SmlItem *item)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, item);
	smlAssert(item);
	
	g_atomic_int_inc(&(item->refCount));
	
	smlTrace(TRACE_EXIT, "%s: New refcount: %i", __func__, item->refCount);
	return item;
}

void smlItemUnref(SmlItem *item)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, item);
	smlAssert(item);
	
	if (g_atomic_int_dec_and_test(&(item->refCount))) {
		smlTrace(TRACE_INTERNAL, "Refcount == 0!");
		
		if (item->source)
			smlLocationUnref(item->source);
			
		if (item->target)
			smlLocationUnref(item->target);
		
		if (item->anchor)
			smlAnchorFree(item->anchor);
		
		if (item->buffer)
			xmlBufferFree(item->buffer);
		
		if (item->contenttype)
			g_free(item->contenttype);
		
		g_free(item);
		item = NULL;
	}
	
	smlTrace(TRACE_EXIT, "%s: %i", __func__, item ? item->refCount : 0);
}

SmlBool smlItemAddData(SmlItem *item, const char *data, unsigned int size, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %i, %p)", __func__, item, data, size, error);
	
	if (item->size && xmlBufferLength(item->buffer) + size > item->size) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unable to add data. size limit reached");
		goto error;
	}
	
	if (data) {
		if (!item->buffer) {
			if (item->size)
				item->buffer = xmlBufferCreateSize(item->size);
			else
				item->buffer = xmlBufferCreateSize(size);
		}
		
		if (xmlBufferAdd(item->buffer, (xmlChar *)data, size) != 0) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Unable to add data.");
			goto error;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
	
}

/** Checks if the item is complete */
SmlBool smlItemCheck(SmlItem *item)
{
	smlAssert(item);
	if (!item->size)
		return TRUE;
		
	if (xmlBufferLength(item->buffer) != item->size)
	{
		smlTrace(TRACE_INTERNAL, "%s: failed because size (%d != %d) does not match (%s).",
			__func__, item->size,
			xmlBufferLength(item->buffer), xmlBufferContent(item->buffer));
		return FALSE;
	}
		
	return TRUE;
}

SmlBool smlItemHasData(SmlItem *item)
{
	smlAssert(item);
	return item->buffer ? TRUE : FALSE;
}

/** Returns the data of the item. The data will not be freed when the item is unrefd. After
 * this call, smlItemHasData will report FALSE */
SmlBool smlItemStealData(SmlItem *item, char **data, unsigned int *size, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, item, data, size, error);
	smlAssert(size);
	
	if (!smlItemCheck(item)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Item check failed");
		goto error;
	}
	
	*size = xmlBufferLength(item->buffer);
	*data = g_strndup(xmlBufferContent(item->buffer), *size);
	xmlBufferFree(item->buffer);
	item->buffer = NULL;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/** Returns a const pointer to the data of the item. the data will disappear when the data is derefd */
SmlBool smlItemGetData(SmlItem *item, char **data, unsigned int *size, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, item, data, size, error);
	
	if (!smlItemCheck(item)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Item check failed");
		goto error;
	}
	
	*data = (char *)xmlBufferContent(item->buffer);
	*size = xmlBufferLength(item->buffer);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

void smlItemSetSource(SmlItem *item, SmlLocation *source)
{
	smlAssert(item);
	smlAssert(source);
	
	item->source = source;
	smlLocationRef(source);
}

SmlLocation *smlItemGetSource(SmlItem *item)
{
	smlAssert(item);
	
	return item->source;
}

void smlItemSetTarget(SmlItem *item, SmlLocation *target)
{
	smlAssert(item);
	smlAssert(target);
	
	item->target = target;
	smlLocationRef(target);
}

SmlLocation *smlItemGetTarget(SmlItem *item)
{
	smlAssert(item);
	
	return item->target;
}

void smlItemSetRaw(SmlItem *item, SmlBool raw)
{
	smlAssert(item);
	
	item->raw = raw;
}

SmlCred *smlCredNewFromString(const char *type,
			 const char *format,
			 const char *data,
			 SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%s, %s, %s, %p)", __func__, data, type, format, error);

	SmlAuthType smlType = SML_AUTH_TYPE_UNKNOWN;
	SmlFormatType smlFormat = SML_FORMAT_TYPE_UNKNOWN;
	
	if (!type || !strcmp(type, SML_AUTH_BASIC)) {
		smlType = SML_AUTH_TYPE_BASIC;
	} else if (!strcmp(type, SML_AUTH_MD5)) {
		smlType = SML_AUTH_TYPE_MD5;
	} else {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unknown type - %s.", type);
		goto error;
	}

	if (!format || !strcmp(format, SML_BASE64)) {
		smlFormat = SML_FORMAT_TYPE_BASE64;
	} else {
		smlErrorSet(error, SML_ERROR_GENERIC, "SyncML credential: Unknown format - %s.", format);
		goto error;
	}

	if (!data)  {
		smlErrorSet(error, SML_ERROR_GENERIC, "Data is missing in %s.", __func__);
		goto error;
	}

	smlTrace(TRACE_EXIT, "%s", __func__);
	return smlCredNew(smlType, smlFormat, data, NULL, error);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;	
}

SmlCred *smlCredNewAuth(SmlAuthType type,
			const char *username,
			const char *password,
			SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%d, %s, %p)", __func__, type, username, error);
        char *data = NULL;
	SmlFormatType format = SML_FORMAT_TYPE_BASE64;

	// build plain data
	if (type == SML_AUTH_TYPE_BASIC) {
		char *plain = g_strjoin(":", username, password, NULL);
		data = g_base64_encode(plain, strlen(plain));
		g_free(plain);
	} else if (type == SML_AUTH_TYPE_MD5) {
		char *plain = g_strjoin(":", username, password, NULL);
		unsigned char digest[16];
		smlMD5GetDigest(plain, strlen(plain), digest);
		g_free(plain);
		data = smlMD5ToString(digest, NULL);
		g_free(digest);
	} else {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unknown type - %s (%s).", type, __func__);
		goto error;
	}

	// the username is sometimes required as location name in the source
	//     the username is readable from data     => OPTIONAL (e.g. syncml:auth-basic)
	//     the username is not readable from data => REQUIRED (e.g. syncml:auth-md5)
	SmlCred *cred = smlCredNew(type, format, data, username, error);
	g_free(data);

	smlTrace(TRACE_EXIT, "%s", __func__);
	return cred;
error:
	if (data)
		g_free(data);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;	
}

SmlCred *smlCredNew(SmlAuthType type,
		SmlFormatType format,
		const char *data,
		const char *username,
		SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%s, %d, %d, %p)", __func__, data, type, format, error);

        SmlCred *cred = smlTryMalloc0(sizeof(SmlCred), error);
        if (!cred)
                goto error;

	cred->type = type;
	cred->format = format;
	cred->data = g_strdup(data);
	if (username)
		cred->locName = g_strdup(username);
	else
		cred->locName = NULL;
	cred->refCount = 1;

	smlTrace(TRACE_EXIT, "%s: %p", __func__, cred);
	return cred;
error:
	if (cred->data)
		g_free(cred->data);
	if (cred->locName)
		g_free(cred->locName);
	if (cred)
		g_free(cred);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlCred *smlCredRef(SmlCred *cred)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, cred);
	smlAssert(cred);
	
	g_atomic_int_inc(&(cred->refCount));
	
	smlTrace(TRACE_EXIT, "%s: New refcount: %i", __func__, cred->refCount);
	return cred;
}

void smlCredUnref(SmlCred *cred)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, cred);
	smlAssert(cred);
	
	if (g_atomic_int_dec_and_test(&(cred->refCount))) {
		smlTrace(TRACE_INTERNAL, "Refcount == 0!");
		
		if (cred->data)
			g_free(cred->data);
		if (cred->locName)
			g_free(cred->locName);
			
		g_free(cred);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

/* FIXME: DEPRECATED */
void smlCredFree(SmlCred *cred)
{
	return smlCredUnref(cred);
}

void smlChalFree(SmlChal *chal)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, chal);
	smlAssert(chal);
	
	if (chal->nonce)
		g_free(chal->nonce);
	
	g_free(chal);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlMapItem *smlMapItemNew(const char *uid, const char *newuid, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%s, %s, %p)", __func__, uid, newuid, error);
	smlAssert(uid);
	smlAssert(newuid);
	
	SmlMapItem *item = smlTryMalloc0(sizeof(SmlMapItem), error);
	if (!item)
		goto error;
	item->refCount = 1;
	
	item->source = smlLocationNew(newuid, NULL, error);
	if (!item->source)
		goto error_free_item;
	
	item->target = smlLocationNew(uid, NULL, error);
	if (!item->target)
		goto error_free_source;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, item);
	return item;
	
error_free_source:
	smlLocationUnref(item->source);
error_free_item:
	g_free(item);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;	
}

SmlMapItem *smlMapItemRef(SmlMapItem *item)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, item);
	smlAssert(item);
	
	g_atomic_int_inc(&(item->refCount));
	
	smlTrace(TRACE_EXIT, "%s: New refcount: %i", __func__, item->refCount);
	return item;
}

void smlMapItemUnref(SmlMapItem *item)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, item);
	smlAssert(item);
	
	if (g_atomic_int_dec_and_test(&(item->refCount))) {
		smlTrace(TRACE_INTERNAL, "Refcount == 0!");
		
		if (item->source)
			smlLocationUnref(item->source);
		
		if (item->target)
			smlLocationUnref(item->target);
			
		g_free(item);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

/*@}*/
