#include "osl/apply_move/applyMove.h"
#include "osl/container/moveVector.h"
#include "osl/hash/hashKey.h"
#include "osl/state/numEffectState.h"
#include "osl/record/kisen.h"
#include "osl/record/csaRecord.h"
#include "osl/checkmate/dualDfpn.h"
#include "osl/eval/see.h"

#include <boost/accumulators/accumulators.hpp>
#include <boost/accumulators/statistics/mean.hpp>
#include <boost/accumulators/statistics/max.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/program_options.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/foreach.hpp>
#include <boost/format.hpp>
#include <iostream>
#include <fstream>

static void convert(const std::vector<std::string> &input_filename,
		    const std::string &output_kisen_filename,
		    size_t checkmate_limit, bool output_ipx)
{
  namespace acc = boost::accumulators;
  acc::accumulator_set<double, acc::features<acc::tag::max, acc::tag::mean> > accumulator;
  std::ofstream ofs(output_kisen_filename.c_str());
  osl::record::OKisenStream ks(ofs);

  boost::scoped_ptr<osl::record::KisenIpxWriter> ipx_writer;
  boost::scoped_ptr<std::ofstream> ipx_ofs;
  if (output_ipx)
  {
    const boost::filesystem::path ipx_path =
      boost::filesystem::change_extension(boost::filesystem::path(output_kisen_filename), ".ipx");
    const std::string ipx = ipx_path.file_string();
    ipx_ofs.reset(new std::ofstream(ipx.c_str()));
    ipx_writer.reset(new osl::record::KisenIpxWriter(*ipx_ofs));
  }

  for (size_t i = 0; i < input_filename.size(); ++i)
  {
    osl::KisenFile kisen(input_filename[i]);
    osl::KisenIpxFile ipx(kisen.ipxFileName());
    for (size_t j=0; j<kisen.size(); ++j) 
    {
      osl::NumEffectState state = kisen.getInitialState();
      osl::vector<osl::Move> moves = kisen.getMoves(j);
      osl::vector<osl::Move> new_moves;
      osl::record::Record new_record;
      new_record.setPlayer(osl::BLACK, ipx.getPlayer(j, osl::BLACK));
      new_record.setPlayer(osl::WHITE, ipx.getPlayer(j, osl::WHITE));
      osl::SimpleState record_state = state;
      osl::record::RecordVisitor visitor;
      visitor.setState(&record_state);
      visitor.setRecord(&new_record);
      osl::DualDfpn dfpn;
      for (size_t k=0; k<moves.size(); ++k) 
      {
	const int see = osl::See::see
	  (state, moves[k], state.pin(state.getTurn()), state.pin(alt(state.getTurn())));
	visitor.addMoveAndAdvance(moves[k]);
	new_moves.push_back(moves[k]);
	osl::ApplyMoveOfTurn::doMove(state, moves[k]);
	if (state.inCheck() && see < 0
	    && dfpn.isLosingState(checkmate_limit, state, 
				  osl::HashKey(state), osl::PathEncoding(state.getTurn())))
	  break;
      }
      new_record.setResult(state.getTurn() == osl::BLACK
			   ? osl::Record::WHITE_WIN : osl::Record::BLACK_WIN);
      accumulator(moves.size() - new_record.getMoves().size());
      if (new_record.getMoves().size() >= 256)
	std::cerr << "long record " << j << ' ' << new_record.getMoves().size() << "\n";
      ks.save(&new_record);
      if (output_ipx)
      {
	ipx_writer->save(new_record, 0, 0, "", "");
      }
      if ((j % 1000) == 999)
	std::cerr << input_filename[i] << " " << j
		  << " max " << acc::max(accumulator)
		  << " mean " << acc::mean(accumulator) << "\n";
    }
    std::cerr << input_filename[i]
	      << " max " << acc::max(accumulator)
	      << " mean " << acc::mean(accumulator) << "\n";
  }
}

int main(int argc, char **argv)
{
  bool output_ipx;
  std::string kisen_filename;
  size_t checkmate_limit;
  boost::program_options::options_description command_line_options;
  command_line_options.add_options()
    ("output-ipx",
     boost::program_options::value<bool>(&output_ipx)->default_value(true),
     "Whether output IPX file in addition to KIF file")
    ("output-kisen-filename,o",
     boost::program_options::value<std::string>(&kisen_filename)->
     default_value("test.kif"),
     "Output filename of Kisen file")
    ("checkmate-limit,l",
     boost::program_options::value<size_t>(&checkmate_limit)->default_value(1000),
     "Whether output IPX file in addition to KIF file")
    ("input-file", boost::program_options::value< std::vector<std::string> >(),
     "input files in kisen format")
    ("help", "Show help message");
  boost::program_options::variables_map vm;
  boost::program_options::positional_options_description p;
  p.add("input-file", -1);

  try
  {
    boost::program_options::store(
      boost::program_options::command_line_parser(
	argc, argv).options(command_line_options).positional(p).run(), vm);
    boost::program_options::notify(vm);
    if (vm.count("help"))
    {
      std::cerr << "Usage: " << argv[0] << " [options] kisen-files\n";
      std::cerr << "       " << argv[0] << " [options]\n";
      std::cout << command_line_options << std::endl;
      return 0;
    }
  }
  catch (std::exception &e)
  {
    std::cerr << "error in parsing options" << std::endl
	      << e.what() << std::endl;
    std::cerr << "Usage: " << argv[0] << " [options] kisen-files\n";
    std::cerr << "       " << argv[0] << " [options]\n";
    std::cerr << command_line_options << std::endl;
    return 1;
  }

  std::vector<std::string> files;
  if (vm.count("input-file"))
    files = vm["input-file"].as<std::vector<std::string> >();

  convert(files, kisen_filename, checkmate_limit, output_ipx);
  return 0;
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
