#include "osl/record/csa.h"
#include "osl/record/csaIOError.h"
#include "osl/state/simpleState.h"
#include "osl/pieceTable.h"
#include <iostream>
#include <stdexcept>
#include <cassert>
#include <string>

/* ------------------------------------------------------------------------- */

osl::Player osl::record::csa::
charToPlayer(char c)
{
  if(c=='+') 
    return BLACK;
  if(c=='-') 
    return WHITE;
  std::cerr << "oops " << (int)c << "\n";
  throw CsaIOError("not a csa PlayerCharacter "+std::string(1,c));
}

const osl::Position osl::record::csa::
strToPos(const std::string& s)
{
  int x=s.at(0)-'0';
  int y=s.at(1)-'0';
  if(x==0 && y==0) 
    return Position::STAND();
  return Position(x,y);
}

osl::Ptype osl::record::csa::
strToPtype(const std::string& s)
{
  for(int i=0;i<16;i++){
    if(s == Ptype_Table.getCsaName(static_cast<Ptype>(i))) 
      return static_cast<Ptype>(i);
  }
  throw CsaIOError("unknown std::string in csa::strToPtype "+s);
}

const osl::Move osl::record::csa::
strToMove(const std::string& s,const SimpleState& state)
{
  if (s == "%KACHI")
    return Move::DeclareWin();
  if (s == "%TORYO")
    return Move::INVALID();
  if (s == "%PASS")		// FIXME: not in CSA protocol
    return Move::PASS(state.getTurn());

  Player pl=csa::charToPlayer(s.at(0));
  Position fromPos=csa::strToPos(s.substr(1,2));
  Position toPos=csa::strToPos(s.substr(3,2));
  Ptype ptype=csa::strToPtype(s.substr(5,2));
  if(fromPos==Position::STAND()){
    if (isPromoted(ptype))
      throw CsaIOError("drop with promote ?! in csa::strToMove "+s);
    return Move(toPos,ptype,pl);
  }
  else{
    Piece p0=state.getPieceAt(fromPos);
    Piece p1=state.getPieceAt(toPos);
    Ptype capturePtype=p1.ptype();
    bool isPromote=(p0.ptype()!=ptype);
    if (! ((p0.ptype()==ptype)||(p0.ptype()==unpromote(ptype))))
      throw CsaIOError("bad promotion in csa::strToMove "+s);
    return Move(fromPos,toPos,ptype,
		capturePtype,isPromote,pl);
  }
}

/* ------------------------------------------------------------------------- */
const std::string osl::record::csa::
show(Player player, std::string& buf, size_t offset)
{
  assert(buf.size() >= offset+1);
  buf[offset] = (player==BLACK) ? '+' : '-';
  return buf;
}

const std::string osl::record::csa::
show(Move move, std::string& buf)
{
  assert(buf.capacity() >= 7);
  buf.resize(7);
  if (move == Move::DeclareWin())
    return buf = "%KACHI";
  if (move.isInvalid())
    return buf = "%TORYO";
  if (move.isPass())
    return buf = "%PASS";		// FIXME: not in CSA protocol
  show(move.player(), buf);
  show(move.from(), buf, 1);
  show(move.to(), buf, 3);
  show(move.ptype(), buf, 5);
  return buf;
}

const std::string osl::record::csa::
show(Position pos, std::string& buf, size_t offset)
{
  assert(buf.size() >= offset+2);
  if (pos.isPieceStand()) 
  {
    buf[0+offset] = '0';
    buf[1+offset] = '0';
    return buf;
  }
  const int x = pos.x();
  const int y = pos.y();
  buf[offset+0] = x + '0';
  buf[offset+1] = y + '0';
  return buf;
}

const std::string osl::record::csa::
show(Ptype ptype, std::string& buf, size_t offset)
{
  assert(buf.size() >= offset+2);
  const char *name = Ptype_Table.getCsaName(ptype);
  buf[0+offset] = name[0];
  buf[1+offset] = name[1];
  return buf;
}

const std::string osl::record::csa::
show(Move move)
{
  // NOTE: copy コピーを返すので dangling pointer ではない
  std::string buf("+7776FU");
  return show(move, buf);
}

const std::string osl::record::csa::
fancyShow(Move move)
{
  std::string ret = show(move);
  if (move.isNormal()) {
    if (move.capturePtype() != PTYPE_EMPTY)
      ret += "x" + show(move.capturePtype());
    if (move.isPromote())
      ret += '+';
  }
  return ret;
}

const std::string osl::record::csa::
show(Player player)
{
  std::string buf("+");
  return show(player, buf);
}

const std::string osl::record::csa::
show(Position position)
{
  std::string buf("00");
  return show(position, buf);
}

const std::string osl::record::csa::
show(Ptype ptype)
{
  std::string buf("OU");
  return show(ptype, buf);
}

const std::string osl::record::csa::
show(Piece piece)
{
  if (piece.isEdge())
    return "   ";
  if (piece.isEmpty())
    return " * ";

  assert(piece.isPiece() && isPiece(piece.ptype()));
  assert(unpromote(piece.ptype()) == Piece_Table.getPtypeOf(piece.number()));
  return show(piece.owner()) 
    + show(piece.ptype());
}

/* ------------------------------------------------------------------------- */

std::ostream& osl::csaShow(std::ostream& os,const Position pos)
{
  return os << record::csa::show(pos);
}

std::ostream& osl::csaShow(std::ostream& os, const Piece piece)
{
  return os << record::csa::show(piece);
}

std::ostream& osl::csaShow(std::ostream& os,const osl::Ptype ptype)
{
  return os << record::csa::show(ptype);
}

std::ostream& osl::csaShow(std::ostream& os,const Move move)
{
  return os << record::csa::show(move);
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
