#include "../recursiveDoUndoMove.h"
#include "osl/record/csaString.h"
#include "osl/record/csaRecord.h"
#include "osl/apply_move/applyMove.h"
#include "osl/state/numEffectState.h"
#include "osl/state/simpleState.tcc"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <fstream>
#include <iostream>
class pawnMaskStateTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE( pawnMaskStateTest );
  CPPUNIT_TEST( testShow );
  CPPUNIT_TEST( testIsValid );
  CPPUNIT_TEST( testDoUndoMove );
  CPPUNIT_TEST( testDoUndoMoveFiles );
  CPPUNIT_TEST( testPieceConsistent );
  CPPUNIT_TEST( testCaptureBug );
  CPPUNIT_TEST( testPieceConsistent );
  CPPUNIT_TEST_SUITE_END();
 private:
  osl::SimpleState state;
 public:
  void setUp();
  void testShow();
  void testIsValid();
  void testPieceHasEffectTo();
  void testDoUndoMove();
  void testDoUndoMoveFiles();
  void testCaptureBug();
  void testPieceConsistent();
};
CPPUNIT_TEST_SUITE_REGISTRATION(pawnMaskStateTest);

using namespace osl;
extern bool isShortTest;

void pawnMaskStateTest::setUp(){
  state=SimpleState(CsaString(
"P1+NY+TO *  *  *  * -OU-KE-KY\n"
"P2 *  *  *  *  * -GI-KI *  *\n"
"P3 * +RY *  * +UM * -KI-FU-FU\n"
"P4 *  * +FU-FU *  *  *  *  *\n"
"P5 *  * -KE * +FU *  * +FU *\n"
"P6-KE *  * +FU+GI-FU *  * +FU\n"
"P7 *  * -UM *  *  *  *  *  *\n"
"P8 *  *  *  *  *  *  *  *  * \n"
"P9 * +OU * -GI *  *  *  * -NG\n"
"P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
"P-00KI00KY00FU00FU\n"
"P-00AL\n"
"+\n").getInitialState());
}

void pawnMaskStateTest::testShow(){
  CPPUNIT_ASSERT(state.isConsistent() || state.isConsistent(true) );
  if (! isShortTest)
  {
    std::cout << state;
    SimpleState state1(HIRATE);
    std::cout << state1;
  }
}

void pawnMaskStateTest::testIsValid(){
  // 後手の手
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(6,9),Position(7,8),SILVER,PTYPE_EMPTY,false,WHITE),false)==false);
  // 空白以外へのput
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(2,1),PAWN,BLACK),false)==false);
  // 持っていない駒
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(9,9),SILVER,BLACK),false)==false);
  // 二歩
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(7,1),PAWN,BLACK),false)==false);
  // 二歩ではない
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(8,2),PAWN,BLACK),false)==true);
  // 動けない場所
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(4,1),PAWN,BLACK),false)==false);
  // fromにあるのがその駒か
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(8,2),Position(7,3),PROOK,PTYPE_EMPTY,false,BLACK),false)==false);
  // toにあるのが，相手の駒か空白か?
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(8,1),Position(9,1),PPAWN,PLANCE,false,BLACK),false)==false);
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(8,1),Position(7,1),PPAWN,PTYPE_EMPTY,false,BLACK),false)==true);
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(5,3),Position(4,2),PBISHOP,SILVER,false,BLACK),false)==true);

      // その offsetの動きがptypeに関してvalidか?
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(8,1),Position(9,2),PPAWN,PTYPE_EMPTY,false,BLACK),false)==false);
      // 離れた動きの時に間が全部空いているか?
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(8,3),Position(6,3),PROOK,PTYPE_EMPTY,false,BLACK),false)==true);
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(8,3),Position(4,3),PROOK,PTYPE_EMPTY,false,BLACK),false)==false);
      // capturePtypeが一致しているか?
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(5,3),Position(4,2),PBISHOP,PTYPE_EMPTY,false,BLACK),false)==false);
      // promoteしている時にpromote可能か
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(8,1),Position(7,1),PPAWN,PTYPE_EMPTY,true,BLACK),false)==false);
      // promoteしていない時に強制promoteでないか?
#if 0
  // 王手をかけられる
  // 現在のpawnMaskStateは判断できない
  CPPUNIT_ASSERT(state.isValidMove(Move(Position(8,9),Position(8,8),KING,PTYPE_EMPTY,false,BLACK),false)==false);
#endif

  const SimpleState s2(CsaString(
"P1-KY *  *  *  *  *  * -KE-KY\n"
"P2 *  *  *  *  *  * -KI * -OU\n"
"P3-FU *  *  *  *  * -GI-GI * \n"
"P4 *  * -FU * -FU * -FU-FU-FU\n"
"P5 * -FU *  *  * +FU * +FU * \n"
"P6 *  * +FU * +FU+KA+FU * +FU\n"
"P7+FU * -TO *  * +KI+KE *  * \n"
"P8+HI *  *  *  *  * +GI+OU * \n"
"P9+KY *  * -RY * +KI *  * +KY\n"
"P+00FU00FU00FU00KE00KI\n"
"P-00KE00GI00KA\n"
"+\n").getInitialState());
  const Move illegal = Move(Position(2,5),Position(2,4),PPAWN,PAWN,true,BLACK);
  CPPUNIT_ASSERT(! s2.isValidMove(illegal,false));
}

void pawnMaskStateTest::testPieceHasEffectTo(){
}

template<typename State>
struct CheckOnBoard{
  State& state;
  int ret;
  CheckOnBoard(State& s) : state(s)
  {
  }
  void operator()(Position /*to*/){
    CPPUNIT_ASSERT(state.isConsistent());
    int count=0;
    for(int num=0;num<40;num++)
      if(state.isOnBoard(num)) count++;
    ret=count;
  }
};

const Move b5364uma = Move(Position(5,3),Position(6,4),PBISHOP,PAWN,false,BLACK);
void pawnMaskStateTest::testDoUndoMove(){
  CheckOnBoard<SimpleState> checkOnBoard(state);
  ApplyMoveOfTurn::doUndoMove(state, b5364uma, checkOnBoard);
  CPPUNIT_ASSERT(state.isConsistent());
  CPPUNIT_ASSERT_EQUAL(25, checkOnBoard.ret);
}



void pawnMaskStateTest::testDoUndoMoveFiles(){
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=100;
  if (isShortTest)
      count=10;
  std::string fileName;
  while((ifs >> fileName) && (++i<count)) {
    if(fileName == "") 
	break;
    if (! isShortTest)
      std::cerr << fileName << " ";

    fileName = OslConfig::testCsaFile(fileName);
    Record rec=CsaFile(fileName).getRecord();
    SimpleState state(rec.getInitialState());
    vector<osl::Move> moves=rec.getMoves();
  
    size_t depth=0;
    if (moves.begin()->player()==BLACK){
      RecursiveDoUndoMove<BLACK,SimpleState>
	func(state, &*moves.begin(), &*moves.end(), &depth);
      ApplyMove<BLACK>::doUndoMove(state, *moves.begin(), func);
    }
    else{
      RecursiveDoUndoMove<WHITE,SimpleState>
	func(state, &*moves.begin(), &*moves.end(), &depth);
      ApplyMove<WHITE>::doUndoMove(state, *moves.begin(), func);
    }
    CPPUNIT_ASSERT_EQUAL(moves.size(), depth);
  }
}


void pieceNumConsistentBeforeCapture(SimpleState& state)
{
  CPPUNIT_ASSERT(state.isConsistent());
  for (int i=0; i<Piece::SIZE; ++i)
  {
    CPPUNIT_ASSERT(state.isOnBoard(i));
    CPPUNIT_ASSERT(! state.isOffBoard(i));

    const Piece target = state.getPieceOf(i);
    const Position pos = target.position();
    const Piece pieceOfSameLocation = state.getPieceAt(pos);
    CPPUNIT_ASSERT_EQUAL(target, pieceOfSameLocation);
  }
}

void pawnMaskStateTest::testPieceConsistent()
{
#if 0
  SimpleState s(HIRATE); // これでもよい?
#endif
  SimpleState s(CsaString(
"P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
"P2 * -HI *  *  *  *  * -KA * \n"
"P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
"P4 *  *  *  *  *  *  *  *  * \n"
"P5 *  *  *  *  *  *  *  *  * \n"
"P6 *  *  *  *  *  *  *  *  * \n"
"P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
"P8 * +KA *  *  *  *  * +HI * \n"
"P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
"+\n").getInitialState());
  pieceNumConsistentBeforeCapture(s);
  ApplyMoveOfTurn::doMove
    (s,
     Move(Position(7,7),Position(7,6),PAWN,PTYPE_EMPTY,false,BLACK));
  pieceNumConsistentBeforeCapture(s);
  ApplyMoveOfTurn::doMove
    (s,
     Move(Position(3,3),Position(3,4),PAWN,PTYPE_EMPTY,false,WHITE));
  pieceNumConsistentBeforeCapture(s);
  const Move move = Move(Position(7,6),Position(7,5),PAWN,PTYPE_EMPTY,false,BLACK);
  ApplyMoveOfTurn::doMove(s,move);
  const Piece expected = s.getPieceAt(move.to());
  pieceNumConsistentBeforeCapture(s);
  for (int i=0; i<Piece::SIZE; ++i)
  {
    CPPUNIT_ASSERT(s.isOnBoard(i));
    CPPUNIT_ASSERT(! s.isOffBoard(i));
    const Piece target = s.getPieceOf(i);
    CPPUNIT_ASSERT((target.position() != move.to())
		   || (expected == target));
    // 動いた駒自身でなければ，駒が重なることはない
  }
}

void pawnMaskStateTest::testCaptureBug()
{
  SimpleState s(CsaString(
    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
    "P2 * -HI *  *  *  *  * -KA * \n"
    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
    "P4 *  *  *  *  *  *  *  *  * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  *  *  *  *  *  *  * \n"
    "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
    "P8 * +KA *  *  *  *  * +HI * \n"
    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
    "+\n").getInitialState());
  ApplyMoveOfTurn::doMove
    (s,
     Move(Position(7,7),Position(7,6),PAWN,PTYPE_EMPTY,false,BLACK));
  ApplyMoveOfTurn::doMove
    (s,
     Move(Position(3,3),Position(3,4),PAWN,PTYPE_EMPTY,false,WHITE));
  CPPUNIT_ASSERT(s.isConsistent());

  const Position from = Position(8,8);
  const Position to = Position(2,2);
  const Move move = Move(from,to,PBISHOP,BISHOP,true,BLACK);
  CPPUNIT_ASSERT(s.isValidMove(move,true));
  ApplyMoveOfTurn::doMove(s, move);
  CPPUNIT_ASSERT(s.isConsistent());
  const Piece expected = s.getPieceAt(move.to());
  for (int i=0; i<Piece::SIZE; ++i)
  {
    if(s.isOnBoard(i)){
      const Piece target = s.getPieceOf(i);
      CPPUNIT_ASSERT((target.position() != move.to())
		     || (expected == target)); // 動いた駒自身でなければ，駒が重なることはない
    }
  }
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
