/**
 * @file element.c Element functions
 *
 * $Id: element.c,v 1.3 2001/07/29 03:35:34 chipx86 Exp $
 *
 * @Copyright (C) 1999-2001 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <gnurdf.h>
#include <libgnurdf/internal.h>

int
rdfGetElementType(RdfElement *element)
{
	if (element == NULL)
		return -1;

	if (!strcmp(element->name, "Bag"))         return RDF_BAG;
	if (!strcmp(element->name, "Seq"))         return RDF_SEQ;
	if (!strcmp(element->name, "Alt"))         return RDF_ALT;
	if (!strcmp(element->name, "Description")) return RDF_DESC;

	return 0;
}

char *
rdfGetElementValue(RdfElement *element)
{
	if (element == NULL)
		return NULL;
		
	if (element->content != NULL)
		return strdup(element->content);

	return xmlNodeListGetString(rdfGetElementSchema(element),
								element->xmlChildrenNode, 1);
}

const char *
rdfGetElementPropertyName(RdfElement *element)
{
	if (element == NULL)
		return NULL;
	
	return element->name;
}

RdfNamespace *
rdfGetElementNamespace(RdfElement *element)
{
	if (element == NULL)
		return NULL;

	return element->ns;
}

void
rdfSetElementValue(RdfElement *element, const char *value)
{
	xmlChar *tmp;
	RdfSchema *schema;
	
	if (element->xmlChildrenNode != NULL)
	{
		fprintf(stderr, "libgnurdf: rdfSetElementValue: "
				"Element %s has children\n",
		        element->name);
		return;
	}

	if (element->content != NULL)
		free(element->content);
	
	if (element->xmlChildrenNode != NULL)
		xmlFreeNodeList(element->xmlChildrenNode);
	
	schema = rdfGetElementSchema(element);
	
	tmp = xmlEncodeEntitiesReentrant(schema, value);
	element->xmlChildrenNode = xmlStringGetNodeList(schema, tmp);

	free(tmp);
}

void
rdfSetElementResource(RdfElement *element, const char *uri)
{
	if (element == NULL || uri == NULL)
		return;
	
	rdfSetNsAttribute(element, "resource", uri,
					  rdfGetRdfNamespace(rdfGetElementSchema(element)));
}

char *
rdfGetElementResource(RdfElement *element)
{
	if (element == NULL)
		return NULL;
	
	return rdfGetNsAttribute(element, "resource",
							 rdfGetRdfNamespace(rdfGetElementSchema(element)));
}

RdfSchema *
rdfGetElementSchema(RdfElement *element)
{
	if (element == NULL)
		return NULL;

	return element->doc;
}
