/* gnome-db-canvas-item.c
 *
 * Copyright (C) 2002 - 2006 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gtk/gtk.h>
#include <libgda/libgda.h>
#include <libgnomedb/marshal.h>
#include "gnome-db-decl.h"
#include "gnome-db-canvas-item.h"
#include "gnome-db-canvas-tip.h"
#include "gnome-db-canvas-cursor.h"


static void gnome_db_canvas_item_class_init (GnomeDbCanvasItemClass * class);
static void gnome_db_canvas_item_init       (GnomeDbCanvasItem * item);
static void gnome_db_canvas_item_finalize    (GObject   * object);

static void gnome_db_canvas_item_set_property    (GObject *object,
					    guint param_id,
					    const GValue *value,
					    GParamSpec *pspec);
static void gnome_db_canvas_item_get_property    (GObject *object,
					    guint param_id,
					    GValue *value,
					    GParamSpec *pspec);
static void m_drag_action  (GnomeDbCanvasItem *citem, GnomeDbCanvasItem *dragged_from, GnomeDbCanvasItem * dragged_to);

struct _GnomeDbCanvasItemPrivate
{
	gboolean            moving;
	double              xstart;
	double              ystart;
	gboolean            allow_move;
	gboolean            allow_drag;
	gchar              *tooltip_text;
	GdaGraphItem        *graph_item;
};

enum
{
	MOVED,
	MOVING,
	SHIFTED,
	DRAG_ACTION,
	LAST_SIGNAL
};

enum
{
	PROP_0,
	PROP_ALLOW_MOVE,
	PROP_ALLOW_DRAG,
	PROP_TOOLTIP_TEXT,
	PROP_GRAPH_ITEM
};

static gint gnome_db_canvas_item_signals[LAST_SIGNAL] = { 0, 0, 0, 0 };

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *base_parent_class = NULL;

GType
gnome_db_canvas_item_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbCanvasItemClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_canvas_item_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbCanvasItem),
			0,
			(GInstanceInitFunc) gnome_db_canvas_item_init
		};
		type = g_type_register_static (GNOME_TYPE_CANVAS_GROUP, "GnomeDbCanvasItem", &info, 0);
	}

	return type;
}


static void
gnome_db_canvas_item_class_init (GnomeDbCanvasItemClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	base_parent_class = g_type_class_peek_parent (class);

	gnome_db_canvas_item_signals[MOVED] =
		g_signal_new ("moved",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbCanvasItemClass, moved),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	gnome_db_canvas_item_signals[MOVING] =
		g_signal_new ("moving",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbCanvasItemClass, moving),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	gnome_db_canvas_item_signals[SHIFTED] =
		g_signal_new ("shifted",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbCanvasItemClass, shifted),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	gnome_db_canvas_item_signals[DRAG_ACTION] =
		g_signal_new ("drag_action",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbCanvasItemClass, drag_action),
			      NULL, NULL,
			      marshal_VOID__POINTER_POINTER,
			      G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_POINTER);

	class->moved = NULL;
	class->moving = NULL;
	class->shifted = NULL;
	class->drag_action = m_drag_action;
	object_class->finalize = gnome_db_canvas_item_finalize;

	/* virtual funstionc */
	class->extra_event = NULL;

	/* Properties */
	object_class->set_property = gnome_db_canvas_item_set_property;
	object_class->get_property = gnome_db_canvas_item_get_property;

	g_object_class_install_property
                (object_class, PROP_ALLOW_MOVE,
                 g_param_spec_boolean ("allow_move", NULL, NULL, TRUE, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property
                (object_class, PROP_ALLOW_DRAG,
                 g_param_spec_boolean ("allow_drag", NULL, NULL, FALSE, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property
		(object_class, PROP_TOOLTIP_TEXT,
		 g_param_spec_string ("tip_text", NULL, NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (
		object_class, PROP_GRAPH_ITEM,
		g_param_spec_object ("graph_item", NULL, NULL, GDA_TYPE_GRAPH_ITEM, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
m_drag_action  (GnomeDbCanvasItem *citem, GnomeDbCanvasItem *dragged_from, GnomeDbCanvasItem *dragged_to)
{
	GnomeCanvasItem *parent;

	/* make sure the parent GnomeDbCanvasItem forwards the DND info... */
	g_object_get (G_OBJECT (citem), "parent", &parent, NULL);
	if (GNOME_DB_IS_CANVAS_ITEM (parent)) {
#ifdef debug_signal
		g_print (">> 'DRAG_ACTION' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (parent), gnome_db_canvas_item_signals[DRAG_ACTION], 0, 
			       dragged_from, dragged_to);
#ifdef debug_signal
		g_print ("<< 'DRAG_ACTION' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
	}
}

static gboolean item_event(GnomeDbCanvasItem *citem, GdkEvent *event, gpointer data);

static void
gnome_db_canvas_item_init (GnomeDbCanvasItem * item)
{
	item->priv = g_new0 (GnomeDbCanvasItemPrivate, 1);
	item->priv->moving = FALSE;
	item->priv->xstart = 0;
	item->priv->ystart = 0;
	item->priv->allow_move = TRUE;
	item->priv->allow_drag = FALSE;
	item->priv->tooltip_text = NULL;
	item->priv->graph_item = NULL;
	
	g_signal_connect (G_OBJECT (item), "event",
			  G_CALLBACK (item_event), NULL);
}

static void graph_item_destroyed_cb (GdaGraphItem *item, GnomeDbCanvasItem *citem);
static void graph_item_moved_cb (GdaGraphItem *item, GnomeDbCanvasItem *citem);

static void
gnome_db_canvas_item_finalize (GObject   * object)
{
	GnomeDbCanvasItem *citem;
	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_CANVAS_ITEM (object));
	
	citem = GNOME_DB_CANVAS_ITEM (object);
	if (citem->priv) {
		if (citem->priv->graph_item)
			g_object_unref (citem->priv->graph_item);

		if (citem->priv->graph_item)
			graph_item_destroyed_cb (citem->priv->graph_item, citem);

		if (citem->priv->tooltip_text) 
			g_free (citem->priv->tooltip_text);

		g_free (citem->priv);
		citem->priv = NULL;
	}

	/* for the parent class */
	base_parent_class->finalize (object);
}

static void 
gnome_db_canvas_item_set_property    (GObject *object,
				guint param_id,
				const GValue *value,
				GParamSpec *pspec)
{
	GnomeDbCanvasItem *citem = NULL;
	const gchar *str = NULL;
	GObject* propobject = NULL;

	citem = GNOME_DB_CANVAS_ITEM (object);

	switch (param_id) {
	case PROP_ALLOW_MOVE:
		citem->priv->allow_move = g_value_get_boolean (value);
		if (citem->priv->allow_move && citem->priv->allow_drag)
			citem->priv->allow_drag = FALSE;
		break;
	case PROP_ALLOW_DRAG:
		citem->priv->allow_drag = g_value_get_boolean (value);
		if (citem->priv->allow_drag && citem->priv->allow_move)
			citem->priv->allow_move = FALSE;
		break;
	case PROP_TOOLTIP_TEXT:
		str = g_value_get_string (value);
		if (citem->priv->tooltip_text) {
			g_free (citem->priv->tooltip_text);
			citem->priv->tooltip_text = NULL;
		}
		if (str)
			citem->priv->tooltip_text = g_strdup (str);
		break;
	case PROP_GRAPH_ITEM:
		propobject = g_value_get_object (value);
		if (propobject == G_OBJECT (citem->priv->graph_item))
			return;

		if (citem->priv->graph_item) {
			g_object_unref (citem->priv->graph_item);
		}

		if (citem->priv->graph_item) {
			graph_item_destroyed_cb (citem->priv->graph_item, citem);
			citem->priv->graph_item = NULL;
		}

		if (propobject) {
			g_return_if_fail (GDA_IS_GRAPH_ITEM (propobject));
			gda_object_connect_destroy (propobject, G_CALLBACK (graph_item_destroyed_cb), citem);
			g_signal_connect (G_OBJECT (propobject), "moved",
					  G_CALLBACK (graph_item_moved_cb), citem);

			/* set the position according to the GdaGraphItem */
			citem->priv->graph_item = GDA_GRAPH_ITEM(propobject);
			g_object_ref (propobject);

			graph_item_moved_cb (citem->priv->graph_item, citem);
		}
	}
}

static void
gnome_db_canvas_item_get_property    (GObject *object,
				guint param_id,
				GValue *value,
				GParamSpec *pspec)
{
	TO_IMPLEMENT;
}

/**
 * gnome_db_canvas_item_get_canvas
 * @item: a #GnomeDbCanvasItem object
 *
 * Get the #GnomeDbCanvas on which @item is drawn
 *
 * Returns: the #GnomeDbCanvas widget
 */
GnomeDbCanvas *
gnome_db_canvas_item_get_canvas (GnomeDbCanvasItem *item)
{
	g_return_val_if_fail (item && GNOME_DB_IS_CANVAS_ITEM (item), NULL);
	return (GnomeDbCanvas *) GNOME_CANVAS_ITEM (item)->canvas;
}

/**
 * gnome_db_canvas_item_get_graph_item
 * @item: a #GnomeDbCanvasItem object
 *
 * Get the associated #GdaGraphItem to @item.
 *
 * Returns: the #GdaGraphItem, or %NULL
 */
GdaGraphItem *
gnome_db_canvas_item_get_graph_item (GnomeDbCanvasItem *item)
{
	g_return_val_if_fail (item && GNOME_DB_IS_CANVAS_ITEM (item), NULL);
	g_return_val_if_fail (item->priv, NULL);

	return item->priv->graph_item;
}

static void
graph_item_destroyed_cb (GdaGraphItem *item, GnomeDbCanvasItem *citem)
{
	g_assert (citem->priv->graph_item == item);
	g_signal_handlers_disconnect_by_func (G_OBJECT (item),
					      G_CALLBACK (graph_item_destroyed_cb), citem);
	g_signal_handlers_disconnect_by_func (G_OBJECT (item),
					      G_CALLBACK (graph_item_moved_cb), citem);
	citem->priv->graph_item = NULL;
}

static void
graph_item_moved_cb (GdaGraphItem *item, GnomeDbCanvasItem *citem)
{
	gdouble x, y;
	gdouble xold, yold;

	g_assert (citem->priv->graph_item == item);
	gda_graph_item_get_position (item, &x, &y);
	/*g_object_set (G_OBJECT (citem), "x", x, "y", y, NULL); => DOES NOT WORK */
	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (citem), &xold, &yold, NULL, NULL);
	gnome_canvas_item_move (GNOME_CANVAS_ITEM (citem), x-xold, y-yold);
#ifdef debug_signal
	g_print (">> 'SHIFTED' from %s::graph_item_moved_cb()\n", __FILE__);
#endif
	g_signal_emit (G_OBJECT (citem), gnome_db_canvas_item_signals[SHIFTED], 0);
#ifdef debug_signal
	g_print ("<< 'SHIFTED' from %s::graph_item_moved_cb()\n", __FILE__);
#endif
}

static void end_of_drag_cb (GnomeDbCanvasItem *citem, GObject *cursor);
static void end_of_drag_cb_d (GnomeCanvasItem *ci, GnomeDbCanvasItem *citem);
static gboolean add_tip_timeout (GnomeDbCanvasItem *citem);
static gboolean display_tip_timeout (GnomeDbCanvasItem *citem);
static gboolean
item_event (GnomeDbCanvasItem *citem, GdkEvent *event, gpointer data)
{
	GnomeDbCanvasItemClass *class = GNOME_DB_CANVAS_ITEM_CLASS (G_OBJECT_GET_CLASS (citem));
	gboolean done = FALSE;
	GnomeCanvasItem *ci;
	GObject   *obj;
	guint id;
	gdouble pos;

	switch (event->type) {
	case GDK_ENTER_NOTIFY:
		/* Drag management */
		ci = g_object_get_data (G_OBJECT (gnome_canvas_root (GNOME_CANVAS_ITEM (citem)->canvas)),
					  "dragged_from");
		if (ci && GNOME_DB_IS_CANVAS_ITEM (ci)) {
#ifdef debug_signal
			g_print (">> 'DRAG_ACTION' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
			g_signal_emit (G_OBJECT (citem), gnome_db_canvas_item_signals[DRAG_ACTION], 0, ci, citem);
#ifdef debug_signal
			g_print ("<< 'DRAG_ACTION' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
			g_object_set_data (G_OBJECT (gnome_canvas_root (GNOME_CANVAS_ITEM (citem)->canvas)),
					   "dragged_from", NULL);
		}
		done = FALSE;
		break;
	case GDK_LEAVE_NOTIFY:
		/* remove tooltip */
		obj = g_object_get_data (G_OBJECT (citem), "tip");
                if (obj) {
                        gtk_object_destroy (GTK_OBJECT (obj));
			g_object_set_data (G_OBJECT (citem), "tip", NULL);
		}

                id = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (citem), "addtipid"));
                if (id != 0) {
                        g_source_remove (id);
                        g_object_set_data (G_OBJECT (citem), "addtipid", NULL);
                }
                id = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (citem), "displaytipid"));
                if (id != 0) {
                        g_source_remove (id);
                        g_object_set_data (G_OBJECT (citem), "displaytipid", NULL);
                }
		done = FALSE;
		break;
	case GDK_BUTTON_PRESS:
		switch (((GdkEventButton *) event)->button) {
		case 1:
			if (citem->priv->allow_move) {
				/* movement management */
				gnome_canvas_item_raise_to_top (GNOME_CANVAS_ITEM (citem));
				citem->priv->xstart = ((GdkEventMotion *) event)->x;
				citem->priv->ystart = ((GdkEventMotion *) event)->y;
				citem->priv->moving = TRUE;
				done = TRUE;
			}
			if (citem->priv->allow_drag) {
				/* remove tooltip */
				obj = g_object_get_data (G_OBJECT (citem), "tip");
				if (obj) {
					gtk_object_destroy (GTK_OBJECT (obj));
					g_object_set_data (G_OBJECT (citem), "tip", NULL);
				}
				id = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (citem), "addtipid"));
				if (id != 0) {
					g_source_remove (id);
					g_object_set_data (G_OBJECT (citem), "addtipid", NULL);
				}
				id = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (citem), "displaytipid"));
				if (id != 0) {
					g_source_remove (id);
					g_object_set_data (G_OBJECT (citem), "displaytipid", NULL);
				}

				/* start dragging */
				if (class->extra_event)
					(class->extra_event) (citem, GDK_LEAVE_NOTIFY);

				ci = gnome_canvas_item_new (GNOME_CANVAS_GROUP (citem),
							    GNOME_DB_TYPE_CANVAS_CURSOR,
							    "x", ((GdkEventButton *) event)->x - 5.,
							    "y", ((GdkEventButton *) event)->y - 5.,
							    "fill_color", "white",
							    NULL);
				g_object_weak_ref (G_OBJECT (ci), (GWeakNotify) end_of_drag_cb, citem);

				/* this weak ref is in case citem is destroyed before ci */
				g_object_weak_ref (G_OBJECT (citem), (GWeakNotify) end_of_drag_cb_d, ci);
				
				GNOME_DB_CANVAS_ITEM (ci)->priv->xstart = ((GdkEventButton *) event)->x;
				GNOME_DB_CANVAS_ITEM (ci)->priv->ystart = ((GdkEventButton *) event)->y;
				GNOME_DB_CANVAS_ITEM (ci)->priv->moving = TRUE;
				gnome_canvas_item_raise_to_top (GNOME_CANVAS_ITEM (ci));
				gnome_canvas_item_reparent (ci, gnome_canvas_root (GNOME_CANVAS_ITEM (citem)->canvas));
				gnome_canvas_item_grab (ci, GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
							NULL, GDK_CURRENT_TIME);
				done = TRUE;
			}
			break;
		default:
			break;
		}
		break;
	case GDK_BUTTON_RELEASE:
		if (citem->priv->allow_move) {
			citem->priv->moving = FALSE;
			if (citem->priv->graph_item) {
				gdouble x, y;
				gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (citem), &x , &y, NULL, NULL);
				gda_graph_item_set_position (citem->priv->graph_item, x, y);
			}
#ifdef debug_signal
			g_print (">> 'MOVED' from %s::item_event()\n", __FILE__);
#endif
			g_signal_emit (G_OBJECT (citem), gnome_db_canvas_item_signals[MOVED], 0);
#ifdef debug_signal
			g_print ("<< 'MOVED' from %s::item_event()\n", __FILE__);
#endif
		}
		break;
	case GDK_MOTION_NOTIFY:
		if (citem->priv->moving) {
			GdkEventMotion *evm = (GdkEventMotion *) (event);

			g_assert (GNOME_DB_IS_CANVAS_ITEM (citem));
			gnome_canvas_item_move (GNOME_CANVAS_ITEM (citem), 
						(double) evm->x - citem->priv->xstart, 
						(double) evm->y - citem->priv->ystart);
			citem->priv->xstart = ((GdkEventMotion *) event)->x;
			citem->priv->ystart = ((GdkEventMotion *) event)->y;
			g_signal_emit (G_OBJECT (citem), gnome_db_canvas_item_signals[MOVING], 0);
		}
		else {
			if (! g_object_get_data (G_OBJECT (citem), "tip")) {
				/* set tooltip */
				id = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (citem), "addtipid"));
				if (id != 0) {
					g_source_remove (id);
					g_object_set_data (G_OBJECT (citem), "addtipid", NULL);
				}
				id = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (citem), "displaytipid"));
				if (id != 0) {
					g_source_remove (id);
					g_object_set_data (G_OBJECT (citem), "displaytipid", NULL);
				}
				id = g_timeout_add (200, (GSourceFunc) add_tip_timeout, citem);
				g_object_set_data (G_OBJECT (citem), "addtipid", GUINT_TO_POINTER (id));
				pos = ((GdkEventMotion *)(event))->x;
				g_object_set_data (G_OBJECT (citem), "mousex", GINT_TO_POINTER ((gint) pos));
				pos = ((GdkEventMotion *)(event))->y;
				g_object_set_data (G_OBJECT (citem), "mousey", GINT_TO_POINTER ((gint) pos));
			}
		}
		break;

	default:
		break;
	}

	return done;
}

static void 
end_of_drag_cb (GnomeDbCanvasItem *citem, GObject *cursor)
{
	g_object_set_data (G_OBJECT (gnome_canvas_root (GNOME_CANVAS_ITEM (citem)->canvas)), 
			   "dragged_from", citem);
	g_object_weak_unref (G_OBJECT (citem), (GWeakNotify) end_of_drag_cb_d, cursor);
}

static void
end_of_drag_cb_d (GnomeCanvasItem *ci, GnomeDbCanvasItem *citem)
{
	g_object_weak_unref (G_OBJECT (ci), (GWeakNotify) end_of_drag_cb, citem);
}

static gboolean add_tip_timeout (GnomeDbCanvasItem *citem)
{
        guint id;

        id = gtk_timeout_add (100, (GSourceFunc) display_tip_timeout, citem);
        g_object_set_data (G_OBJECT (citem), "addtipid", NULL);
        g_object_set_data (G_OBJECT (citem), "displaytipid", GUINT_TO_POINTER (id));
        return FALSE;
}


static void tip_destroy (GnomeDbCanvasItem *citem, GObject *tip);
static gboolean display_tip_timeout (GnomeDbCanvasItem *citem)
{
        GnomeCanvasItem *tip;
        gdouble x, y;

	if (citem->priv->tooltip_text) {
		/* display the tip */
		g_object_set_data (G_OBJECT (citem), "displaytipid", NULL);
		x = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (citem), "mousex"));
		y = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (citem), "mousey"));
		
		tip = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS_ITEM (citem)->canvas),
					     GNOME_DB_TYPE_CANVAS_TIP,
					     "x", x+7.,
					     "y", y+3.,
					     "tip_text", citem->priv->tooltip_text,
					     NULL);

		g_object_weak_ref (G_OBJECT (tip), (GWeakNotify) tip_destroy, citem);

		g_object_set_data (G_OBJECT (citem), "tip", tip);
	}
	return FALSE;
}

static void tip_destroy (GnomeDbCanvasItem *citem, GObject *tip)
{
        g_object_set_data (G_OBJECT (citem), "tip", NULL);
}

