/*
 * Galago Service API
 *
 * Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_SERVICE_H_
#define _GALAGO_SERVICE_H_

typedef struct _GalagoService        GalagoService;
typedef struct _GalagoServiceClass   GalagoServiceClass;
typedef struct _GalagoServicePrivate GalagoServicePrivate;

typedef enum
{
	/* Preserve spaces during normalization */
	GALAGO_PRESERVE_SPACES = 1 << 0,

	/* Preserve case during normalization */
	GALAGO_PRESERVE_CASE   = 1 << 1,

	/* Strip a slash and everything after it during normalization. */
	GALAGO_STRIP_SLASH     = 1 << 2
} GalagoServiceFlags;

#include <libgalago/galago-account.h>
#include <libgalago/galago-object.h>

struct _GalagoService
{
	GalagoObject parent_object;

	GalagoServicePrivate *priv;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

/**
 * A service's class.
 */
struct _GalagoServiceClass
{
	GalagoObjectClass parent_class;

	/* Signals */
	void (*account_added)(GalagoService *service, GalagoAccount *account);
	void (*account_removed)(GalagoService *service, GalagoAccount *account);

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

#define GALAGO_TYPE_SERVICE (galago_service_get_type())
#define GALAGO_SERVICE(obj) \
		(G_TYPE_CHECK_INSTANCE_CAST((obj), GALAGO_TYPE_SERVICE, \
									GalagoService))
#define GALAGO_SERVICE_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_CAST((klass), GALAGO_TYPE_SERVICE, \
								 GalagoServiceClass))
#define GALAGO_IS_SERVICE(obj) \
		(G_TYPE_CHECK_INSTANCE_TYPE((obj), GALAGO_TYPE_SERVICE))
#define GALAGO_IS_SERVICE_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_TYPE((klass), GALAGO_TYPE_SERVICE))
#define GALAGO_SERVICE_GET_CLASS(obj) \
		(G_TYPE_INSTANCE_GET_CLASS((obj), GALAGO_TYPE_SERVICE, \
								   GalagoServiceClass))

#define GALAGO_DBUS_SERVICE_INTERFACE "org.freedesktop.Galago.Service"

G_BEGIN_DECLS

/**************************************************************************/
/** Common Service Definitions                                            */
/**************************************************************************/

#define GALAGO_SERVICE_ID_AIM          "aim"
#define GALAGO_SERVICE_ID_GADUGADU     "gadugadu"
#define GALAGO_SERVICE_ID_GROUPWISE    "groupwise"
#define GALAGO_SERVICE_ID_ICQ          "icq"
#define GALAGO_SERVICE_ID_IRC          "irc"
#define GALAGO_SERVICE_ID_JABBER       "jabber"
#define GALAGO_SERVICE_ID_MSN          "msn"
#define GALAGO_SERVICE_ID_NAPSTER      "napster"
#define GALAGO_SERVICE_ID_SILC         "silc"
#define GALAGO_SERVICE_ID_TREPIA       "trepia"
#define GALAGO_SERVICE_ID_YAHOO        "yahoo"
#define GALAGO_SERVICE_ID_ZEPHYR       "zephyr"


/**************************************************************************/
/** Service API                                                           */
/**************************************************************************/

GType galago_service_get_type(void);

const char *galago_service_get_id(const GalagoService *service);
const char *galago_service_get_name(const GalagoService *service);
GalagoServiceFlags galago_service_get_flags(const GalagoService *service);

GalagoAccount *galago_service_create_account(GalagoService *service,
											 GalagoPerson *person,
											 const char *username);
GalagoAccount *galago_service_get_account(const GalagoService *service,
										  const char *username,
										  gboolean query);
GList *galago_service_get_accounts(const GalagoService *service,
								   gboolean query);

char *galago_service_normalize(const GalagoService *service,
							   const char *username);

G_END_DECLS

#endif /* _GALAGO_SERVICE_H_ */
