/** -*- C++ -*-
    @file cache/entity/entity.h
    @author Peter Rockai <me@mornfall.net>
    @todo Parametrize the (end user) entities over some
    representation, which gets wrapped by Iterable template (which is
    then specialised)
*/

#include <wibble/amorph.h>
#include <ept/forward.h>

#ifndef EPT_CACHE_ENTITY_H
#define EPT_CACHE_ENTITY_H

namespace ept {
namespace t {
namespace cache {

struct Entity;

struct EntityInterface {
public:
    virtual ~EntityInterface() {}
    virtual bool valid() const = 0;
    virtual Entity stable() const = 0;
};

struct NamedEntity {
    virtual std::string name() { return std::string(); }
    virtual ~NamedEntity() {}
};

template< typename W >
struct EntityMorph : wibble::Morph< EntityMorph< W >, W, EntityInterface >
// wibble::mixin::Comparable< EntityMorph< W > >
{
    typedef typename W::EntityImplementation Wrapped;
    EntityMorph( const W &w ) : wibble::Morph< EntityMorph, W, EntityInterface >( w ) {}
    virtual Entity stable() const;
    virtual bool valid() const { return this->wrapped().valid(); }
};

template< typename Self >
struct EntityMixin : wibble::mixin::Comparable< Self > {

    typedef Self EntityImplementation;

    Entity stable() const;
    const Self &self() const { return *static_cast< const Self * >( this ); }

    bool operator<=( const Self &o ) const {
        if ( self().valid() < o.valid() )
            return true;
        else if ( self().valid() > o.valid() )
            return false;
        if ( !self().valid() ) return true;

        if ( &self().aggregator() < &o.aggregator() )
            return true;
        if ( self().id() <= o.id() )
            return true;
        return false;
    }

    void checkValid() const throw (wibble::exception::Consistency) {
        if (self().valid())
            return;
        throw wibble::exception::Consistency( "Tried to use invalid Entity" );
    }

};

template< typename Self >
struct InstallableMixin {

    struct State {
        enum Query {
            Install = 1 << 0,
            Upgrade = 1 << 1,
            Keep = 1 << 2,
            Remove = 1 << 3,
            Installed = 1 << 4,
            Upgradable = 1 << 5,
            NowBroken = 1 << 6,
            WillBreak = 1 << 7,
            ReInstall = 1 << 8,
            Purge = 1 << 9
        };

        typedef unsigned state;

        operator unsigned() { return m_state; };

        State &operator=( unsigned i ) {
            m_state = i;
            return *this;
        }

        State &operator|=( const State &s ) {
            m_state |= s.m_state;
            return *this;
        }

        State( unsigned a ) {
            m_state = a;
        }

        State() : m_state( 0 ) {}

    protected:
        unsigned m_state;
    };


    const Self &self() const { return *static_cast< const Self * >( this ); }

    bool markedInstall() const { return (self().state() & State::Install)
            || markedReInstall(); }
    bool markedUpgrade() const { return markedInstall() && isUpgradable(); }
    bool markedNewInstall() const { return markedInstall() && !isUpgradable()
            && !markedReInstall(); }
    bool markedReInstall() const { return self().state() & State::ReInstall; }
    bool markedKeep() const { return self().state() & State::Keep; }
    bool markedRemove() const { return self().state() & State::Remove; }
    bool markedPurge() const { return self().state() & State::Purge; }

    bool isInstalled() const { return self().state() & State::Installed; }
    bool isUpgradable() const { return self().state() & State::Upgradable; }
    bool isBroken() const { return self().state() & State::NowBroken; }
    bool willBreak() const { return self().state() & State::WillBreak; }

    bool canInstall() const {
        return ((! isInstalled()) && ! markedInstall()) || markedRemove()
            || isBroken(); }
    bool canRemove() const { return isInstalled() && not markedRemove(); }
    bool canKeep() const {
        return markedUpgrade() || markedRemove() || markedInstall(); }
    bool canUpgrade() const { return isUpgradable () && ! markedUpgrade (); }
    bool canReInstall() const { return isInstalled() && !isUpgradable()
            && !markedReInstall(); }
    bool canRevertInstall() const { return markedNewInstall(); }

};

struct Entity : wibble::Amorph< Entity, EntityInterface, 7 >, EntityMixin< Entity >
{
public:
    typedef wibble::Amorph< Entity, EntityInterface, 7 > Super;
    template< typename C >
    Entity( const C &i ) : Super( EntityMorph< C >( i ) ) {}
    Entity() {}

    Entity stable() const { return implementation()->stable(); }
    bool valid() const { return implementation()->valid(); }
    bool operator<=( const Entity &i ) const { return leq( i ); }

    ~Entity() {}
};

template< typename W >
Entity EntityMorph< W >::stable() const {
    return this->wrapped().stable();
}

template< typename S >
Entity EntityMixin< S >::stable() const {
    return self();
}

}
}
}
#endif
