/** -*- C++ -*-
    @file ept/actor.h
    @author Peter Rockai <me@mornfall.net>
*/

#include <wibble/amorph.h>

#ifndef EPT_ACTOR_H
#define EPT_ACTOR_H

namespace ept {
namespace actor {

template< typename T >
struct ActorInterface {
    virtual void act( T& = 0 ) = 0;
    virtual ~ActorInterface() {}
};


template< typename T, typename Self >
struct ActorMixin
{
    typedef Self ActorImplementation;
    typedef T argument_type;
    typedef void result_type;
    Self &self() { return *static_cast< const Self * >( this ); }

    bool operator<=( const Self &o ) const { // by-address ordering
        return this <= &o;
    }

    void operator()( T &a = *(static_cast< T * >( 0 )) ) {
        std::cerr << "ActorMixin: parameter = " << &a << std::endl;
        if ( &a != 0 )
            return self().act( a );
    }
};

template< typename T >
struct MemberActor : ActorMixin< T, MemberActor< T > >
{
    typedef void (T::*Method)();
    MemberActor( Method m )
        : m_method( m )
    {
    }

    void act( T &t ) {
        (t.*m_method)();
    }

protected:
    Method m_method;
};

template< typename T, typename Fun >
struct FunctorActor : ActorMixin< T, FunctorActor< T, Fun > >
{
    FunctorActor( Fun f )
        : m_fun( f )
    {
    }

    void act( T &t ) {
        m_method( t );
    }

protected:
    Fun m_fun;
};

template< typename T, typename Inner >
struct OnMember : ActorMixin< T, OnMember< T, Inner > >
{
    typedef typename Inner::argument_type inner_type;
    typedef inner_type (T::*Extract)();
    OnMember( Extract m, Inner i ) : m_extract( m ), m_inner( i ) {}
    void act( T &t ) {
        return m_inner( (t.*m_extract)() );
    }
protected:
    Extract m_extract;
    Inner m_inner;
};

template< typename Adapt, typename Actor >
struct Adaptor : Actor, ActorMixin< Adapt, Adaptor< Adapt, Actor > > {
    typedef Adaptor ActorImplementation;
    typedef Adapt argument_type;
    Adaptor( const Actor &a ) : Actor( a ) {}
    // Actor &actor() { return m_act; }
    Actor &actor() { return *static_cast< Actor * >( this ); }
    const Actor &actor() const { return *static_cast< const Actor * >( this ); }
    void act( Adapt &a ) {
        a.template ifType< typename Actor::argument_type >( actor() );
    }
    bool operator<=( const Adaptor &o ) const { return actor() <= o.actor(); }
};

template< typename T >
MemberActor< T > actor( typename MemberActor< T >::Method m )
{
    return MemberActor< T >( m );
}

template< typename T, typename I >
OnMember< T, I > onMember( typename OnMember< T, I >::Extract m, I inner )
{
    return OnMember< T, I >( m, inner );
}

template< typename T, typename W >
struct ActorMorph: wibble::Morph< ActorMorph< T, W >, W, ActorInterface< T > >
{
    ActorMorph( const W &w ) :
        wibble::Morph< ActorMorph, W, ActorInterface< T > >( w ) {}
    virtual void act( T &t ) { return this->wrapped().act( t ); }
};

template< typename T >
struct Actor : wibble::Amorph< Actor< T >, ActorInterface< T > >, ActorMixin< T, Actor< T > >
{
    typedef wibble::Amorph< Actor< T >, ActorInterface< T > > Super;
    template< typename C >
    Actor( const C &i,
                       typename wibble::IsType< int, typename
                       C::ActorImplementation >::T fake = 0 )
        : Super( ActorMorph< T, C >( i ) ) { /*(void)fake;*/ }
    Actor() {}
    void act( T &t ) { return this->implementation()->act( t ); }
};

template< typename Adapt, typename Actor >
Adaptor< Adapt, Actor > adapt( Actor op ) {
    return Adaptor< Adapt, Actor >( op );
}

}
}

#endif
