/***************************************************************************
 $RCSfile: geldkarte.cpp,v $
 -------------------
 cvs         : $Id: geldkarte.cpp,v 1.7 2003/05/08 11:01:23 aquamaniac Exp $
 begin       : Sat Dec 21 2002
 copyright   : (C) 2002 by Martin Preuss
 email       : martin@libchipcard.de

 ****************************************************************************
 * This program is free software; you can redistribute it and/or modify     *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation; either version 2 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA *
 ****************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

/* Internationalization */
#ifdef HAVE_GETTEXT_ENVIRONMENT
# include <libintl.h>
# include <locale.h>
# define I18N(m) gettext(m)
#else
# define I18N(m) m
#endif
#define I18NT(m) m


#include <stdio.h>
#include <errno.h>
#include <chipcard.h>
#include <chameleon/logger.h>


#define k_PRG "geldkarte"
#define k_PRG_VERSION_INFO \
    "GeldKarte v0.3  (part of libchipcard v"k_CHIPCARD_VERSION_STRING")\n"\
    "(c) 2003 Martin Preuss<martin@libchipcard.de>\n" \
    "This program is free software licensed under GPL.\n"\
    "See COPYING for details.\n"

void usage(string name) {
  fprintf(stderr,"%s%s%s%s",
	  I18N("GeldKarte - A tool to read information from a GeldKarte\n"
	       "(c) 2003 Martin Preuss<martin@libchipcard.de>\n"
	       "This library is free software; you can redistribute it and/or\n"
	       "modify it under the terms of the GNU Lesser General Public\n"
	       "License as published by the Free Software Foundation; either\n"
	       "version 2.1 of the License, or (at your option) any later version.\n"
	       "\n"
	       "Usage:\n"
	       "%s COMMAND [-C CONFIGFILE]\n"
	       " COMMAND\n"
	       "  loaded  : show amount of money stored on the card \n"
	       "  maxload : show amount of money which can be stored on the card\n"
	       "  maxxfer : show amount of money which can be transferred in\n"
	       "            one setting\n"
	       " OPTIONS\n"
	       " -C CONFIGFILE    - configuration file of libchipcard to be used\n"
	       " --logfile FILE   - use given FILE as log file\n"
	       " --logtype TYPE   - use given TYPE as log type\n"
	       "                    These are the valid types:\n"
	       "                     stderr (log to standard error channel)\n"
	       "                     file   (log to the file given by --logfile)\n"),
#ifdef HAVE_SYSLOG_H
	  I18N("                     syslog (log via syslog)\n"),
#else
	  "",
#endif
	  I18N("                    Default is stderr\n"
	       " --loglevel LEVEL - set the loglevel\n"
	       "                    Valid levels are:\n"
	       "                     emergency, alert, critical, error,\n"
	       "                     warning, notice, info and debug\n"
	       "                    Default is \"warning\".\n")
	  ,
	  name.c_str());
}


struct s_args {
  string configFile;    // -C
  string logFile;         // --logfile
  LOGGER_LOGTYPE logType; // --logtype
  LOGGER_LEVEL logLevel;  // --loglevel
  list<string> params;
};


int checkArgs(s_args &args, int argc, char **argv) {
  int i;
  string tmp;

  i=1;
  args.configFile=CHIPCARDC_CFGFILE;
  args.logFile="geldkarte.log";
  args.logType=LoggerTypeConsole;
  args.logLevel=LoggerLevelWarning;

  while (i<argc){
    tmp=argv[i];
    if (tmp=="-C") {
      i++;
      if (i>=argc)
	return 1;
      args.configFile=argv[i];
    }
    else if (tmp=="--logfile") {
      i++;
      if (i>=argc)
	return 1;
      args.logFile=argv[i];
    }

    else if (tmp=="--logtype") {
      i++;
      if (i>=argc)
	return 1;
      if (argv[i]=="stderr")
	args.logType=LoggerTypeConsole;
      else if (argv[i]=="file")
	args.logType=LoggerTypeFile;
#ifdef HAVE_SYSLOG_H
      else if (argv[i]=="syslog")
	args.logType=LoggerTypeSyslog;
#endif
      else {
	fprintf(stderr,I18N("Unknown log type \"%s\"\n"),
		argv[i]);
	return 1;
      }
    }

    else if (tmp=="--loglevel") {
      i++;
      if (i>=argc)
	return 1;
      if (strcmp(argv[i], "emergency")==0)
	args.logLevel=LoggerLevelEmergency;
      else if (strcmp(argv[i], "alert")==0)
	args.logLevel=LoggerLevelAlert;
      else if (strcmp(argv[i], "critical")==0)
	args.logLevel=LoggerLevelCritical;
      else if (strcmp(argv[i], "error")==0)
	args.logLevel=LoggerLevelError;
      else if (strcmp(argv[i], "warning")==0)
	args.logLevel=LoggerLevelWarning;
      else if (strcmp(argv[i], "notice")==0)
	args.logLevel=LoggerLevelNotice;
      else if (strcmp(argv[i], "info")==0)
	args.logLevel=LoggerLevelInfo;
      else if (strcmp(argv[i], "debug")==0)
	args.logLevel=LoggerLevelDebug;
      else {
	fprintf(stderr,
		I18N("Unknown log level \"%s\"\n"),
		argv[i]);
	return 1;
      }
    }

    else if (tmp=="-h" || tmp=="--help") {
      usage(argv[0]);
      return -1;
    }
    else if (tmp=="-V" || tmp=="--version") {
      fprintf(stdout,k_PRG_VERSION_INFO);
      return -1;
    }
    else
      // otherwise add param
      args.params.push_back(tmp);
    i++;
  } // while
  // that's it
  return 0;
}


int openCard(CTPointer<CTCard> &card, bool quiet=false) {
  CTPointer<CTCardTrader> trader;
  CTError err;
  CTCard *cp;

  trader=new CTCardTrader(false,
			  0,
			  0,
			  CHIPCARD_STATUS_INSERTED,
			  CHIPCARD_STATUS_INSERTED |
			  CHIPCARD_STATUS_LOCKED_BY_OTHER,
			  CHIPCARD_STATUS_INSERTED);
  err=trader.ref().start();
  if (!err.isOk()) {
    fprintf(stderr, I18N("Could not initialize trader"));
    return 2;
  }

  if (!quiet)
    fprintf(stderr, I18N("Please insert your card into any reader\n"));
  // get the card
  err=trader.ref().getNext(cp, 30);
  if (!err.isOk()) {
    if (err.code()==k_CTERROR_API &&
	(err.subcode1()==CHIPCARD_ERROR_NO_TRANSPORT ||
	 err.subcode1()==CHIPCARD_ERROR_NO_REQUEST)) {
      fprintf(stderr,
	      I18N("Service unreachable, maybe \"chipcardd\" is not running?\n")
	     );
      return 2;
    }

    fprintf(stderr,I18N("No card inserted within some seconds, aborting.\n"));
    return 2;
  }

  err=trader.ref().stop();
  if (!err.isOk()) {
    fprintf(stderr, I18N("Could not stop trader"));
    return 2;
  }

  card=cp;
  if (!(card.ref().isProcessorCard())) {
    fprintf(stderr, I18N("Not a GeldKarte"));
    return 3;
  }

  if (!quiet)
    fprintf(stderr,I18N("Card is inserted, working.\n"));

  return 0;
}


int loaded(s_args &args) {
  CTPointer<CTGeldKarte> card;
  CTPointer<CTCard> basecard;
  CTError err;
  float v;
  int rv;

  try {
    rv=openCard(basecard);
    if (rv!=0)
      return rv;

    card=new CTGeldKarte(basecard.ref());
    basecard=0;
    err=card.ref().openCard();
    if (!err.isOk()) {
      fprintf(stderr, I18N("Geldkarte not recognized (%s)\n"),
	      err.errorString().c_str());
      card.ref().closeCard();
      return 3;
    }
    v=card.ref().readLoadedValue();
    fprintf(stderr,
	    I18N("Card is loaded with %6.2f %s\n"),
	    v,
	    card.ref().cardData().currency().c_str());
  }
  catch(CTError xerr) {
    fprintf(stderr,"%s\n",xerr.errorString().c_str());
    card.ref().closeCard();
    return 3;
  }

  err=card.ref().closeCard();
  if (!err.isOk(0x62)) {
    fprintf(stderr,"%s\n",err.errorString().c_str());
    return 2;
  }
  return 0;
}


int maxLoad(s_args &args) {
  CTPointer<CTGeldKarte> card;
  CTPointer<CTCard> basecard;
  CTError err;
  float v;
  int rv;

  try {
    rv=openCard(basecard);
    if (rv!=0)
      return rv;
    card=new CTGeldKarte(basecard.ref());
    basecard=0;
    err=card.ref().openCard();
    if (!err.isOk()) {
      fprintf(stderr, I18N("Geldkarte not recognized (%s)\n"),
	      err.errorString().c_str());
      card.ref().closeCard();
      return 3;
    }
    v=card.ref().readMaxLoadedValue();
    fprintf(stderr,
	    "Card can be loaded with %6.2f %s\n",
	    v,
	    card.ref().cardData().currency().c_str());
  }
  catch(CTError xerr) {
    fprintf(stderr,"%s\n",xerr.errorString().c_str());
    card.ref().closeCard();
    return 3;
  }

  err=card.ref().closeCard();
  if (!err.isOk(0x62)) {
    fprintf(stderr,"%s\n",err.errorString().c_str());
    return 2;
  }
  return 0;
}


int maxXfer(s_args &args) {
  CTPointer<CTGeldKarte> card;
  CTPointer<CTCard> basecard;
  CTError err;
  float v;
  int rv;

  try {
    rv=openCard(basecard);
    if (rv!=0)
      return rv;
    card=new CTGeldKarte(basecard.ref());
    basecard=0;
    err=card.ref().openCard();
    if (!err.isOk()) {
      fprintf(stderr, I18N("Geldkarte not recognized (%s)\n"),
	      err.errorString().c_str());
      card.ref().closeCard();
      return 3;
    }
    v=card.ref().readMaxLoadedValue();
    fprintf(stderr,
	    I18N("Card can transfer up to %6.2f %s\n"),
	    v,
	    card.ref().cardData().currency().c_str());
  }
  catch(CTError xerr) {
    fprintf(stderr,"%s\n",xerr.errorString().c_str());
    card.ref().closeCard();
    return 3;
  }

  err=card.ref().closeCard();
  if (!err.isOk(0x62)) {
    fprintf(stderr,"%s\n",err.errorString().c_str());
    return 2;
  }
  return 0;
}




int main(int argc, char **argv) {
  s_args args;
  int rv;
  string cmd;

#ifdef HAVE_GETTEXT_ENVIRONMENT
  setlocale(LC_ALL,"");
  if (bindtextdomain("geldkarte",  I18N_PATH)==0) {
    fprintf(stderr," Error bindtextdomain()\n");
  }
  if (textdomain("geldkarte")==0) {
    fprintf(stderr," Error textdomain()\n");
  }
#endif

  rv=checkArgs(args,argc,argv);
  if (rv==-1)
    return 0;
  else if (rv)
    return rv;
  if (args.params.empty()) {
    usage(argv[0]);
    return 1;
  }

  if (Logger_Open("geldkarte",
		  args.logFile.c_str(),
		  args.logType,
		  LoggerFacilityUser)) {
    fprintf(stderr,I18N("Could not start logging, aborting.\n"));
    return 2;
  }

  Logger_SetLevel(args.logLevel);

  rv=ChipCard_Init(args.configFile.c_str(),0);
  if (rv!=CHIPCARD_SUCCESS) {
    fprintf(stderr,
	    I18N("Error initializing libchipcard (%d), aborting.\n"),rv);
    return 2;
  }

  cmd=args.params.front();
  if (cmd=="loaded")
    rv=loaded(args);
  else if (cmd=="maxload")
    rv=maxLoad(args);
  else if (cmd=="maxxfer")
    rv=maxXfer(args);
  else {
    fprintf(stderr,I18N("Unknown command.\n"));
    usage(argv[0]);
    rv=1;
  }
  ChipCard_Fini();
  return rv;
}



