/***************************************************************************
 $RCSfile: readerclient.c,v $
 -------------------
 cvs         : $Id: readerclient.c,v 1.12 2003/05/08 12:26:36 aquamaniac Exp $
 begin       : Sat Jan 11 2003
 copyright   : (C) 2003 by Martin Preuss
 email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_CHIPCARD_DLL
#  define CHIPCARD_API __declspec (dllexport)
# else /* Not BUILDING_CHIPCARD_DLL */
#  define CHIPCARD_API __declspec (dllimport)
# endif /* Not BUILDING_CHIPCARD_DLL */
#else
# define CHIPCARD_API
#endif

#include "ctclient.h"
#include "readerclient.h"
#include "readerservice.h"
#include <chameleon/debug.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include <chameleon/chameleon.h> /* debugging */



READERCLIENT_READERDESCR *ReaderClient_ReaderDescr_new() {
  READERCLIENT_READERDESCR *rd;

  rd=(READERCLIENT_READERDESCR *)malloc(sizeof(READERCLIENT_READERDESCR));
  assert(rd);
  memset(rd,0,sizeof(READERCLIENT_READERDESCR));
  return rd;
}


void ReaderClient_ReaderDescr_free(READERCLIENT_READERDESCR *rd) {
  if (rd)
    free(rd);
}



ERRORCODE ReaderClient_RequestPing(CTCLIENTDATA *cd,
				   int *requestid,
				   int serviceid){
  CTSERVICEREQUEST *req;
  ERRORCODE err;
  int localreqid;

  DBG_ENTER;
  /* assign a request id */
  localreqid=++(cd->nextMessageId);

  /* create the request stub */
  req=CTService_Request_Create(serviceid,
			       READERSERVICE_MSGCODE_RQ_PING,
			       READERSERVICE_MSGCODE_RQ_PING_VERSION,
			       localreqid,
			       0,
			       128);
  if (!req) {
    DBG_ERROR("Could not create the request");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  }

  /* finish message */
  err=IPCMessage_BuildMessage(req->message);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }

  /* send message */
  err=CTClient_SendRequest(cd,
			   req,
			   serviceid);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }
  /* done */
  *requestid=localreqid;
  return 0;
}


ERRORCODE ReaderClient_CheckPing(CTCLIENTDATA *cd,
				 int requestid){
  CTSERVICEREQUEST *req;
  IPCMESSAGE *msg;
  ERRORCODE err;

  DBG_ENTER;
  /* search request */
  req=CTClient_FindRequest(cd, requestid);
  if (!req)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  /* check for response */
  msg=CTService_Request_NextResponse(req);
  if (!msg)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_MESSAGE);

  /* check for error message */
  err=CTClient_CheckErrorMessage(cd,
				 msg);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* check for message code and version */
  err=CTService_CheckMsgCodeAndVersion(msg,
					READERSERVICE_MSGCODE_RP_PING,
					READERSERVICE_MSGCODE_RP_PING_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  DBG_DEBUG("Dequeuing request");
  CTClient_DequeueRequest(cd, req);
  CTService_Request_free(req);
  IPCMessage_free(msg);
  DBG_INFO("Ping request finished");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderClient_RequestAllocReader(CTCLIENTDATA *cd,
					  int *requestid,
					  int serviceid,
					  int rid){
  CTSERVICEREQUEST *req;
  ERRORCODE err;
  int localreqid;

  DBG_ENTER;
  assert(cd);

  /* assign a request id */
  localreqid=++(cd->nextMessageId);

  /* create the request stub */
  req=CTService_Request_Create(serviceid,
			       READERSERVICE_MSGCODE_RQ_ALLOC,
			       READERSERVICE_MSGCODE_RQ_ALLOC_VERSION,
			       localreqid,
			       0,
			       256);
  if (!req) {
    DBG_ERROR("Could not create the request");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  }

  /* add reader id  */
  err=IPCMessage_AddIntParameter(req->message, rid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* finish message */
  err=IPCMessage_BuildMessage(req->message);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }

  /* send message */
  err=CTClient_SendRequest(cd,
			   req,
			   serviceid);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }
  /* done */
  *requestid=localreqid;
  return 0;
}


ERRORCODE ReaderClient_CheckAllocReader(CTCLIENTDATA *cd,
					int requestid,
					int *tid,
					int *serviceId,
					READERCLIENT_READERDESCR **rd) {
  CTSERVICEREQUEST *req;
  IPCMESSAGE *msg;
  ERRORCODE err;
  int i;
  READERCLIENT_READERDESCR *lrd;
  char *p;
  int s;

  DBG_ENTER;
  /* search request */
  req=CTClient_FindRequest(cd, requestid);
  if (!req)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  /* check for response */
  msg=CTService_Request_NextResponse(req);
  if (!msg)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_MESSAGE);

  /* check for error message */
  err=CTClient_CheckErrorMessage(cd,
				 msg);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  err=CTService_CheckMsgCodeAndVersion(msg,
				       READERSERVICE_MSGCODE_RP_ALLOC,
				       READERSERVICE_MSGCODE_RP_ALLOC_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get terminal id */
  err=IPCMessage_IntParameter(msg,4,&i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }
  *tid=i;
  *serviceId=req->serviceId;;

  /* get terminal description */
  lrd=ReaderClient_ReaderDescr_new();
  /* name */
  err=IPCMessage_NextParameter(msg,&p, &s);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    ReaderClient_ReaderDescr_free(lrd);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }
  if (s>sizeof(lrd->name)) {
    DBG_ERROR("Buffer too small");
    ReaderClient_ReaderDescr_free(lrd);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_BUFFER);
  }
  strcpy(lrd->name,p);

  /* get flags */
  err=IPCMessage_NextIntParameter(msg,&i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    ReaderClient_ReaderDescr_free(lrd);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }
  lrd->flags=i;

  /* type */
  err=IPCMessage_NextParameter(msg,&p, &s);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    ReaderClient_ReaderDescr_free(lrd);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }
  if (s>sizeof(lrd->type)) {
    DBG_ERROR("Buffer too small");
    ReaderClient_ReaderDescr_free(lrd);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_BUFFER);
  }
  strcpy(lrd->type,p);

  *rd=lrd;

  DBG_INFO("Allocated terminal with id %d.", *tid);

  DBG_DEBUG("Dequeuing request");
  CTClient_DequeueRequest(cd, req);
  CTService_Request_free(req);
  IPCMessage_free(msg);
  DBG_INFO("AllocReader request finished");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderClient_RequestReleaseReader(CTCLIENTDATA *cd,
					    int *requestid,
					    int serviceid,
					    int tid) {
  CTSERVICEREQUEST *req;
  ERRORCODE err;
  int localreqid;

  DBG_ENTER;
  assert(cd);

  /* assign a request id */
  localreqid=++(cd->nextMessageId);

  /* create the request stub */
  req=CTService_Request_Create(serviceid,
			       READERSERVICE_MSGCODE_RQ_RELEASE,
			       READERSERVICE_MSGCODE_RQ_RELEASE_VERSION,
			       localreqid,
			       0,
			       256);
  if (!req) {
    DBG_ERROR("Could not create the request");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  }

  /* add reader id  */
  err=IPCMessage_AddIntParameter(req->message, tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* finish message */
  err=IPCMessage_BuildMessage(req->message);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }

  /* send message */
  err=CTClient_SendRequest(cd,
			   req,
			   serviceid);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }
  /* done */
  *requestid=localreqid;
  return 0;
}


ERRORCODE ReaderClient_CheckReleaseReader(CTCLIENTDATA *cd,
					  int requestid) {
  CTSERVICEREQUEST *req;
  IPCMESSAGE *msg;
  ERRORCODE err;

  DBG_ENTER;
  /* search request */
  req=CTClient_FindRequest(cd, requestid);
  if (!req)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  /* check for response */
  msg=CTService_Request_NextResponse(req);
  if (!msg)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_MESSAGE);

  /* check for error message */
  err=CTClient_CheckErrorMessage(cd,
				 msg);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }


  err=CTService_CheckMsgCodeAndVersion(msg,
				       READERSERVICE_MSGCODE_RP_RELEASE,
				       READERSERVICE_MSGCODE_RP_RELEASE_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  DBG_DEBUG("Released terminal");

  DBG_DEBUG("Dequeuing request");
  CTClient_DequeueRequest(cd, req);
  CTService_Request_free(req);
  IPCMessage_free(msg);
  DBG_INFO("ReleaseReader request finished");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderClient_RequestConnectReader(CTCLIENTDATA *cd,
					    int *requestid,
					    int serviceid,
					    int tid,
					    int cardId,
					    int waitForIt) {
  CTSERVICEREQUEST *req;
  ERRORCODE err;
  int localreqid;

  DBG_ENTER;
  assert(cd);

  /* assign a request id */
  localreqid=++(cd->nextMessageId);

  /* create the request stub */
  req=CTService_Request_Create(serviceid,
			       READERSERVICE_MSGCODE_RQ_CONNECT,
			       READERSERVICE_MSGCODE_RQ_CONNECT_VERSION,
			       localreqid,
			       0,
			       256);
  if (!req) {
    DBG_ERROR("Could not create the request");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  }

  /* add reader id  */
  err=IPCMessage_AddIntParameter(req->message, tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* add card id  */
  err=IPCMessage_AddIntParameter(req->message, cardId);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* add wait flag  */
  err=IPCMessage_AddIntParameter(req->message, waitForIt);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* finish message */
  err=IPCMessage_BuildMessage(req->message);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }

  /* send message */
  err=CTClient_SendRequest(cd,
			   req,
			   serviceid);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }
  /* done */
  *requestid=localreqid;
  return 0;
}


ERRORCODE ReaderClient_CheckConnectReader(CTCLIENTDATA *cd,
					  int requestid,
					  int *result,
					  char *atrbuffer,
					  int *atrbufferlength) {
  CTSERVICEREQUEST *req;
  IPCMESSAGE *msg;
  ERRORCODE err;
  int i;
  char *p;
  int s;

  DBG_ENTER;
  /* search request */
  req=CTClient_FindRequest(cd, requestid);
  if (!req)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  /* check for response */
  msg=CTService_Request_NextResponse(req);
  if (!msg)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_MESSAGE);

  /* check for error message */
  err=CTClient_CheckErrorMessage(cd,
				 msg);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  err=CTService_CheckMsgCodeAndVersion(msg,
				       READERSERVICE_MSGCODE_RP_CONNECT,
				       READERSERVICE_MSGCODE_RP_CONNECT_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get result */
  err=IPCMessage_IntParameter(msg,4,&i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }
  *result=i;
  DBG_DEBUG("Connected terminal");

  if (*result==CTSERVICE_SUCCESS) {
    /* get ATR */
    err=IPCMessage_NextParameter(msg,&p, &s);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      CTClient_DequeueRequest(cd, req);
      CTService_Request_free(req);
    IPCMessage_free(msg);
      DBG_LEAVE;
      return err;
    }
    if (s>*atrbufferlength) {
      DBG_ERROR("ATR buffer too small");
      CTClient_DequeueRequest(cd, req);
      CTService_Request_free(req);
      IPCMessage_free(msg);
      return Error_New(0,
		       ERROR_SEVERITY_ERR,
		       Error_FindType(CTSERVICE_ERROR_TYPE),
		       CTSERVICE_ERROR_BAD_BUFFERSIZE);
    }
    *atrbufferlength=s;
    if (s)
      memmove(atrbuffer, p, s);
  }

  DBG_DEBUG("Dequeuing request");
  CTClient_DequeueRequest(cd, req);
  CTService_Request_free(req);
  IPCMessage_free(msg);
  DBG_INFO("ConnectReader request finished");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderClient_RequestDisconnectReader(CTCLIENTDATA *cd,
					       int *requestid,
					       int serviceid,
					       int tid) {
  CTSERVICEREQUEST *req;
  ERRORCODE err;
  int localreqid;

  DBG_ENTER;
  assert(cd);

  /* assign a request id */
  localreqid=++(cd->nextMessageId);

  /* create the request stub */
  req=CTService_Request_Create(serviceid,
			       READERSERVICE_MSGCODE_RQ_DISCONNECT,
			       READERSERVICE_MSGCODE_RQ_DISCONNECT_VERSION,
			       localreqid,
			       0,
			       256);
  if (!req) {
    DBG_ERROR("Could not create the request");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  }

  /* add reader id  */
  err=IPCMessage_AddIntParameter(req->message, tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* finish message */
  err=IPCMessage_BuildMessage(req->message);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }

  /* send message */
  err=CTClient_SendRequest(cd,
			   req,
			   serviceid);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }
  /* done */
  *requestid=localreqid;
  return 0;
}


ERRORCODE ReaderClient_CheckDisconnectReader(CTCLIENTDATA *cd,
					     int requestid,
					     int *result) {
  CTSERVICEREQUEST *req;
  IPCMESSAGE *msg;
  ERRORCODE err;
  int i;

  DBG_ENTER;
  /* search request */
  req=CTClient_FindRequest(cd, requestid);
  if (!req)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  /* check for response */
  msg=CTService_Request_NextResponse(req);
  if (!msg)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_MESSAGE);

  /* check for error message */
  err=CTClient_CheckErrorMessage(cd,
				 msg);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  err=CTService_CheckMsgCodeAndVersion(msg,
				       READERSERVICE_MSGCODE_RP_DISCONNECT,
				       READERSERVICE_MSGCODE_RP_DISCONNECT_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get result */
  err=IPCMessage_IntParameter(msg,4,&i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }
  *result=i;
  DBG_DEBUG("Disconnected terminal");

  DBG_DEBUG("Dequeuing request");
  CTClient_DequeueRequest(cd, req);
  CTService_Request_free(req);
  IPCMessage_free(msg);
  DBG_INFO("DisconnectReader request finished");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderClient_RequestCommandReader(CTCLIENTDATA *cd,
					    int *requestid,
					    int serviceid,
					    int tid,
					    const char *sendBuffer,
					    int sendBufferLength) {
  CTSERVICEREQUEST *req;
  ERRORCODE err;
  int localreqid;

  DBG_ENTER;
  assert(cd);

  /* assign a request id */
  localreqid=++(cd->nextMessageId);

  /* create the request stub */
  req=CTService_Request_Create(serviceid,
			       READERSERVICE_MSGCODE_RQ_COMMAND,
			       READERSERVICE_MSGCODE_RQ_COMMAND_VERSION,
			       localreqid,
			       0,
			       512);
  if (!req) {
    DBG_ERROR("Could not create the request");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  }

  /* add reader id */
  err=IPCMessage_AddIntParameter(req->message, tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* add sendbuffer */
  err=IPCMessage_AddParameter(req->message,
			      sendBuffer,
			      sendBufferLength);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* finish message */
  err=IPCMessage_BuildMessage(req->message);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }

  /* send message */
  err=CTClient_SendRequest(cd,
			   req,
			   serviceid);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }
  /* done */
  *requestid=localreqid;
  return 0;
}


ERRORCODE ReaderClient_CheckCommandReader(CTCLIENTDATA *cd,
					  int requestid,
					  int *result,
					  char *recvBuffer,
					  int *recvBufferLength) {
  CTSERVICEREQUEST *req;
  IPCMESSAGE *msg;
  ERRORCODE err;
  int i;
  char *p;
  int s;

  DBG_ENTER;
  /* search request */
  req=CTClient_FindRequest(cd, requestid);
  if (!req)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  /* check for response */
  msg=CTService_Request_NextResponse(req);
  if (!msg)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_MESSAGE);

  /* check for error message */
  err=CTClient_CheckErrorMessage(cd,
				 msg);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  err=CTService_CheckMsgCodeAndVersion(msg,
				       READERSERVICE_MSGCODE_RP_COMMAND,
				       READERSERVICE_MSGCODE_RP_COMMAND_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get result */
  err=IPCMessage_IntParameter(msg,4,&i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }
  *result=i;
  DBG_INFO("Command result is %d",i);

  if (*result==CTSERVICE_SUCCESS) {
    /* get response */
    err=IPCMessage_NextParameter(msg,&p, &s);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      CTClient_DequeueRequest(cd, req);
      CTService_Request_free(req);
      IPCMessage_free(msg);
      DBG_LEAVE;
      return err;
    }
    if (s>*recvBufferLength) {
      DBG_ERROR("Recvbuffer too small");
      CTClient_DequeueRequest(cd, req);
      CTService_Request_free(req);
      IPCMessage_free(msg);
      return Error_New(0,
		       ERROR_SEVERITY_ERR,
		       Error_FindType(CTSERVICE_ERROR_TYPE),
		       CTSERVICE_ERROR_BAD_BUFFERSIZE);
    }
    *recvBufferLength=s;
    if (s)
      memmove(recvBuffer, p, s);
  }

  DBG_DEBUG("Dequeuing request");
  CTClient_DequeueRequest(cd, req);
  CTService_Request_free(req);
  IPCMessage_free(msg);
  DBG_INFO("CommandReader request finished");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderClient_RequestFindReader(CTCLIENTDATA *cd,
					 int *requestid,
					 int serviceid,
					 const char *readerType,
					 unsigned int readerFlags,
					 unsigned int readerFlagsMask){
  CTSERVICEREQUEST *req;
  ERRORCODE err;
  int localreqid;

  DBG_ENTER;
  assert(cd);

  /* assign a request id */
  localreqid=++(cd->nextMessageId);

  /* create the request stub */
  req=CTService_Request_Create(serviceid,
			       READERSERVICE_MSGCODE_RQ_FIND,
			       READERSERVICE_MSGCODE_RQ_FIND_VERSION,
			       localreqid,
			       0,
			       256);
  if (!req) {
    DBG_ERROR("Could not create the request");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  }

  /* add readerType */
  err=IPCMessage_AddStringParameter(req->message, readerType);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* add readerFlags */
  err=IPCMessage_AddIntParameter(req->message, readerFlags);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* add readerFlagsMask */
  err=IPCMessage_AddIntParameter(req->message, readerFlagsMask);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* finish message */
  err=IPCMessage_BuildMessage(req->message);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }

  /* send message */
  err=CTClient_SendRequest(cd,
			   req,
			   serviceid);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }
  /* done */
  *requestid=localreqid;
  return 0;
}


ERRORCODE ReaderClient_CheckFindReader(CTCLIENTDATA *cd,
				       int requestid,
				       int *readerbuffer,
				       int *readerBufferLength){
  CTSERVICEREQUEST *req;
  IPCMESSAGE *msg;
  ERRORCODE err;
  int i;
  int found;

  DBG_ENTER;
  /* search request */
  req=CTClient_FindRequest(cd, requestid);
  if (!req)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  /* check for response */
  msg=CTService_Request_NextResponse(req);
  if (!msg)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_MESSAGE);

  /* check for error message */
  err=CTClient_CheckErrorMessage(cd,
				 msg);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  err=CTService_CheckMsgCodeAndVersion(msg,
				       READERSERVICE_MSGCODE_RP_FIND,
				       READERSERVICE_MSGCODE_RP_FIND_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get number of matching readers */
  err=IPCMessage_IntParameter(msg,4,&found);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }
  if (found>*readerBufferLength)
    found=*readerBufferLength;
  else
    *readerBufferLength=found;

  DBG_INFO("Found %d matching readers", found);
  for (i=0; i<found; i++) {
    err=IPCMessage_NextIntParameter(msg,&readerbuffer[i]);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      IPCMessage_free(msg);
      DBG_LEAVE;
      return err;
    }
  } /* for */

  DBG_DEBUG("Dequeuing request");
  CTClient_DequeueRequest(cd, req);
  CTService_Request_free(req);
  IPCMessage_free(msg);
  DBG_INFO("FindReader request finished");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderClient_RequestStatReader(CTCLIENTDATA *cd,
					 int *requestid,
					 int serviceid,
					 int tid) {
  CTSERVICEREQUEST *req;
  ERRORCODE err;
  int localreqid;

  DBG_ENTER;
  assert(cd);

  /* assign a request id */
  localreqid=++(cd->nextMessageId);

  /* create the request stub */
  req=CTService_Request_Create(serviceid,
			       READERSERVICE_MSGCODE_RQ_STATUS,
			       READERSERVICE_MSGCODE_RQ_STATUS_VERSION,
			       localreqid,
			       0,
			       256);
  if (!req) {
    DBG_ERROR("Could not create the request");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  }

  /* add reader id  */
  err=IPCMessage_AddIntParameter(req->message, tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* finish message */
  err=IPCMessage_BuildMessage(req->message);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }

  /* send message */
  err=CTClient_SendRequest(cd,
			   req,
			   serviceid);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }
  /* done */
  *requestid=localreqid;
  return 0;
}


ERRORCODE ReaderClient_CheckStatReader(CTCLIENTDATA *cd,
				       int requestid,
				       int *result,
				       unsigned int *status,
				       char *atrbuffer,
				       int *atrlen) {
  CTSERVICEREQUEST *req;
  IPCMESSAGE *msg;
  ERRORCODE err;
  int i;
  char *p;
  int s;

  DBG_ENTER;
  /* search request */
  req=CTClient_FindRequest(cd, requestid);
  if (!req)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  /* check for response */
  msg=CTService_Request_NextResponse(req);
  if (!msg)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_MESSAGE);

  /* check for error message */
  err=CTClient_CheckErrorMessage(cd,
				 msg);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  err=CTService_CheckMsgCodeAndVersion(msg,
				       READERSERVICE_MSGCODE_RP_STATUS,
				       READERSERVICE_MSGCODE_RP_STATUS_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get result */
  err=IPCMessage_IntParameter(msg,4,&i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }
  *result=i;

  if (i==0) {
    /* ok, so read the status */
    err=IPCMessage_NextIntParameter(msg,&i);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      CTClient_DequeueRequest(cd, req);
      CTService_Request_free(req);
      IPCMessage_free(msg);
      DBG_LEAVE;
      return err;
    }
    *status=i;
    DBG_DEBUG("Stated terminal (status=%04x)", (unsigned int)i);
    /* get ATR */
    err=IPCMessage_NextParameter(msg,&p, &s);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      CTClient_DequeueRequest(cd, req);
      CTService_Request_free(req);
      IPCMessage_free(msg);
      DBG_LEAVE;
      return err;
    }
    /* copy atr */
    if (s>*atrlen)
      s=*atrlen;
    if (s)
      memmove(atrbuffer, p, s);
    *atrlen=s;
  }

  DBG_DEBUG("Dequeuing request");
  CTClient_DequeueRequest(cd, req);
  CTService_Request_free(req);
  IPCMessage_free(msg);
  DBG_INFO("StatReader request finished");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderClient_RequestWaitReader(CTCLIENTDATA *cd,
					 int *requestid,
					 int serviceid,
					 int mustChange,
					 const char *readerType,
					 unsigned int readerFlags,
					 unsigned int readerFlagsMask,
					 unsigned int status,
					 unsigned int statusMask,
					 unsigned int statusDelta){
  CTSERVICEREQUEST *req;
  ERRORCODE err;
  int localreqid;

  DBG_ENTER;
  assert(cd);

  /* assign a request id */
  localreqid=++(cd->nextMessageId);

  /* create the request stub */
  req=CTService_Request_Create(serviceid,
			       READERSERVICE_MSGCODE_RQ_WAIT,
			       READERSERVICE_MSGCODE_RQ_WAIT_VERSION,
			       localreqid,
			       0,
			       256);
  if (!req) {
    DBG_ERROR("Could not create the request");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  }

  /* add readerType */
  err=IPCMessage_AddStringParameter(req->message, readerType);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* add readerFlags */
  err=IPCMessage_AddIntParameter(req->message, readerFlags);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* add readerFlagsMask */
  err=IPCMessage_AddIntParameter(req->message, readerFlagsMask);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* add status */
  err=IPCMessage_AddIntParameter(req->message, status);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* add statusMask */
  err=IPCMessage_AddIntParameter(req->message, statusMask);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* add mustChange */
  err=IPCMessage_AddIntParameter(req->message, mustChange);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* add statusDelta */
  err=IPCMessage_AddIntParameter(req->message, statusDelta);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* finish message */
  err=IPCMessage_BuildMessage(req->message);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }

  /* send message */
  err=CTClient_SendRequest(cd,
			   req,
			   serviceid);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }
  /* done */
  *requestid=localreqid;
  return 0;
}


ERRORCODE ReaderClient_CheckWaitReader(CTCLIENTDATA *cd,
				       int requestid,
				       int *tid,
				       unsigned int *status,
				       unsigned int *readerflags,
				       unsigned int *cardId) {
  CTSERVICEREQUEST *req;
  IPCMESSAGE *msg;
  ERRORCODE err;
  int lstatus;
  int ltid;
  int lflags;
  int msgVersion;
  char *p;
  int lcardId;

  DBG_ENTER;
  /* search request */
  req=CTClient_FindRequest(cd, requestid);
  if (!req)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  /* check for response */
  msg=CTService_Request_NextResponse(req);
  if (!msg)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_MESSAGE);

  /* get msgversion (needed later) */
  err=IPCMessage_IntParameter(msg, 1, &msgVersion);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* check for error message */
  err=CTClient_CheckErrorMessage(cd,
				 msg);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  err=CTService_CheckMsgCodeAndVersion(msg,
				       READERSERVICE_MSGCODE_RP_WAIT,
				       READERSERVICE_MSGCODE_RP_WAIT_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get reader status */
  err=IPCMessage_IntParameter(msg,4,&lstatus);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get reader id */
  err=IPCMessage_NextIntParameter(msg,&ltid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get reader flags */
  err=IPCMessage_NextIntParameter(msg,&lflags);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get reader type */
  err=IPCMessage_NextStringParameter(msg, &p);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get reader name */
  err=IPCMessage_NextStringParameter(msg, &p);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  if ((msgVersion&0xff)>0) {
    /* get card id if newer message version */
    err=IPCMessage_NextIntParameter(msg,&lcardId);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      IPCMessage_free(msg);
      DBG_LEAVE;
      return err;
    }
  }
  else
    lcardId=0;

  /* Since this is a response to a persisting request the request will NOT
   * be dequeued.
   */
  *tid=ltid;
  *cardId=lcardId;
  *status=(unsigned int)lstatus;
  *readerflags=(unsigned int)lflags;

  DBG_INFO("WaitReader response handled");
  IPCMessage_free(msg);
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderClient_RequestStopWaitReader(CTCLIENTDATA *cd,
					     int *requestid,
					     int serviceid,
					     int prevRequest) {
  CTSERVICEREQUEST *req;
  ERRORCODE err;
  int localreqid;

  DBG_ENTER;
  assert(cd);

  /* TODO: delete old request */

  /* assign a request id */
  localreqid=++(cd->nextMessageId);

  /* create the request stub */
  req=CTService_Request_Create(serviceid,
			       READERSERVICE_MSGCODE_RQ_STOPWAIT,
			       READERSERVICE_MSGCODE_RQ_STOPWAIT_VERSION,
			       localreqid,
			       0,
			       256);
  if (!req) {
    DBG_ERROR("Could not create the request");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  }

  /* add old request id */
  err=IPCMessage_AddIntParameter(req->message, prevRequest);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* finish message */
  err=IPCMessage_BuildMessage(req->message);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }

  /* send message */
  err=CTClient_SendRequest(cd,
			   req,
			   serviceid);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }
  /* done */
  *requestid=localreqid;
  return 0;
}


ERRORCODE ReaderClient_CheckStopWaitReader(CTCLIENTDATA *cd,
					   int requestid,
					   int *result) {
  CTSERVICEREQUEST *req;
  IPCMESSAGE *msg;
  ERRORCODE err;
  int lresult;

  DBG_ENTER;
  /* search request */
  req=CTClient_FindRequest(cd, requestid);
  if (!req)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  /* check for response */
  msg=CTService_Request_NextResponse(req);
  if (!msg)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_MESSAGE);

  /* check for error message */
  err=CTClient_CheckErrorMessage(cd,
				 msg);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  err=CTService_CheckMsgCodeAndVersion(msg,
				       READERSERVICE_MSGCODE_RP_STOPWAIT,
				       READERSERVICE_MSGCODE_RP_STOPWAIT_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get result */
  err=IPCMessage_NextIntParameter(msg,&lresult);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  *result=lresult;

  DBG_DEBUG("Dequeuing request");
  CTClient_DequeueRequest(cd, req);
  CTService_Request_free(req);
  IPCMessage_free(msg);
  DBG_INFO("StopWaitReader request finished");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderClient_RequestStopOpenReader(CTCLIENTDATA *cd,
					     int *requestid,
					     int serviceid,
					     int prevRequest) {
  CTSERVICEREQUEST *req;
  ERRORCODE err;
  int localreqid;

  DBG_ENTER;
  assert(cd);

  /* TODO: delete old request */

  /* assign a request id */
  localreqid=++(cd->nextMessageId);

  /* create the request stub */
  req=CTService_Request_Create(serviceid,
			       READERSERVICE_MSGCODE_RQ_STOPOPEN,
			       READERSERVICE_MSGCODE_RQ_STOPOPEN_VERSION,
			       localreqid,
			       0,
			       256);
  if (!req) {
    DBG_ERROR("Could not create the request");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  }

  /* add old request id */
  err=IPCMessage_AddIntParameter(req->message, prevRequest);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return 0;
  }

  /* finish message */
  err=IPCMessage_BuildMessage(req->message);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }

  /* send message */
  err=CTClient_SendRequest(cd,
			   req,
			   serviceid);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTService_Request_free(req);
    DBG_LEAVE;
    return err;
  }
  /* done */
  *requestid=localreqid;
  return 0;
}


ERRORCODE ReaderClient_CheckStopOpenReader(CTCLIENTDATA *cd,
					   int requestid,
					   int *result) {
  CTSERVICEREQUEST *req;
  IPCMESSAGE *msg;
  ERRORCODE err;
  int lresult;

  DBG_ENTER;
  /* search request */
  req=CTClient_FindRequest(cd, requestid);
  if (!req)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_REQUEST);
  /* check for response */
  msg=CTService_Request_NextResponse(req);
  if (!msg)
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_MESSAGE);

  /* check for error message */
  err=CTClient_CheckErrorMessage(cd,
				 msg);
  if (!Error_IsOk(err)) {
    DBG_NOTICE_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  err=CTService_CheckMsgCodeAndVersion(msg,
				       READERSERVICE_MSGCODE_RP_STOPOPEN,
				       READERSERVICE_MSGCODE_RP_STOPOPEN_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* get result */
  err=IPCMessage_NextIntParameter(msg,&lresult);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTClient_DequeueRequest(cd, req);
    CTService_Request_free(req);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  *result=lresult;

  DBG_DEBUG("Dequeuing request");
  CTClient_DequeueRequest(cd, req);
  CTService_Request_free(req);
  IPCMessage_free(msg);
  DBG_INFO("StopWaitReader request finished");
  DBG_LEAVE;
  return 0;
}




