/***************************************************************************
 $RCSfile: ctcore.c,v $
                             -------------------
    cvs         : $Id: ctcore.c,v 1.29 2003/05/08 13:32:49 aquamaniac Exp $
    begin       : Fri Nov 22 2002
    copyright   : (C) 2001 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_CHIPCARD_DLL
#  define CHIPCARD_API __declspec (dllexport)
# else /* Not BUILDING_CHIPCARD_DLL */
#  define CHIPCARD_API __declspec (dllimport)
# endif /* Not BUILDING_CHIPCARD_DLL */
#else
# define CHIPCARD_API
#endif

#include "ctcore.h"

#ifdef USE_PCSC
# include "ctdriver_pcsc.h"
#endif
#include "ctdriver_ctapi.h"

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <time.h>

#include "chameleon/debug.h"

#ifdef OS_WIN32
# include <process.h> /* for getpid() */
#endif


/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                               CTCore module
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */



const char *CTCore_ErrorString(int c) {
  const char *s;

  DBG_ENTER;
  switch(c) {
  case 0:
    s="Success";
    break;

  case CTCORE_ERROR_OPEN_DRIVER_PATH:
    s="Could not open driver (bad path)";
    break;

  case CTCORE_ERROR_DRIVER_IO:
    s="Driver IO error";
    break;

  case CTCORE_ERROR_BUFFER:
    s="Buffer too small";
    break;

  case CTCORE_ERROR_NO_DRIVER:
    s="Could not allocate driver";
    break;

  case CTCORE_ERROR_LOCKED:
    s="Reader is locked";
    break;

  case CTCORE_ERROR_DRIVER_SOFT:
    s="Card command error";
    break;

  case CTCORE_ERROR_DRIVER_BUSY:
    s="Ressource temporarily not available";
    break;

  case CTCORE_ERROR_INVALID:
    s="Invalid argument";
    break;

  case CTCORE_ERROR_BAD_READER:
    s="Bad readers found";
    break;

  default:
    s=(const char*)0;
    break;

  } /* switch */
  DBG_LEAVE;
  return s;
}

int ctcore_is_initialized=0;
ERRORTYPEREGISTRATIONFORM ctcore_error_descr= {
  CTCore_ErrorString,
  0,
  CTCORE_ERROR_TYPE};



ERRORCODE CTCore_ModuleInit(){
  DBG_ENTER;
  if (!ctcore_is_initialized) {
    if (!Error_RegisterType(&ctcore_error_descr)) {
      DBG_LEAVE;
      return Error_New(0,
		       ERROR_SEVERITY_ERR,
		       ERROR_TYPE_ERROR,
		       ERROR_COULD_NOT_REGISTER);
    }
    ctcore_is_initialized=1;
  }
  DBG_LEAVE;
  return 0;
}


ERRORCODE CTCore_ModuleFini(){
  DBG_ENTER;
  if (ctcore_is_initialized) {
    ctcore_is_initialized=0;
    if (!Error_UnregisterType(&ctcore_error_descr)){
      DBG_LEAVE;
      return Error_New(0,
		       ERROR_SEVERITY_ERR,
		       ERROR_TYPE_ERROR,
		       ERROR_COULD_NOT_UNREGISTER);
    }
  }
  DBG_LEAVE;
  return 0;
}



/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                               CTCore_ReaderDescr
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */



CTREADERDESCRIPTION*
CTCore__FindReaderDescr(CTCORETABLE *ct, int id) {
  CTREADERDESCRIPTION *curr;

  DBG_ENTER;
  assert(ct);
  curr=ct->readerDescriptions;
  while(curr) {
    if (curr->id==id) {
      DBG_LEAVE;
      return curr;
    }
    curr=curr->next;
  } /* while */

  DBG_LEAVE;
  return 0;
}


void CTCore__AddReaderDescr(CTCORETABLE *ct, CTREADERDESCRIPTION *rd) {
  CTREADERDESCRIPTION *curr;
  CTREADERDESCRIPTION *rdcopy;

  DBG_ENTER;
  assert(ct);
  assert(rd);

  if (rd->name[0]) {
    /* name given, try to find this reader */
    curr=ct->readerDescriptions;
    DBG_DEBUG("Looking for reader \"%s\"", rd->name);
    while(curr) {
      if (strcmp(rd->name, curr->name)==0) {
	/* reader found, modify it */
	DBG_NOTICE("Modifying existing reader \"%s\"",rd->name);
	curr->readerFlags=rd->readerFlags;
	if (curr->typeName[0]==0) {
#ifdef HAVE_STRNCPY
	  strncpy(curr->typeName,sizeof(curr->typeName), rd->typeName);
#else
	  strcpy(curr->typeName,rd->typeName);
#endif
	}
	rd->id=curr->id;
	return;
      }
      curr=curr->next;
    } /* while */
  }

  DBG_DEBUG("Will now add reader description");
  /* update information */
  rd->id=++(ct->nextReaderId);
  /* copy information */
  rdcopy=CTCore_ReaderDescr_new();
  *rdcopy=*rd;
  curr=ct->readerDescriptions;
  if (!curr) {
    ct->readerDescriptions=rdcopy;
  }
  else {
    /* find last */
    while(curr->next) {
      curr=curr->next;
    } /* while */
    curr->next=rdcopy;
  }
  DBG_NOTICE("Added reader \"%s\" (id=%d)",
	     rd->name, rd->id);
  DBG_LEAVE;
}


void CTCore__RemoveReaderDescr(CTCORETABLE *ct, CTREADERDESCRIPTION *rd) {
  CTREADERDESCRIPTION *curr;

  DBG_ENTER;
  assert(ct);
  assert(rd);

  curr=ct->readerDescriptions;
  if (curr) {
    if (curr==rd) {
      ct->readerDescriptions=curr->next;
    }
    else {
      /* find predecessor */
      while(curr->next!=rd) {
	curr=curr->next;
      } /* while */
      if (curr)
	curr->next=rd->next;
    }
  }
  DBG_LEAVE;
}



/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                               CTCore_Reader
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */



CTREADERTABLE *CTCore_Reader_new(){
  CTREADERTABLE *rt;

  DBG_ENTER;
  rt=(CTREADERTABLE *)malloc(sizeof(CTREADERTABLE));
  assert(rt);
  memset(rt,0,sizeof(CTREADERTABLE));
  DBG_LEAVE;
  return rt;
}


void CTCore_Reader_free(CTREADERTABLE *rt){
  DBG_ENTER;
  if (rt)
    free(rt);
  DBG_LEAVE;
}


CTREADERTABLE *CTCore__FindReader(CTCORETABLE *ct, int id) {
  CTREADERTABLE *curr;

  DBG_ENTER;
  assert(ct);
  curr=ct->readers;
  while(curr) {
    if (curr->descr)
      if (curr->descr->id==id) {
	DBG_LEAVE;
	return curr;
      }
    curr=curr->next;
  } /* while */

  DBG_LEAVE;
  return 0;
}


void CTCore__AddReader(CTCORETABLE *ct, CTREADERTABLE *rt) {
  CTREADERTABLE *curr;

  DBG_ENTER;
  assert(ct);
  assert(rt);

  curr=ct->readers;
  if (!curr) {
    ct->readers=rt;
  }
  else {
    /* find last */
    while(curr->next) {
      curr=curr->next;
    } /* while */
    curr->next=rt;
  }
  DBG_LEAVE;
}


void CTCore__RemoveReader(CTCORETABLE *ct, CTREADERTABLE *rt) {
  CTREADERTABLE *curr;

  DBG_ENTER;
  assert(ct);
  assert(rt);

  curr=ct->readers;
  if (curr) {
    if (curr==rt) {
      ct->readers=curr->next;
    }
    else {
      /* find predecessor */
      while(curr->next!=rt) {
	curr=curr->next;
      } /* while */
      if (curr)
	curr->next=rt->next;
    }
  }
  DBG_LEAVE;
}



/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                               CTCore_Driver
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */



CTDRIVERTABLE *CTCore_Driver_new(){
  CTDRIVERTABLE *dt;

  DBG_ENTER;
  dt=(CTDRIVERTABLE *)malloc(sizeof(CTDRIVERTABLE));
  assert(dt);
  memset(dt,0,sizeof(CTDRIVERTABLE));
  DBG_LEAVE;
  return dt;
}


void CTCore_Driver_free(CTDRIVERTABLE *dt){
  DBG_ENTER;
  if (dt)
    free(dt);
  DBG_LEAVE;
}


CTDRIVERTABLE *CTCore__FindDriver(CTCORETABLE *ct, const char *name) {
  CTDRIVERTABLE *curr;

  DBG_ENTER;
  assert(ct);
  curr=ct->drivers;
  while(curr) {
    if (strcmp(curr->name,name)==0){
      DBG_LEAVE;
      return curr;
    }
    curr=curr->next;
  } /* while */

  DBG_LEAVE;
  return 0;
}


void CTCore__AddDriver(CTCORETABLE *ct, CTDRIVERTABLE *dt) {
  CTDRIVERTABLE *curr;

  DBG_ENTER;
  assert(ct);
  assert(dt);

  curr=ct->drivers;
  if (!curr) {
    ct->drivers=dt;
  }
  else {
    /* find last */
    while(curr->next) {
      curr=curr->next;
    } /* while */
    curr->next=dt;
  }
  DBG_LEAVE;
}


void CTCore__RemoveDriver(CTCORETABLE *ct, CTDRIVERTABLE *dt) {
  CTDRIVERTABLE *curr;

  DBG_ENTER;
  assert(ct);
  assert(dt);

  curr=ct->drivers;
  if (curr) {
    if (curr==dt) {
      ct->drivers=curr->next;
    }
    else {
      /* find predecessor */
      while(curr->next!=dt) {
	curr=curr->next;
      } /* while */
      if (curr)
	curr->next=dt->next;
    }
  }
  DBG_LEAVE;
}



/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                               CTCore_Client
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */



CTCLIENTTABLE *CTCore_Client_new(){
  CTCLIENTTABLE *ct;

  DBG_ENTER;
  ct=(CTCLIENTTABLE *)malloc(sizeof(CTCLIENTTABLE));
  assert(ct);
  memset(ct,0,sizeof(CTCLIENTTABLE));
  DBG_LEAVE;
  return ct;
}


void CTCore_Client_free(CTCLIENTTABLE *ct){
  DBG_ENTER;
  if (ct)
    free(ct);
  DBG_LEAVE;
}


CTCLIENTTABLE *CTCore__FindClient(CTCORETABLE *ct, int id) {
  CTCLIENTTABLE *curr;

  DBG_ENTER;
  assert(ct);
  curr=ct->clients;
  while(curr) {
    if (curr->id==id) {
      DBG_LEAVE;
      return curr;
    }
    curr=curr->next;
  } /* while */

  DBG_LEAVE;
  return 0;
}


void CTCore__AddClient(CTCORETABLE *ct, CTCLIENTTABLE *cl) {
  CTCLIENTTABLE *curr;

  DBG_ENTER;
  assert(ct);
  assert(cl);

  curr=ct->clients;
  if (!curr) {
    ct->clients=cl;
  }
  else {
    /* find last */
    while(curr->next) {
      curr=curr->next;
    } /* while */
    curr->next=cl;
  }
  DBG_LEAVE;
}


void CTCore__RemoveClient(CTCORETABLE *ct, CTCLIENTTABLE *cl) {
  CTCLIENTTABLE *curr;

  DBG_ENTER;
  assert(ct);
  assert(cl);

  curr=ct->clients;
  if (curr) {
    if (curr==cl) {
      ct->clients=curr->next;
    }
    else {
      /* find predecessor */
      while(curr->next!=cl) {
	curr=curr->next;
      } /* while */
      if (curr)
	curr->next=cl->next;
    }
  }
  DBG_LEAVE;
}



/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                                 CTCore
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */



CTCORETABLE *CTCore_new(){
  CTCORETABLE *ct;

  DBG_ENTER;
  ct=(CTCORETABLE *)malloc(sizeof(CTCORETABLE));
  assert(ct);
  memset(ct,0,sizeof(CTCORETABLE));
#ifdef HAVE_RANDOM
  srandom(time(0));
  ct->nextClientId=((unsigned int)random())&0xfff;;
  ct->nextReaderId=((unsigned int)random())&0xfff;
#else
  srand(time(0));
  ct->nextClientId=((unsigned int)rand())&0xfff;;
  ct->nextReaderId=((unsigned int)rand())&0xfff;
#endif
  DBG_LEAVE;
  return ct;
}


void CTCore_free(CTCORETABLE *ct){
  DBG_ENTER;
  if (ct) {
    if (ct->driverDescriptions)
      Config_free(ct->driverDescriptions);
    free(ct);
  }
  DBG_LEAVE;
}


CTDRIVERTABLE *CTCore__AllocDriver(CTCORETABLE *ct, CTREADERDESCRIPTION *rd){
  CTDRIVERTABLE *dt;
  ERRORCODE err;

  DBG_ENTER;
  assert(ct);
  assert(rd);
  /* check if the driver already exists */
  dt=CTCore__FindDriver(ct, rd->driverName);
  if (!dt) {
    /* it doesn't, so create it */
    dt=CTCore_Driver_new();
    if (strlen(rd->driverName)+1<sizeof(dt->name))
      strcpy(dt->name,rd->driverName);

    switch(rd->driverType) {
    case DriverTypePCSC:
#ifdef USE_PCSC
      err=CTDriver_PCSC_Open(dt, rd, ct->driverDescriptions);
      if (!Error_IsOk(err)) {
	DBG_WARN_ERR(err);
	CTCore_Driver_free(dt);
	dt=0;
      }
#else
      err=Error_New(0,
		    ERROR_SEVERITY_ERR,
		    Error_FindType(CTCORE_ERROR_TYPE),
		    CTCORE_ERROR_NO_DRIVER);
      DBG_WARN("PCSC not supported.");
      CTCore_Driver_free(dt);
      dt=0;
#endif
      break;

    case DriverTypeCTAPI:
      err=CTDriver_CTAPI_Open(dt, rd, ct->driverDescriptions);
      if (!Error_IsOk(err)) {
	DBG_ERROR_ERR(err);
	CTCore_Driver_free(dt);
	dt=0;
      }
      break;

    default:
      DBG_ERROR("Unknown driver type for reader \"%d\": %d",
		rd->id, rd->driverType);
      CTCore_Driver_free(dt);
      dt=0;
      break;
    } /* switch */
    if (dt)
      CTCore__AddDriver(ct,dt);
  }
  /* increase usageCount */
  if (dt)
    dt->usageCount++;
  DBG_LEAVE;
  return dt;
}


void CTCore__ReleaseDriver(CTCORETABLE *ct, CTDRIVERTABLE *dt) {
  ERRORCODE err;

  DBG_ENTER;
  assert(ct);
  assert(dt);

  dt->usageCount--;
  if (dt->usageCount<1) {
    /* close driver */
    assert(dt->closeDriver);
    err=dt->closeDriver(dt);
    if (!Error_IsOk(err)) {
      DBG_WARN_ERR(err);
    }
    CTCore__RemoveDriver(ct, dt);
    CTCore_Driver_free(dt);
  }
  DBG_LEAVE;
}



CTREADERTABLE *CTCore__AllocReader(CTCORETABLE *ct, CTREADERDESCRIPTION *rd){
  CTREADERTABLE *rt;
  ERRORCODE err;

  DBG_ENTER;
  assert(ct);
  assert(rd);
  /* check if the reader already exists */
  rt=CTCore__FindReader(ct, rd->id);
  if (!rt) {
    /* it doesn't, so create it */
    rt=CTCore_Reader_new();
    rt->descr=rd;
    rt->driver=CTCore__AllocDriver(ct, rd);
    if (!rt->driver) {
      DBG_WARN("Could not allocate driver for reader \"%d\"\n",
	       rd->id);
      CTCore_Reader_free(rt);
      rt=0;
    }
    else {
      assert(rt->driver->allocTerm);
      err=rt->driver->allocTerm(rt);
      if (!Error_IsOk(err)) {
	DBG_ERROR_ERR(err);
	CTCore__ReleaseDriver(ct, rt->driver);
	CTCore_Reader_free(rt);
	rt=0;
      }
    }
    /* add new reader to the list */
    if (rt)
      CTCore__AddReader(ct,rt);
  }
  /* increase usageCount */
  if (rt)
    rt->usageCount++;
  DBG_LEAVE;
  return rt;
}


void CTCore__ReleaseReader(CTCORETABLE *ct,
			   CTREADERTABLE *rt){
  ERRORCODE err;

  DBG_ENTER;
  assert(ct);
  assert(rt);
  assert(rt->driver);

  rt->usageCount--;
  if (rt->usageCount<1) {
    /* disconnect terminal if it is connected */
    if (rt->openCount>0) {
      assert(rt->driver->disconnectTerm);
      err=rt->driver->disconnectTerm(rt);
      if (!Error_IsOk(err)){
	DBG_WARN_ERR(err);
      }
      rt->openCount=0;
    }

    /* unregister from driver, eventually remove reader */
    rt->usageCount=0;
    assert(rt->driver->releaseTerm);
    err=rt->driver->releaseTerm(rt);
    if (!Error_IsOk(err)){
      DBG_ERROR_ERR(err);
    }
    CTCore__RemoveReader(ct, rt);
    CTCore__ReleaseDriver(ct, rt->driver);
    CTCore_Reader_free(rt);
  }
  DBG_LEAVE;
}


int CTCore_RegisterClient(CTCORETABLE *ct){
  CTCLIENTTABLE *cl;

  DBG_ENTER;
  cl=CTCore_Client_new();
  cl->id=ct->nextClientId++;
  CTCore__AddClient(ct, cl);
  DBG_LEAVE;
  DBG_INFO("Registered client %d",cl->id);
  return cl->id;
}


void CTCore_UnregisterClient(CTCORETABLE *ct, int id){
  CTCLIENTTABLE *cl;
  CTREADERTABLE *curr;
  int i;
  ERRORCODE err;

  DBG_ENTER;
  cl=CTCore__FindClient(ct,id);
  if (cl) {
    /* unregister all used readers */
    for (i=0; i<CT_READERS_PER_CLIENT; i++) {
      curr=cl->readers[i];
      if (curr) {
	/* unlock if locked by this client */
	if (curr->lockedByClient==cl->id) {
	  assert(curr->descr);
	  DBG_WARN("Reader \"%s\" still locked by client %d, disconnecting",
		   curr->descr->name, cl->id);
	  err=curr->driver->disconnectTerm(curr);
	  curr->lockedByClient=0;
	  curr->softDeltaStatus|=CTREADERSTATUS_LOCKED_BY_OTHER;
	  if (!Error_IsOk(err)) {
	    DBG_NOTICE_ERR(err);
	  }
	}
	CTCore__ReleaseReader(ct, curr);
      }
    } /* for */

    CTCore__RemoveClient(ct,cl);
    CTCore_Client_free(cl);
  }
  DBG_LEAVE;
}


int CTCore_AllocTerminal(CTCORETABLE *ct, int cid, int tid,
			 CTREADERDESCRIPTION **descr){
  CTCLIENTTABLE *cl;
  CTREADERDESCRIPTION *rd;
  int i;

  DBG_ENTER;
  /* find client */
  cl=CTCore__FindClient(ct,cid);
  if (!cl) {
    DBG_WARN("Client %d not found", cid);
    DBG_LEAVE;
    return -1;
  }

  /* find reader description */
  rd=CTCore__FindReaderDescr(ct,tid);
  if (!rd) {
    DBG_ERROR("Reader \"%d\" not found",tid);
    DBG_LEAVE;
    return -1;
  }

  /* find free terminal slot */
  for (i=0; i<CT_READERS_PER_CLIENT; i++) {
    if (cl->readers[i]==0)
      break;
  } /* for */
  if (i>=CT_READERS_PER_CLIENT) {
    DBG_ERROR("All reader slots in use");
    DBG_LEAVE;
    return -1;
  }

  /* allocate reader */
  cl->readers[i]=CTCore__AllocReader(ct, rd);
  if (cl->readers[i]==0) {
    DBG_WARN("Could not allocate reader");
    DBG_LEAVE;
    return -1;
  }

  /* return reader id */
  *descr=rd;
  DBG_LEAVE;
  return i;
}


void CTCore_ReleaseTerminal(CTCORETABLE *ct, int cid, int tid){
  CTCLIENTTABLE *cl;

  DBG_ENTER;
  if (tid>=CT_READERS_PER_CLIENT || tid<0) {
    DBG_WARN("Invalid reader id %d",tid);
    DBG_LEAVE;
    return;
  }

  /* find client */
  cl=CTCore__FindClient(ct,cid);
  if (!cl) {
    DBG_WARN("Client %d not found",cid);
    DBG_LEAVE;
    return;
  }

  /* check reader */
  if (cl->readers[tid]==0) {
    DBG_WARN("Reader %d not allocated", tid);
    DBG_LEAVE;
    return;
  }

  /* really release reader */
  CTCore__ReleaseReader(ct, cl->readers[tid]);
  cl->readers[tid]=0;
  DBG_LEAVE;
}


int CTCore__CheckLock(CTREADERTABLE *rt,
		      int cid){
  assert(rt);
  if (cid==0)
    return 1;
  return (rt->lockedByClient==0 || rt->lockedByClient==cid);
}



ERRORCODE CTCore_ConnectTerminal(CTCORETABLE *ct, int cid, int tid,
				 char *atrbuffer,
				 int *atrbufferlen){
  CTCLIENTTABLE *cl;
  ERRORCODE err;

  DBG_ENTER;
  if (tid>=CT_READERS_PER_CLIENT || tid<0) {
    DBG_WARN("Invalid reader id %d",tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }
  /* find client */
  cl=CTCore__FindClient(ct,cid);
  if (!cl) {
    DBG_WARN("Client %d not found",cid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_CLIENT);
  }

  /* check reader */
  if (cl->readers[tid]==0) {
    DBG_WARN("Reader %d not allocated", tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }

  if (!CTCore__CheckLock(cl->readers[tid], cl->id)) {
    DBG_WARN("Reader %d is locked by someone else", tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_LOCKED);
  }
  /* now lock the reader for this client */
  cl->readers[tid]->lockedByClient=cl->id;
  cl->readers[tid]->lockedSince=time(0);
  DBG_DEBUG("Reader %d locked for %d", tid, cl->id);

  assert(cl->readers[tid]->driver);
  assert(cl->readers[tid]->driver->connectTerm);
  err=cl->readers[tid]->driver->connectTerm(cl->readers[tid],
					    atrbuffer,
					    atrbufferlen);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    cl->readers[tid]->lockedByClient=0;
    DBG_DEBUG("Reader %d unlocked for %d", tid, cl->id);
    DBG_LEAVE;
    return err;
  }

  DBG_LEAVE;
  return 0;
}


ERRORCODE CTCore_DisconnectTerminal(CTCORETABLE *ct, int cid, int tid){
  CTCLIENTTABLE *cl;
  ERRORCODE err;

  DBG_ENTER;
  if (tid>=CT_READERS_PER_CLIENT || tid<0) {
    DBG_WARN("Invalid reader id %d",tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }
  /* find client */
  cl=CTCore__FindClient(ct,cid);
  if (!cl) {
    DBG_WARN("Client %d not found",cid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_CLIENT);
  }

  /* check reader */
  if (cl->readers[tid]==0) {
    DBG_WARN("Reader %d not allocated", tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }

  if (!CTCore__CheckLock(cl->readers[tid], cl->id)) {
    DBG_WARN("Reader %d is locked by someone else", tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_LOCKED);
  }
  cl->readers[tid]->lockedByClient=0;
  DBG_DEBUG("Reader %d unlocked for %d", tid, cl->id);
  assert(cl->readers[tid]->driver);
  assert(cl->readers[tid]->driver->disconnectTerm);
  err=cl->readers[tid]->driver->disconnectTerm(cl->readers[tid]);
  cl->readers[tid]->softDeltaStatus|=CTREADERSTATUS_LOCKED_BY_OTHER;
  if (!Error_IsOk(err)) {
    DBG_WARN_ERR(err);
    DBG_LEAVE;
    return err;
  }

  DBG_LEAVE;
  return 0;
}


ERRORCODE CTCore_CommandTerminal(CTCORETABLE *ct, int cid, int tid,
				 const char *sendBuffer,
				 int sendBufferLength,
				 char *recvBuffer,
				 int *recvBufferLength){
  CTCLIENTTABLE *cl;
  ERRORCODE err;

  DBG_ENTER;
  if (tid>=CT_READERS_PER_CLIENT || tid<0) {
    DBG_WARN("Invalid reader id %d",tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }
  /* find client */
  cl=CTCore__FindClient(ct,cid);
  if (!cl) {
    DBG_WARN("Client %d not found",cid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_CLIENT);
  }

  /* check reader */
  if (cl->readers[tid]==0) {
    DBG_WARN("Reader %d not allocated", tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }

  if (!CTCore__CheckLock(cl->readers[tid], cl->id)) {
    DBG_WARN("Reader %d is locked by someone else", tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_LOCKED);
  }

  assert(cl->readers[tid]->driver);
  assert(cl->readers[tid]->driver->commandTerm);
  err=cl->readers[tid]->driver->commandTerm(cl->readers[tid],
					    sendBuffer,
					    sendBufferLength,
					    recvBuffer,
					    recvBufferLength);
  if (!Error_IsOk(err)) {
    DBG_WARN_ERR(err);
    DBG_LEAVE;
    return err;
  }

  DBG_LEAVE;
  return 0;

}


int CTCore_GetClientReaderId(CTCORETABLE *ct, int id, int tid) {
  CTCLIENTTABLE *curr;

  DBG_ENTER;
  assert(ct);
  if (tid>=CT_READERS_PER_CLIENT) {
    DBG_ERROR("Index number too high (%d)", tid);
  }
  else {
    curr=ct->clients;
    while(curr) {
      if (curr->id==id) {
	DBG_LEAVE;
	if (curr->readers[tid]==0) {
	  DBG_ERROR("reader not allocated (%d)", tid);
	  break;
	}
        assert(curr->readers[tid]->descr);
	return curr->readers[tid]->descr->id;
      }
      curr=curr->next;
    } /* while */
  }

  DBG_LEAVE;
  return -1;
}



/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                             CTCore init/deinit
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


ERRORCODE CTCore_Init(CTCORETABLE *ct, CONFIGGROUP *driverdescrs){
#ifdef USE_PCSC
  CTDRIVERTABLE *dt;
  CTREADERDESCRIPTION *rd;
#endif
  DBG_ENTER;
#ifdef USE_PCSC
  /* enumerate pcsc readers */
  DBG_NOTICE("Checking for PC/SC readers");
  rd=CTCore_ReaderDescr_new();
  rd->driverType=DriverTypePCSC;
  if (sizeof(rd->driverName)>=strlen("PCSC")+1)
    strcpy(rd->driverName,"PCSC");
  dt=CTCore__AllocDriver(ct, rd);
  CTCore_ReaderDescr_free(rd);
  if (!dt) {
    DBG_WARN("PCSC supported but not available");
  }
  else {
    /* I know that the PCSC driver can enumerate terminals (I wrote it),
     * hence the assertion */
    assert(dt->enumTerms);
    ct->readerDescriptions=dt->enumTerms(dt);
    rd=ct->readerDescriptions;
    /* assign ids to the readers */
    DBG_INFO("Assigning terminal ids:");
    while (rd) {
      rd->id=++(ct->nextReaderId);
      DBG_INFO("  Terminal %d: %s\n",rd->id, rd->name);
      rd=rd->next;
    } /* while */
  }
#else
  DBG_WARN("No support for PCSC");
#endif

  ct->driverDescriptions=driverdescrs;
  DBG_LEAVE;
  return 0;
}


ERRORCODE CTCore_Fini(CTCORETABLE *ct){
  CTREADERDESCRIPTION *currrd, *nextrd;
  CTCLIENTTABLE *currcl, *nextcl;
  CTDRIVERTABLE *currdt, *nextdt;

  DBG_ENTER;
  /* unregister all clients */
  currcl=ct->clients;
  while(currcl) {
    nextcl=currcl->next;
    DBG_INFO("Unregistering client %d\n",currcl->id);
    CTCore_UnregisterClient(ct, currcl->id);
    currcl=nextcl;
  } /* while */
  ct->clients=0;

  /* release all drivers */
  currdt=ct->drivers;
  while(currdt) {
    nextdt=currdt->next;
    DBG_INFO("Releasing driver \"%s\"",currdt->name);
    CTCore__ReleaseDriver(ct, currdt);
    currdt=nextdt;
  } /* while */
  ct->drivers=0;

  /* free all reader descriptions */
  currrd=ct->readerDescriptions;
  while(currrd) {
    nextrd=currrd->next;
    DBG_INFO("Removing reader description for \"%s\"",
	     currrd->name);
    CTCore_ReaderDescr_free(currrd);
    currrd=nextrd;
  } /* while */
  ct->readerDescriptions=0;

  if (ct->clients)
    DBG_WARN("CTCore_Fini: Still clients enlisted !");
  if (ct->readers)
    DBG_WARN("CTCore_Fini: Still readers enlisted !");
  if (ct->drivers)
    DBG_WARN("CTCore_Fini: Still drivers enlisted !");

  DBG_LEAVE;
  return 0;
}


ERRORCODE CTCore_AddReader(CTCORETABLE *ct, CTREADERDESCRIPTION *rd){
  DBG_ENTER;
  assert(ct);
  assert(rd);
  CTCore__AddReaderDescr(ct, rd);
  DBG_LEAVE;
  return 0;
}


void CTCore__ToString(const char *s, int l, char *buffer, int bs) {
  unsigned int i;
  char lbuffer[32];

  assert(buffer);
  assert(bs);
  buffer[bs-1]=0;
  buffer[0]=0;
  for (i=0; i<l; i++) {
    sprintf(lbuffer, "%02x",(unsigned char)s[i]);
    if (strlen(buffer)+strlen(lbuffer)+1>bs)
      return;
    strcat(buffer, lbuffer);
  } /* for */
}


void CTCore_WalkTerminals(CTCORETABLE *ct){
  CTREADERTABLE *curr;
  ERRORCODE err;
  char atrbuffer[256];
  int atrbufferlen;

  DBG_ENTER;
  curr=ct->readers;
  while(curr) {
    DBG_DEBUG("Checking terminal status");
    assert(curr->driver);
    assert(curr->driver->statTerm);
    atrbufferlen=sizeof(atrbuffer);
    err=curr->driver->statTerm(curr, atrbuffer, &atrbufferlen);
    if (!Error_IsOk(err)) {
      DBG_DEBUG_ERR(err);
    }
    else {
      if ((curr->deltaStatus & CTREADERSTATUS_CONNECTED) &&
	  (curr->lastStatus & CTREADERSTATUS_CONNECTED)) {
	char sbuffer[256];

	/* increment card Id */
	curr->cardId++;
	CTCore__ToString(atrbuffer, atrbufferlen,
			 sbuffer, sizeof(sbuffer));
	DBG_NOTICE("Card inserted (ATR=%s) (cardId=%x)",
		   sbuffer,
		   curr->cardId);
      }
    }
    if (curr->softDeltaStatus) {
      /* include soft delta status. This status is set by CTCore when
       * disconnecting a card reader, so that other waiting clients may get
       * access to this reader
       */
      curr->deltaStatus|=curr->softDeltaStatus;
      curr->softDeltaStatus=0;
    }
    curr=curr->next;
  } /* while */
  DBG_LEAVE;
}


ERRORCODE CTCore_CheckReaderStatus(CTCORETABLE *ct, int cid, int tid,
				   unsigned int *status,
				   char *atrbuffer,
				   int *atrbufferlen){
  CTCLIENTTABLE *cl;
  ERRORCODE err;

  DBG_ENTER;
  if (tid>=CT_READERS_PER_CLIENT || tid<0) {
    DBG_WARN("Invalid reader id %d",tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }
  /* find client */
  cl=CTCore__FindClient(ct,cid);
  if (!cl) {
    DBG_WARN("Client %d not found",cid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_CLIENT);
  }

  /* check reader */
  if (cl->readers[tid]==0) {
    DBG_WARN("Reader %d not allocated", tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }

  assert(cl->readers[tid]->driver);
  assert(cl->readers[tid]->driver->statTerm);
  err=cl->readers[tid]->driver->statTerm(cl->readers[tid],
					 atrbuffer, atrbufferlen);
  if (!Error_IsOk(err)) {
    DBG_WARN_ERR(err);
    return err;
  }
  *status=cl->readers[tid]->lastStatus;
  if (cl->readers[tid]->lockedByClient!=0 &&
      cl->readers[tid]->lockedByClient!=cl->id)
    *status|=CTREADERSTATUS_LOCKED_BY_OTHER;

  return 0;
}



ERRORCODE CTCore_GetReaderStatus(CTCORETABLE *ct, int cid, int tid,
				 unsigned int *status,
				 char *atrbuffer,
				 int *atrbufferlen){
  CTCLIENTTABLE *cl;
  int s;

  DBG_ENTER;
  if (tid>=CT_READERS_PER_CLIENT || tid<0) {
    DBG_WARN("Invalid reader id %d",tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }
  /* find client */
  cl=CTCore__FindClient(ct,cid);
  if (!cl) {
    DBG_WARN("Client %d not found",cid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_CLIENT);
  }

  /* check reader */
  if (cl->readers[tid]==0) {
    DBG_WARN("Reader %d not allocated", tid);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }
  *status=cl->readers[tid]->lastStatus;
  if (cl->readers[tid]->lockedByClient!=0 &&
      cl->readers[tid]->lockedByClient!=cl->id)
    *status|=CTREADERSTATUS_LOCKED_BY_OTHER;

  /* copy ATR */
  s=(cl->readers[tid]->atrlen>*atrbufferlen)?*atrbufferlen:
    cl->readers[tid]->atrlen;
  if (s<cl->readers[tid]->atrlen) {
    DBG_WARN("ATR buffer too short, copying partially");
  }
  if (s)
    memmove(atrbuffer, cl->readers[tid]->atr, s);
  *atrbufferlen=s;

  return 0;
}







