/***************************************************************************
 $RCSfile: ctcardbase.cpp,v $
                             -------------------
    cvs         : $Id: ctcardbase.cpp,v 1.3 2003/05/13 22:32:10 aquamaniac Exp $
    begin       : Wed Apr 23 2003
    copyright   : (C) 2003 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_CHIPCARD_DLL
#  define CHIPCARD_API __declspec (dllexport)
# else /* Not BUILDING_CHIPCARD_DLL */
#  define CHIPCARD_API __declspec (dllimport)
# endif /* Not BUILDING_CHIPCARD_DLL */
#else
# define CHIPCARD_API
#endif


#include "ctcardbase.h"
#include "chameleon/chameleon.h"
#include "chameleon/debug.h"
#include "ctmisc.h"
#include <assert.h>
#include <time.h>



/*___________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                             CTCardBase
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


CTCardBase::CTCardBase(const CTReaderContext &ctx)
:_openCount(0)
,_timeout(30)
,_reader(ctx)
{
}


CTCardBase::CTCardBase(const CTCardBase &c)
{

  *this=c;
  _reader=c._reader;
}


CTCardBase::~CTCardBase(){
}


CTCardBase::CallBackResult CTCardBase::callback(bool first){
  return CallBackContinue;
}


CTError CTCardBase::open(){
  CTError err;

  if (_openCount) {
    _openCount++;
    DBG_INFO("Card already open (%d)", _openCount-1);
    return CTError();
  }

  err=allocate();
  if (!err.isOk()) {
    DBG_ERROR("Error opening card: %s",err.errorString().c_str());
    return CTError("CTCardBase::open", err);
  }

  err=connect(_atr);
  if (!err.isOk()) {
    DBG_ERROR("Error opening card: %s",err.errorString().c_str());
    release();
    return CTError("CTCardBase::open", err);
  }

  _openCount++;
  return CTError();
}


CTError CTCardBase::close(bool force){
  CTError err1;
  CTError err2;

  if (_openCount<1) {
    DBG_INFO("Card is not open");
    return CTError("CTCard::close()",
		   k_CTERROR_INVALID, 0, 0,
		   "Card is not open");
  }

  _openCount--;

  if (_openCount<1 || force) {
    // need to deallocate the reader context
    _openCount=0;
    err1=disconnect();
    err2=release();
    if (!err2.isOk()) {
      DBG_ERROR("Error closing card: %s",err2.errorString().c_str());
      return CTError("CTCardBase::close", err2);
    }
    if (!err1.isOk()) {
      DBG_ERROR("Error closing card: %s",err1.errorString().c_str());
      return CTError("CTCardBase::close", err1);
    }
  }

  return CTError();
}


unsigned int CTCardBase::readerFlags() const {
  return _reader.readerFlags();
}


unsigned int CTCardBase::readerStatus() const {
  return _reader.readerStatus();
}


const CHIPCARD_READERDESCR &CTCardBase::readerDescription() const {
  return _reader.readerDescription();
}


CTError CTCardBase::sendAPDU(const string &apdu, string &response){
  CTError err;

  if (_openCount<1) {
    DBG_INFO("Card is not open");
    return CTError("CTCard::sendAPDU()",
		   k_CTERROR_INVALID, 0, 0,
		   "Card is not open");
  }

  err=command(apdu, response);
  return CTError("CTCardBase::sendAPDU", err);
}






CTError CTCardBase::allocate(){
  int err;
  int requestid;
  CHIPCARD_READERDESCR *lrd;
  int lth;

  DBG_DEBUG("Will allocate terminal %0x\n",_reader.readerId());

  /* allocate reader */
  err=ChipCard_RequestAllocReader(&requestid,
				  _reader.readerId());
  if (err!=CHIPCARD_SUCCESS) {
    DBG_NOTICE("Chipcard error %d",err);
    return CTError("CTCardBase::allocate",
		   k_CTERROR_API, err, 0,
		   "Unable to allocate reader");
  }

  err=_responseLoop(requestid, _timeout);
  if (err!=0) {
    DBG_NOTICE("No response");
    return CTError("CTCardBase::allocate",
		   k_CTERROR_API, err, 0,
		   "Unable to allocate reader");
  }
  err=ChipCard_CheckAllocReader(requestid,&lth,&lrd);
  if (err!=0) {
    DBG_NOTICE("No response");
    return CTError("CTCardBase::allocate",
		   k_CTERROR_API, err, 0,
		   "Unable to allocate reader");
  }

  _reader.setReaderDescription(*lrd);
  _reader.setReaderNumber(lth);
  return CTError();

}


CTError CTCardBase::release(){
  int err;
  int requestid;

  if (_reader.readerNumber()==-1) {
    DBG_NOTICE("Reader not allocated");
    return CTError("CTCardBase::release",
		   k_CTERROR_INVALID, 0, 0,
		   "Reader not allocated");
  }

  err=ChipCard_RequestReleaseReader(&requestid, _reader.readerNumber());
  if (err!=CHIPCARD_SUCCESS) {
    DBG_NOTICE("Chipcard error %d\n",
	       err);
    return CTError("CTCardBase::release",
		   k_CTERROR_API, err, 0,
		   "Unable to release reader");
  }

  err=_responseLoop(requestid, _timeout);
  if (err!=0) {
    DBG_NOTICE("No response");
    return CTError("CTCardBase::release",
		   k_CTERROR_API, err, 0,
		   "Unable to release reader");
  }

  err=ChipCard_CheckReleaseReader(requestid);
  if (err!=CHIPCARD_SUCCESS) {
    DBG_NOTICE("Chipcard error %d\n",
	       err);
    return CTError("CTCardBase::release",
		   k_CTERROR_API, err, 0,
		   "Unable to release reader");
  }
  if (err!=0) {
    DBG_NOTICE("Chipcard error %d\n",
	       err);
    return CTError("CTCardBase::release",
		   k_CTERROR_API, err, 0,
		   "Unable to release reader");
  }

  _reader.setReaderNumber(-1);
  return CTError();
}


CTError CTCardBase::connect(string &atr){
  int err;
  int requestid;
  char atrbuffer[300];
  int atrlen;
  int result;

  if (_reader.readerNumber()==-1) {
    DBG_NOTICE("Reader not allocated");
    return CTError("CTCardBase::connect",
		   k_CTERROR_INVALID, 0, 0,
		   "Reader not allocated");
  }

  if (_reader.connected()) {
    DBG_NOTICE("Reader already connected");
    return CTError("CTCardBase::connect",
		   k_CTERROR_INVALID, 0, 0,
		   "Reader already connected");
  }


  /* find matching reader */
  DBG_NOTICE("Will connect to %x (%d)",_reader.readerId(), _reader.cardId());
  err=ChipCard_RequestConnect(&requestid,
			      _reader.readerNumber(),
			      _reader.cardId(), _timeout!=0);
  if (err!=CHIPCARD_SUCCESS) {
    DBG_NOTICE("Chipcard error %d\n",
	       err);
    return CTError("CTCardBase::connect",
		   k_CTERROR_API, err, 0,
		   "Unable to connect to reader");
  }

  err=_responseLoop(requestid, _timeout);
  if (err!=CHIPCARD_SUCCESS) {
    CTError lerr;

    DBG_NOTICE("No response");
    if (err!=CHIPCARD_ERROR_UNREACHABLE) {
      /* do not send a new request if the server was not reachable */
      lerr=_abortConnect(requestid, 2);
      if (!lerr.isOk()) {
	DBG_ERROR("Error aborting connect request: %s",
		  lerr.errorString().c_str());
      }
    }
    return CTError("CTCardBase::connect",
		   k_CTERROR_API, err, 0,
		   "Unable to connect to reader");
  }

  atrlen=sizeof(atrbuffer);
  err=ChipCard_CheckConnect(requestid,
			    &result,
			    atrbuffer,
			    &atrlen);
  if (err!=CHIPCARD_SUCCESS) {
    DBG_NOTICE("Chipcard error %d\n",
	       err);
    return CTError("CTCardBase::connect",
		   k_CTERROR_API, err, 0,
		   "Unable to connect to reader");
  }
  if (result!=0) {
    DBG_NOTICE("Chipcard error %d\n",
	       result);
    return CTError("CTCardBase::connect",
		   k_CTERROR_API, CHIPCARD_ERROR_NO_CARD, 0,
		   "Unable to connect to reader");
  }

  atr=string(atrbuffer, atrlen);
  _reader.setConnected(true);
  return CTError();
}


CTError CTCardBase::disconnect(){
  int err;
  int requestid;
  int result;

  if (!_reader.connected()) {
    DBG_NOTICE("Reader not connected");
    return CTError("CTCardBase::disconnect",
		   k_CTERROR_INVALID, 0, 0,
		   "Reader not connected");
  }

  err=ChipCard_RequestDisconnect(&requestid, _reader.readerNumber());
  if (err!=CHIPCARD_SUCCESS){
    DBG_NOTICE("Chipcard error %d\n",
	       err);
    return CTError("CTCardBase::disconnect",
		   k_CTERROR_API, err, 0,
		   "Unable to disconnect from reader");
  }

  err=_responseLoop(requestid, _timeout);
  if (err!=CHIPCARD_SUCCESS) {
    DBG_NOTICE("Chipcard error %d\n",
	       err);
    return CTError("CTCardBase::disconnect",
		   k_CTERROR_API, err, 0,
		   "Unable to disconnect from reader");
  }

  err=ChipCard_CheckDisconnect(requestid,
			       &result);
  if (err!=CHIPCARD_SUCCESS) {
    DBG_NOTICE("Chipcard error %d\n",
	       err);
    return CTError("CTCardBase::disconnect",
		   k_CTERROR_API, err, 0,
		   "Unable to disconnect from reader");
  }
  if (result!=0) {
    DBG_NOTICE("Chipcard error %d\n",
	       result);
    return CTError("CTCardBase::disconnect",
		   k_CTERROR_API, CHIPCARD_ERROR_NO_CARD, 0,
		   "Unable to disconnect from reader");
  }
  _reader.setConnected(false);
  return CTError();
}


CTError CTCardBase::command(const string &apdustr,
			    string &response){
  int err;
  int requestid;
  char buffer[300];
  int bufferlen;
  int result;
  CTError error;

  if (!_reader.connected()) {
    DBG_NOTICE("Reader not connected");
    return CTError("CTCardBase::command",
		   k_CTERROR_INVALID, 0, 0,
		   "Reader not connected");
  }

  DBG_INFO("Sending command %s",
           CTMisc::bin2hex(apdustr,1).c_str());

  err=ChipCard_RequestCommand(&requestid,
			      _reader.readerNumber(),
			      apdustr.data(),
			      apdustr.size());

  err=_responseLoop(requestid, _timeout);
  if (err!=CHIPCARD_SUCCESS) {
    DBG_NOTICE("Chipcard error %d\n",
	       err);
    return CTError("CTCardBase::command",
		   k_CTERROR_API, err, 0,
		   "Unable to send command");
  }

  bufferlen=sizeof(buffer);
  err=ChipCard_CheckCommand(requestid,
			    &result,
			    buffer,
			    &bufferlen);
  if (err!=CHIPCARD_SUCCESS) {
    DBG_NOTICE("Chipcard error %d\n",
	       err);
    return CTError("CTCardBase::command",
		   k_CTERROR_API, err, 0,
		   "Unable to send command");
  }
  if (result!=0) {
    DBG_NOTICE("Could not execute command %s",
	       CTMisc::bin2hex(apdustr,1).c_str());
    return CTError("CTCardBase::command",
		   k_CTERROR_API, err, 0,
		   "Unable to execute command");
  }

  // check response length
  if (bufferlen<2) {
    DBG_NOTICE("Bad length of result (%d)", response.length());
    return CTError("CTCardBase::command",
		   k_CTERROR_API, CHIPCARD_ERROR_DRIVER, 0,
		   "Bad response");
  }

  // check response
  error=CTError("CTCardBase::command",
		k_CTERROR_OK,
		(unsigned char)buffer[bufferlen-2],
		(unsigned char)buffer[bufferlen-1]);
  response.assign(buffer, bufferlen-2);

  if (!error.isOk()) {
    DBG_NOTICE("Error executing command %s (%s)",
	       CTMisc::bin2hex(apdustr).c_str(),
	       error.errorString().c_str());
  }

  return error;
}


int CTCardBase::_responseLoop(int reqid, int timeout) {
  int err;
  int i;
  time_t startTime;
  CallBackResult rv;

  i=0;
  startTime=time(NULL);
  while(1) {
    if (timeout) {
      if (difftime(time(NULL),startTime)>=timeout) {
	DBG_NOTICE("Timeout");
	return CHIPCARD_ERROR_NO_MESSAGE;
      }
    }
    err=ChipCard_CheckResponse(reqid);
    if (err!=CHIPCARD_SUCCESS) {
      if (err!=CHIPCARD_ERROR_NO_MESSAGE) {
	DBG_NOTICE("Chipcard error %d\n",
		   err);
	if (err==CHIPCARD_ERROR_UNREACHABLE ||
	    err==CHIPCARD_ERROR_NO_REQUEST ||
	    err==CHIPCARD_ERROR_INTERRUPTED)
	  return err;
      }
    }
    else {
      return CHIPCARD_SUCCESS;
    }
    err=ChipCard_Work();
    if (err!=CHIPCARD_SUCCESS) {
      DBG_NOTICE("Chipcard error %d\n",
		 err);
      if (err==CHIPCARD_ERROR_INTERRUPTED ||
	  err==CHIPCARD_ERROR_NO_TRANSPORT)
	return err;
    }
    rv=callback(i==0);
    if (rv==CallBackAbort) {
      DBG_INFO("Aborted by callback");
      return CHIPCARD_ERROR_ABORTED;
    }
    i++;
  } /* while */

  return CHIPCARD_ERROR_NO_MESSAGE;
}


CTError CTCardBase::_abortConnect(int prevreq, int timeout){
  int err;
  int requestid;
  int result;

  if (_reader.readerNumber()==-1) {
    DBG_NOTICE("Reader not allocated");
    return CTError("CTCardBase::_abortConnect",
		   k_CTERROR_INVALID, 0, 0,
		   "Reader not allocated");
  }

  if (_reader.connected()) {
    DBG_NOTICE("Reader already connected");
    return CTError("CTCardBase::_abortConnect",
		   k_CTERROR_INVALID, 0, 0,
		   "Reader already connected");
  }


  err=ChipCard_RequestStopConnect(&requestid,_reader.readerId(), prevreq);
  if (err!=CHIPCARD_SUCCESS) {
    DBG_NOTICE("Chipcard error %d\n",
	       err);
    return CTError("CTCardBase::_abortConnect",
		   k_CTERROR_API, err, 0,
		   "Unable to abort connect");
  }

  err=_responseLoop(requestid, _timeout);
  if (err!=CHIPCARD_SUCCESS) {
    DBG_NOTICE("No response");
    return CTError("CTCardBase::_abortConnect",
		   k_CTERROR_API, err, 0,
		   "Unable to abort connect");
  }

  err=ChipCard_CheckStopConnect(requestid,
				&result);
  if (err!=CHIPCARD_SUCCESS) {
    DBG_NOTICE("Chipcard error %d\n",
	       err);
    return CTError("CTCardBase::_abortConnect",
		   k_CTERROR_API, err, 0,
		   "Unable to abort connect");
  }
  if (result!=0) {
    DBG_NOTICE("Chipcard error %d\n",
	       result);
    return CTError("CTCardBase::_abortConnect",
		   k_CTERROR_INVALID, 0, 0,
		   "Unable to abort connect");
  }

  return CTError();
}













