/* LogJam, a GTK LiveJournal client.
 * Copyright (C) 2000,2001 Evan Martin <evan@livejournal.com>
 * vim:ts=4:sw=4:
 *
 * $Id: menu.c,v 1.8 2002/01/05 14:21:15 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <libhalfgnome/halfgnome.h>

#include "dotconf.h"
#include "network.h"
#include "menu.h"
#include "friends.h"
#include "history.h"
#include "util.h"
#include "gtkspell.h"

static void html_escape_selected(ljwin *ljw);

extern void lj_save_entry(ljwin *ljw);

extern void settings_dialog(GtkWidget *mainwin);
extern void lj_spell_restart(ljwin *ljw);
static void
run_settings_dlg(ljwin *ljw) {
	settings_dialog(ljw->win);
	lj_spell_restart(ljw);
}
static void
run_friends_dlg(ljwin *ljw) {
	friends_dialog(ljw->win);
}
extern void about_dlg(GtkWidget *mainwin);
static void
run_about_dlg(ljwin *ljw) {
	about_dlg(ljw->win);
}
static void
run_history_dlg(ljwin *ljw) {
	history_dialog(ljw->win);
}
static void
run_history_last_dlg(ljwin *ljw) {
	history_load_latest(ljw->win);
}
extern void link_dialog_run(GtkWidget *parent, GtkEditable *gtktext);
static void
run_link_dlg(ljwin *ljw) {
	link_dialog_run(ljw->win, GTK_EDITABLE(ljw->eentry));
}
extern void console_dialog_run(GtkWidget *parent);
static void
run_console_dlg(ljwin *ljw) {
	console_dialog_run(ljw->win);
}

static GtkWidget*
menu_item_from_path(GtkItemFactory *item_factory, char *path) {
	GtkWidget *menu, *item;
	menu = gtk_item_factory_get_widget(item_factory, path);
	item = gtk_menu_get_attach_widget(GTK_MENU(menu));
	return item;
}

GtkWidget* 
menu_make_bar(ljwin *ljw) {
	GtkWidget *bar;
	GtkAccelGroup *accelgroup = NULL;
	GtkItemFactory *item_factory = NULL;

GtkItemFactoryEntry menu_items[] = {
	{ "/_" PROGRAMNAME,                  NULL,         NULL, 0, "<Branch>" },
	{ "/" PROGRAMNAME "/Save As...",     NULL,         lj_save_entry, 0, NULL },
	{ "/" PROGRAMNAME "/sep1",           NULL,         NULL, 0, "<Separator>" },
	{ "/" PROGRAMNAME "/_Settings...",   NULL,         run_settings_dlg, 0, NULL },
	{ "/" PROGRAMNAME "/sep2",           NULL,         NULL, 0, "<Separator>" },
	{ "/" PROGRAMNAME "/_Login",         NULL,         lj_do_loginout, 0, NULL },
	{ "/" PROGRAMNAME "/_Quit",          "<control>Q", gtk_main_quit, 0, NULL },
	
	{ "/_Journal",                       NULL,         NULL, 0, "<Branch>" },
	{ "/Journal/_History Calendar...",   "<control>H", run_history_dlg, 0, NULL },
	{ "/Journal/Edit _Last Entry...",    "<control>L", run_history_last_dlg, 0, NULL },
	{ "/Journal/sep1",                   NULL,         NULL, 0, "<Separator>" },
	{ "/Journal/_Friends...",            "<shift><control>F", run_friends_dlg, 0, NULL },
	{ "/Journal/Post As sep",            NULL,         NULL, 0, "<Separator>" },
	{ "/Journal/_Post As",               NULL,         NULL, 0, "<Branch>" },

	{ "/_Tools",                         NULL,         NULL, 0, "<Branch>" },
	{ "/Tools/_Make Link...",            "<control>M", run_link_dlg, 0, NULL },
	{ "/Tools/_HTML Escape",             "<shift><control>E", html_escape_selected, 0, NULL },			
	{ "/Tools/_LiveJournal Console...",  NULL,         run_console_dlg, 0, NULL },

	{ "/_Web Links",                     NULL,         NULL, 0, "<Branch>" },

	{ "/_Help",                          NULL,         NULL, 0, "<LastBranch>" },
	{ "/Help/_About " PROGRAMNAME "...", NULL,         run_about_dlg, 0, NULL }
};
	int itemcount = sizeof(menu_items) / sizeof(menu_items[0]);

	accelgroup = gtk_accel_group_new();
	item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", accelgroup);
	gtk_item_factory_create_items(item_factory, itemcount, menu_items, ljw);
	gtk_window_add_accel_group(GTK_WINDOW(ljw->win), accelgroup);

	bar = gtk_item_factory_get_widget(item_factory, "<main>");

	ljw->mjournal = menu_item_from_path(item_factory, "/Journal");
	gtk_widget_set_sensitive(ljw->mjournal, FALSE);
	ljw->mweb = menu_item_from_path(item_factory, "/Web Links");
	gtk_widget_set_sensitive(ljw->mweb, FALSE);
	ljw->musejournal = menu_item_from_path(item_factory, "/Journal/Post As");
	ljw->musejournalsep = gtk_item_factory_get_widget(item_factory, "/Journal/Post As sep");
	ljw->mloginout = gtk_item_factory_get_widget(item_factory, "/" PROGRAMNAME "/Login");

	return bar;
}

static void
usejournal_activate(GtkWidget *w, ljwin *ljw) {
	char *user;

	gtk_label_get(GTK_LABEL(GTK_BIN(w)->child), &user);

	if (strcasecmp(user, conf.username) == 0) {
		string_replace(&conf.usejournal, NULL); /* post as normal user. */
	} else {
		string_replace(&conf.usejournal, g_strdup(user));
	}
	lj_update_title(ljw);
}

static void 
menu_update_usejournal(ljwin *ljw, GHashTable *result) {
	GtkWidget *menu;
	GSList *group = NULL;
	GtkWidget *item;

	char *value;
	int i, accesscount = 0;
	char key[30];

	if (result != NULL && (value = g_hash_table_lookup(result, "access_count")) != NULL)
		accesscount = atoi(value);

	if (accesscount == 0) {
		gtk_widget_hide(ljw->musejournal);
		gtk_widget_hide(ljw->musejournalsep);
		ljw->musejournal_defaultjournal = NULL;
		return;
	} 

	/* otherwise, create the access count menu. */
	menu = gtk_menu_new(); 

	item = gtk_radio_menu_item_new_with_label(group, conf.username);
	ljw->musejournal_defaultjournal = item;
	group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(item));
	gtk_signal_connect(GTK_OBJECT(item), "activate",
			GTK_SIGNAL_FUNC(usejournal_activate), ljw);

	/* if they haven't specified a usejournal, default to their username. */
	if (conf.usejournal == NULL) 
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);

	gtk_widget_show(item);
	gtk_menu_append(GTK_MENU(menu), item);

	for (i = 1; i < accesscount+1; i++) {
		char *journal;

		g_snprintf(key, 30, "access_%d", i);
		journal = g_hash_table_lookup(result, key);

		item = gtk_radio_menu_item_new_with_label(group, journal);
		gtk_signal_connect(GTK_OBJECT(item), "activate",
				GTK_SIGNAL_FUNC(usejournal_activate), ljw);
		if (conf.usejournal && strcmp(conf.usejournal, journal) == 0)
			gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);
		group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(item));
		gtk_widget_show(item);
		gtk_menu_append(GTK_MENU(menu), item);
	}

	gtk_menu_item_remove_submenu(GTK_MENU_ITEM(ljw->musejournal));
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(ljw->musejournal), menu);

	gtk_widget_show(ljw->musejournal);
	gtk_widget_show(ljw->musejournalsep);
}

static GtkWidget*
web_recurse(GHashTable *hash, int base) {
	GtkWidget *menu, *item;
	char buf[100];
	char *text, *str, *url, *sub, *dupstr;
	int i, count;

	sprintf(buf, "menu_%d_count", base);
	str = g_hash_table_lookup(hash, buf);
	if (str == NULL) {
		g_warning("menu with no elements?");
		return NULL;
	}
	count = atoi(str);

	menu = gtk_menu_new();

	for (i = 1; i < count+1; i++) {
		sprintf(buf, "menu_%d_%d_text", base, i);
		text = g_hash_table_lookup(hash, buf);
		if (text == NULL) {
			g_warning("menu item has no text?");
			continue;
		}

		sprintf(buf, "menu_%d_%d_url", base, i);
		url = g_hash_table_lookup(hash, buf);
		sprintf(buf, "menu_%d_%d_sub", base, i);
		sub = g_hash_table_lookup(hash, buf);

		if (text[0] == '-' && text[1] == 0) {
			/* seperator */
			item = gtk_menu_item_new();
			gtk_menu_append(GTK_MENU(menu), item);
		} else if (url != NULL) {
			/* url menu item */
			dupstr = g_strdup(url);
			item = gtk_menu_item_new_with_label(text);
			gtk_signal_connect_object(GTK_OBJECT(item), "activate",
				GTK_SIGNAL_FUNC(halfgnome_spawn_url), (GtkObject*)dupstr);
			/* free the dup'd string after the menu item is destroyed! */
			gtk_signal_connect_object(GTK_OBJECT(item), "destroy",
					GTK_SIGNAL_FUNC(g_free), (GtkObject*) dupstr);
			/* that GtkObject* cast is GTK hackery */
			gtk_menu_append(GTK_MENU(menu), item);
		} else if (sub != NULL) {
			/* submenu ! */
			GtkWidget *submenu;
			int subbase;

			subbase = atoi(sub);
			item = gtk_menu_item_new_with_label(text);
			submenu = web_recurse(hash, subbase);
			gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);
			gtk_menu_append(GTK_MENU(menu), item);
		} else {
			g_warning("unknown menu item type...");
		}
	}
	return menu;
}

static void 
menu_update_web(ljwin *ljw, GHashTable *hash) {
	gtk_menu_item_remove_submenu(GTK_MENU_ITEM(ljw->mweb));
	if (hash) {
		gtk_menu_item_set_submenu(GTK_MENU_ITEM(ljw->mweb),
				web_recurse(hash, 0));
		gtk_widget_show_all(ljw->mweb);
		gtk_widget_set_sensitive(ljw->mweb, TRUE);
	} else {
		gtk_widget_set_sensitive(ljw->mweb, FALSE);
	}
}


void
menu_update(ljwin *ljw, GHashTable *result) {
	menu_update_usejournal(ljw, result);
	menu_update_web(ljw, result);
	gtk_label_set_text(GTK_LABEL(GTK_BIN(ljw->mloginout)->child), conf.loginok ? "Logout" : "Login");
	gtk_label_set_pattern(GTK_LABEL(GTK_BIN(ljw->mloginout)->child), "_");
	gtk_widget_set_sensitive(ljw->mjournal, conf.loginok);
	gtk_widget_set_sensitive(ljw->mweb, conf.loginok);
}

typedef struct {
	char c;
	char *escaped;
} htmlescape;
htmlescape htmlescapes[] = {
	{ '<', "&lt;" },
	{ '>', "&gt;" },
	{ '&', "&amp;" },
	/* FIXME: add more entities here; 
	 * the above are all that are really necessary, though. */
	{ 0, 0 }
};

static void
html_escape_selected(ljwin *ljw) {
	guint start, end, pos;
	char c;
	GtkEditable *editable = GTK_EDITABLE(ljw->eentry);
	GtkText *text = GTK_TEXT(ljw->eentry);
	htmlescape *esc;
	int len;

	start = editable->selection_start_pos;
	end = editable->selection_end_pos;
	if (start == end) {
		/* no selection; give them some help. */
		lj_messagebox(ljw->win, "HTML Escape Help",
				"Some characters (such as '<' and '&') are reserved in HTML, and may not display correctly in your post.\n"
				"To properly escape these characters, first select a block of text and try this command again.\n");
		return;
	}
	if (start > end) {
		guint t = end;
		end = start;
		start = t;
	}

	if (conf.usespellcheck) {
		/* we need to turn off gtkspell so it doesn't freak out. */
		gtkspell_detach(text);
	}

	gtk_text_freeze(text);
	for (pos = start; pos < end; pos++) {
		c = GTK_TEXT_INDEX(text, pos);
		for (esc = htmlescapes; esc->c != 0; esc++) {
			if (esc->c == c) break;
		}
		if (esc->c == 0) continue;

		/* if we get here, we have a to-be-escaped character */
		gtk_editable_delete_text(editable, pos, pos+1);
		len = strlen(esc->escaped);
		gtk_editable_insert_text(editable, esc->escaped, len, &pos);
		pos--; /* to counteract the pos++ in the for loop */
		end += len-1; /* subtract one to counteract the removed character */
	}
	gtk_editable_select_region(editable, start, end);
	gtk_text_thaw(text);

	if (conf.usespellcheck) {
		gtkspell_attach(text);
		gtkspell_check_all(text);
	}
}
