/* LogJam, a GTK LiveJournal client.
 * Copyright (C) 2000,2001 Evan Martin <evan@livejournal.com>
 * vim:ts=4:sw=4:
 *
 * $Id: init.c,v 1.4 2001/11/07 02:03:37 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>

#define _GNU_SOURCE
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <locale.h>

#include "network.h"
#include "dotconf.h"
#include "util.h"
#include "login.h"
#include "lj.h"

void print_header(char *name) {
	g_print(PROGRAMNAME " " VERSION "\nCopyright (C) 2000,2001 Evan Martin and Brad Fitzpatrick\n\n");
}

void print_help(char *name) {
#include "helptext.h" /* autogenerated */
	print_header(name);
	g_print(help, name);
#ifndef HAVE_GETOPT_LONG
	g_print("(Long options aren't supported on this system.)\n");
#endif
}

char* readfile(FILE *f) {
	GString *content;
	char buf[1024];
	int len;
	char *ret;

	content = g_string_sized_new(1024);
	while (!feof(f)) {
		len = fread(buf, 1, 1023, f);
		buf[len] = 0;
		g_string_append(content, buf);
	}
	ret = content->str;
	g_string_free(content, FALSE);
	return ret;
}

static int post_cli(char *subject, char *post) {
	GHashTable *request, *response;
	time_t curtime_time_t = time(NULL);
	struct tm *curtime = localtime(&curtime_time_t);

	if (post == NULL) {
		g_print("No post specified!\n");
		return -1;
	}
	g_print("User '%s'", conf.username);
	if (conf.usejournal) 
		g_print(" (posting as '%s')", conf.usejournal);
	g_print(".\n");

	if (conf.password == NULL) {
		char *password;

		if ((password = getpass("Password: ")) != NULL)
			conf.password = g_strdup(password);
	}

	if (conf.password == NULL) {
		g_print("No password specified!\n");
		return -1;
	}

	if (subject == NULL) {
		subject = "";
	} else {
		g_print("Subject '%s'.\n", subject);
	}

	request = net_request_new("postevent");

	g_hash_table_insert(request, g_strdup("subject"), g_strdup(subject));
	g_hash_table_insert(request, g_strdup("event"), g_strdup(post));
	g_hash_table_insert(request, g_strdup("year"), 
			g_strdup_printf("%d", curtime->tm_year+1900));
	g_hash_table_insert(request, g_strdup("mon"), 
			g_strdup_printf("%d", curtime->tm_mon+1));
	g_hash_table_insert(request, g_strdup("day"), 
			g_strdup_printf("%d", curtime->tm_mday));
	g_hash_table_insert(request, g_strdup("hour"), 
			g_strdup_printf("%d", curtime->tm_hour));
	g_hash_table_insert(request, g_strdup("min"), 
			g_strdup_printf("%d", curtime->tm_min));

	response = net_request_run_cli(request);
	hash_destroy(request);
	
	if (net_request_succeeded(response)) {
		g_print("Success.\n");
	} else {
		g_print("Error: %s\n", net_errmsg(response));
	}
	hash_destroy(response);

	return 0;
}

int parse_options(int argc, char* argv[]) {
#include "cmdlineopts_short.h"
#ifdef HAVE_GETOPT_LONG
#include "cmdlineopts_long.h"
#endif
	int c = 0;
	char *subject = NULL;
	char *post = NULL;
	char *username = NULL, *password = NULL, *postas = NULL;

	while (c != -1) {

#ifdef HAVE_GETOPT_LONG
		c = getopt_long(argc, argv, short_options, long_options, NULL);
#else
		c = getopt(argc, argv, short_options);
#endif
		
		switch (c) {
			case 'h':
				print_help(argv[0]);
				return -1;
			case 'v':
				print_header(argv[0]);
				return -1;
			case 'u':
				username = optarg;
				break;
			case 'a':
				postas = optarg;
				break;
			case 'p':
				password = optarg;
				break;
			case 's':
				subject = optarg;
				break;
			case 'f': {
				FILE *f;
				conf.postmode = POSTMODE_CMD;
				if ((f = fopen(optarg, "ra")) == NULL) {
					g_print("Can't open '%s': %s\n", optarg, strerror(errno));
					return -1;
				}
				post = readfile(f);
				fclose(f);
			} 
				break;
			case '?': /* unknown option character. */
			case ':': /* missing option character. */
				return -1;
		}
	}

	if (optind < argc) {
		conf.postmode = POSTMODE_CMD;
		if (strcmp(argv[optind], "-") == 0) {
			post = readfile(stdin);
		} else {
			post = g_strdup(argv[optind]);
		}
	}

	if (username) {
		user *u;

		if (conf.username) g_free(conf.username);
		conf.username = g_strdup(username);

		u = conf_user_by_name(&conf, conf.username);
		if (u) 
			string_replace(&conf.password, g_strdup(u->password));
		else
			string_replace(&conf.password, NULL);

		if (postas) 
			string_replace(&conf.usejournal, g_strdup(postas));
	}
	if (password) {
		string_replace(&conf.password, password);
	}

	if (conf.postmode == POSTMODE_CMD) {
		if (conf.username == NULL) {
			g_print("No username specified!\n");
			return -1;
		}
		if (post == NULL) {
			g_print("No post specified!\n");
			return -1;
		}
		return post_cli(subject, post);
	}

	return 0;
}

int main(int argc, char* argv[]) {
	gboolean has_gtk;

	/*setlocale(LC_ALL, ""); set to the globalish locale, i guess? */
	gtk_set_locale();

	g_thread_init(NULL);
	has_gtk = gtk_init_check(&argc, &argv);

	config_read(&conf);

	signal(SIGCHLD, SIG_IGN);

	if (parse_options(argc, argv) < 0)
		return 0;

	if (conf.postmode == POSTMODE_CMD) {
		return 0;
	} else if (!has_gtk) {
		g_print("GTK init failed -- are you in X?\n\n");
		print_help(argv[0]);
		return -1;
	}

	gdk_threads_enter();
	lj_run();
	gdk_threads_leave();

	config_write(&conf);

	return 0;
}

