package tests::SumTest;

use strict;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::Sum;
use Lire::ReportSpec;
use Lire::Report::Subreport;
use Lire::Report::TableInfo;
use tests::MockAggregator;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_test_schema();

    $self->{'spec'} = new Lire::ReportSpec();
    $self->{'spec'}->superservice( 'test' );
    $self->{'spec'}->id( 'test-sum' );

    $self->_set_up_flat_subreport();
    $self->_set_up_nested_subreport();

    $self->{'cfg'}{'lr_scale_numbers'} = 0;
    $self->{'cfg'}{'lr_scale_bytes'} = 1;
    $self->{'cfg'}{'lr_scale_seconds'} = 1;

    return  $self;
}

sub _set_up_flat_subreport {
    my $self = $_[0];

    my $mock = new tests::MockAggregator( 'report_spec' => $self->{'spec'} );

    $self->{'sum'} = new Lire::Sum( 'report_spec' => $self->{'spec'},
                                    'parent' => $mock,
                                    'field' => 'file_size',
                                    'name' => 'sum' );

    $self->{'weight_sum'} = new Lire::Sum( 'report_spec' => $self->{'spec'},
                                           'parent' => $mock,
                                           'field' => 'file_size',
                                           'weight' => 'file_size',
                                           'name' => 'weight_sum' );

    $self->{'flat_subreport'} = new Lire::Report::Subreport( 'test', 'test' );
    my $info = new Lire::Report::TableInfo();
    $self->{'flat_subreport'}->table_info( $info );
    $info->create_column_info( 'sum', 'numerical', 'bytes' );

    return;
}

sub _set_up_nested_subreport {
    my $self = $_[0];

    my $mock = new tests::MockAggregator( 'report_spec' => $self->{'spec'} );

    $self->{'ratio_sum'} = new Lire::Sum( 'report_spec' => $self->{'spec'},
                                          'parent' => $mock,
                                          'field' => 'file_size',
                                          'ratio' => 'table',
                                          'name' => 'ratio_sum' );

    $self->{'nested_subreport'} =
      new Lire::Report::Subreport(  'test', 'test' );
    my $info = new Lire::Report::TableInfo();
    $self->{'nested_subreport'}->table_info( $info );
    my $group1_info = $info->create_group_info( 'group1' );
    $group1_info->create_column_info( 'ratio_sum', 'numerical', 'bytes' );
    return;
}

sub tear_down {
    my $self = shift->SUPER::tear_down();

    return  $self;
}

sub test_sql_aggr_expr {
    my $self = $_[0];

    $self->{'sum'}{'field'} = 'file-size';

    $self->assert_equals( 'sum("file-size")', $self->{'sum'}->sql_aggr_expr());
}

sub test_sql_aggr_expr_weight {
    my $self = $_[0];

    $self->{'sum'}{'field'} = 'file-size';
    $self->{'sum'}{'weight'} = 'connection-id';

    $self->assert_equals( 'sum("file-size"*"connection-id")',
                          $self->{'sum'}->sql_aggr_expr());
}

sub test_create_value_value {
    my $self = $_[0];

    my $table = $self->{'flat_subreport'};
    my $value = $self->{'sum'}->create_value( $table, { 'sum' => 2048*10,
                                                        '_lr_sum_mc' => 2 });
    $self->assert_deep_equals( { 'content' => '20k',
                                 'value' => 2048*10,
                                 'missing_cases' => 2 }, $value );
}

sub test_create_value_table_ratio_boot {
    my $self = $_[0];

    my $table = $self->{'nested_subreport'};
    $self->{'ratio_sum'}{'ratio'} = 'table';
    my $table_value = 
      $self->{'ratio_sum'}->create_value( $table, { 'ratio_sum' => 100 * 2048,
                                                    '_lr_ratio_sum_mc' => 0 });
    $self->assert_deep_equals( { 'content' => '100.0',
                                 'value' => 2048*100,
                                 'missing_cases' => 0 },
                               $table_value );
}

sub test_create_value_table_ratio {
    my $self = $_[0];

    my $table = $self->{'nested_subreport'};
    $table->set_summary_value( 'ratio_sum', 'value' => 100*2048 );
    my $group1 = $table->create_entry()->create_group();

    $self->{'ratio_sum'}{'ratio'} = 'table';
    my $table_value = 
      $self->{'ratio_sum'}->create_value( $table, { 'ratio_sum' => 100 * 2048,
                                                    '_lr_ratio_sum_mc' => 0 });
    $self->assert_deep_equals( { 'content' => '100.0',
                                 'value' => 2048*100,
                                 'missing_cases' => 0 },
                               $table_value );

    my $group1_value =
      $self->{'ratio_sum'}->create_value( $group1, { 'ratio_sum' =>  5 * 2048,
                                                     '_lr_ratio_sum_mc' => 0});
    $self->assert_deep_equals( { 'content' => '5.0',
                                 'value' => 2048*5,
                                 'missing_cases' => 0 },
                               $group1_value );
}

sub test_create_value_group_ratio {
    my $self = $_[0];

    my $table = $self->{'nested_subreport'};
    $table->set_summary_value( 'ratio_sum', 'value' => 100*2048 );
    my $group1 = $table->create_entry()->create_group();

    $self->{'ratio_sum'}{'ratio'} = 'group';
    my $table_value =
      $self->{'ratio_sum'}->create_value( $table,
                                          { 'ratio_sum' =>  100 * 2048,
                                            '_lr_ratio_sum_mc' => 0 } );
    $self->assert_deep_equals( { 'content' => '100.0',
                                 'value' => 2048*100,
                                 'missing_cases' => 0 },
                               $table_value );

    $group1->set_summary_value( 'ratio_sum', 'value' => 5*2048 );
    my $group1_value =
      $self->{'ratio_sum'}->create_value( $group1,
                                          { 'ratio_sum' =>  1 * 2048,
                                            '_lr_ratio_sum_mc' => 0 } );

    $self->assert_deep_equals( { 'content' => '20.0',
                                 'value' => 1 * 2048,
                                 'missing_cases' => 0 },
                               $group1_value );
}

1;
