#include <stdio.h>
#include <syslog.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <asm/io.h>
#include "lists.h"
#include "../common.h"
#include "srv_def.h"
#include "global_vars.h"
#include "cfg.h"
#include "iface.h"
#include "con.h"
#include "proc.h"
#include "pinger.h"
#include "cmd.h"
#include "cmd_lcp3.h"
#include "proc_supp.h"
#include "lines.h"
#include "html_status.h"
#include "../config.h"

// I wasn't able to find a .h for ioperm(...) and iopl(...)
// but it linked well when declared as extern... ??
#ifdef LPT_MONITOR
	extern int ioperm(unsigned  long  from,  unsigned  long num, int turn_on);
	extern int iopl(int level);
#endif

void sig_handler(int signum)
{
 /* right, this has to be empty! */
}

void cleanup(int sig)
{ // this func may only return with a call to exit()
 char *trmmsg = "terminated on SIG";
 int active = 0;
 struct client_t* clt = (struct client_t*)cltlist.first;
 time_t then = time(NULL) + 10;
 signal(SIGSEGV, SIG_IGN);
 syslog(LOG_DEBUG, "cleanup(): cought signal %d", sig);
 if ( sig == SIGSEGV )
 {
     syslog(LOG_ERR, "SEGFAULT, please report to sfuchs@gmx.net");
     syslog(LOG_INFO, "Version: %s; Value of segf_indicator: %d",
	 		LCS_VERSION_STR, segf_indicator);
 }
 else
     switch ( sig )
     {
         case SIGTERM: syslog(LOG_INFO, "%s %sTERM.", LCS_NAME, trmmsg); break;
         case SIGQUIT: syslog(LOG_INFO, "%s %sQUIT.", LCS_NAME, trmmsg); break;
         case SIGINT: syslog(LOG_INFO, "%s %sINT.", LCS_NAME, trmmsg); break;
         default: syslog(LOG_INFO, "%s selftermination.", LCS_NAME);
     }
 lines_unlock_estab();
 set_offline(NULL);
 /* close lines */
 while ( (active = lines_active_srv()) && (then > time(NULL)) )
 {
     sleep(1);
     proc_con_action(); // check for a status change
 }
 /* if it's still CST_UP_SERVER, notify clients */
 if ( active )
 {
     syslog(LOG_ERR, "%d lines are still active. Close them manually!", active);
 }
 lines_shutdown_log();
 /* notify clients about server termination */
 if ( (sig == SIGQUIT) || (sig == SIGTERM) )
 {
     cmd_broadcast(CBR_TERMED, NULL, 0);
     lcp3_cmd_broadcast(CBR3_SRVDEATH, NULL, 0, NULL);
     while ( clt )
     {
         if ( client_get_sock_type(clt) == SOCK_TYPE_TCP )
	     close(clt->tcp_sock);
	 clt = (struct client_t*)clt->next;
     }
 }
 /* final shutdown */
 pinger_cleanup();
 list_flush(&cltlist);
 list_flush(&ordq);
 list_flush(&lcp3_ordq);
 list_flush(&ackq);
 list_flush(&ipl);
 list_flush(&shtdn_ipl);
 list_flush(&lstlines);
 if ( server->logfile ) fclose(server->logfile);
 closelog();
 close(server->udp_socket);
 exit(0);
}

int init(int argc, char *argv[])
/*
 * set signals
 * initialize lists
 * load config-file
 * initialize server config-/status-record
 * get udp-socket
 */
{
 int retval = 0;
 char *logn = LCS_LOGNAME;
 struct sockaddr_in maddr;
 struct sigaction sact;
 struct line_t *line;
 sigset_t sigs;
 sigemptyset(&sigs);
 sigaddset(&sigs, SIGALRM);
 sigaddset(&sigs, SIGVTALRM);
 sigaddset(&sigs, SIGPROF);
 sigprocmask(SIG_BLOCK, &sigs, NULL);
 signal(SIGINT, cleanup);
 signal(SIGQUIT, cleanup);
 signal(SIGTERM, cleanup);
 signal(SIGSEGV, cleanup);
 sact.sa_handler = SIG_IGN;
 sigemptyset(&sact.sa_mask);
 sact.sa_flags = 0;
 sigaction(SIGCHLD, &sact, NULL); // auto anti zombie
 sigaction(SIGPIPE, &sact, NULL);
 // set the function for html_status printout.
 // external process can call it via SIGUSR1
 sact.sa_handler = &html_status;
 sigaction(SIGUSR1, &sact, NULL);
 openlog(logn, LOG_PID, LOG_DAEMON);
 // read_config will also load defaults!
 /* init lists (to avoid wild pointers...) */
 list_init(&cltlist);
 list_init(&ordq);
 list_init(&ackq);
 list_init(&ipl);
 list_init(&shtdn_ipl);
 list_init(&lstlines);
 /* read the config-file */
 if ( read_config(argc, argv) != 0 ) return(-2);
 /* LineControl Protocol 3.0 needs some initializing */
 lcp3_init();
 /* do we have to initialize the pinger? */
 // if a line needs the pinger, init it once. We don't have
 // to init the pinger for every line...
 line = (struct line_t*)lstlines.first;
 while ( line )
 {
     if ( line->pinger )
     {
         if ( pinger_init() ) return(-1);
	 break;
     }
     line = (struct line_t*)line->next;
 }
 /* get the tcp-socket */
 if ( (server->tcp_socket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1 )
 {	// could not get socket
     fprintf(stderr, "Could not get tcp socket: socket(): %d\n", errno);
     return(-3);
 }
 maddr.sin_family = AF_INET;
 maddr.sin_port = server->port;
 maddr.sin_addr.s_addr = server->bind_to;
 if ( bind(server->tcp_socket, (struct sockaddr*)&maddr, sizeof(struct sockaddr_in)) != 0 )
 {	// could not bind to socket
     fprintf(stderr, "Could not bind tcp socket: bind(): %d\n", errno);
     return(-4);
 }
 if ( listen(server->tcp_socket, 4) )
 {
     close(server->tcp_socket);
     fprintf(stderr, "Unable to listen on tcp socket.\n");
     return(-5);
 }
 fprintf(stdout, "got tcp socket on %s:%d\n", inet_ntoa(maddr.sin_addr), ntohs(maddr.sin_port));
 /* get the udp-socket */
 if ( (server->udp_socket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1 )
 {	// could not get socket
     fprintf(stderr, "Could not get udp socket: socket(): %d\n", errno);
     return(-6);
 }
 if ( bind(server->udp_socket, (struct sockaddr*)&maddr, sizeof(struct sockaddr_in)) != 0 )
 {	// could not bind to socket
     fprintf(stderr, "Could not bind udp socket: bind(): %d\n", errno);
     return(-7);
 }
 fprintf(stdout, "got udp socket on %s:%d\n", inet_ntoa(maddr.sin_addr), ntohs(maddr.sin_port));
 /* check whether the interface is up or not */
 fprintf(stdout, "checking linestatus...\n");
 line = (struct line_t*)lstlines.first;
 while ( line )
 {
     retval = iface_up(server->tcp_socket, line);
     switch (retval) {
         case -1:
		     line->con_stat = CST_DOWN;
		     fprintf(stdout, "   %s: down\n", line->linename);
	    	 break;
         case 0:
		     if ( line->allow_manually )
		     {
	    	     line->con_stat = CST_UP_USER;
				 fprintf(stdout, "   %s: up_user\n", line->linename);
		     }
		     else
	    	 {
	         	line->con_stat = CST_UP_SERVER;
				fprintf(stdout, "   %s: up_server\n", line->linename);
		     }
		     line->con_up_time = time(NULL);
             init_throughput(line);
		     break;
         default:
		     line->con_stat = CST_DOWN;	// should I set an error?
	    	 fprintf(stdout, "   %s: down, error\n", line->linename);
     }
     line = (struct line_t*)line->next;
 }
 /* open/create the logfile */
 if ( server->logfile_name[0] )
     if ( !(server->logfile = fopen(server->logfile_name, "a")) )
         fprintf(stdout, "could not open/create logfile '%s'\n", server->logfile_name);
 /* init the LPT-status feature */
 #ifdef LPT_MONITOR
	if ( server->lpt_status )
	{
		if ( geteuid() )
		{
			fprintf(stdout, "LPT-status feature switched off; you're not root!\n");
		}
		else
		{
			if ( iopl(3) || ioperm(server->lpt_addr, 1, 1) )
			{
				fprintf(stdout, "iopl() or ioperm() failed; unable to use the LPT-status feature!\n");
				server->lpt_status = 0;
				iopl(0);
			}
		}
	}
 #endif // LPT_MONITOR
 return(0);
}
