###
### $Rev: 48 $
### $Release: 0.6.0 $
### copyright(c) 2005 kuwata-lab all rights reserved.
###

require 'kwalify/messages'

module Kwalify

   class KwalifyError < StandardError
   end


   class AssertionError < KwalifyError
      def initialize(msg)
         super("*** assertion error: " + msg)
      end
   end


   class BaseError < KwalifyError
      def initialize(message="", path=nil, value=nil, rule=nil, error_symbol=nil)
         super(message)
         @path        = path
         @rule         = rule
         @value        = value
         @error_symbol = error_symbol
      end
      attr_reader :error_symbol, :rule, :path, :value
      attr_accessor :linenum

      def path
         return @path == '' ? "/" : @path
      end

      alias :_to_s :to_s

      def message
         _to_s
      end

      def to_s
         return "[#{path()}] #{message()}"
      end

      def <=>(ex)
        return @linenum <=> ex.linenum
      end
   end


   class SchemaError < BaseError
      def initialize(message="", path=nil, rule=nil, value=nil, error_symbol=nil)
         super(message, path, rule, value, error_symbol)
      end
   end


   class ValidationError < BaseError
      def initialize(message="", path=nil, rule=nil, value=nil, error_symbol=nil)
         super(message, path, rule, value, error_symbol)
      end
   end


   class YamlSyntaxError < KwalifyError
      def initialize(msg, linenum, error_symbol)
         super("line #{linenum}: #{msg}")
         @linenum = linenum
         @error_symbol
      end
      attr_accessor :linenum, :error_symbol
   end


   module ErrorHelper

      def assert_error(message="")
         raise AssertionError.new(message)
      end

      def validate_error(error_symbol, rule, path, val, args=nil)
         msg = _build_message(error_symbol, val, args);
         return ValidationError.new(msg, path, val, rule, error_symbol)
      end

      def schema_error(error_symbol, rule, path, val, args=nil)
         msg = _build_message(error_symbol, val, args);
         return SchemaError.new(msg, path, val, rule, error_symbol)
      end

      def _build_message(message_key, val, args)
         msg = Kwalify.msg(message_key)
         assert_error("message_key=#{message_key.inspect}") unless msg
         msg = msg % args if args
         msg = "'#{val.to_s.gsub(/\n/, '\n')}': #{msg}" if val != nil && Types.scalar?(val)
         return msg;
      end

   end

   extend ErrorHelper

end
