// =============================================================================
//
//      --- kvi_listview_qt.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviListView"

#include <qbitarray.h>
#include <qcursor.h>
#include <qscrollbar.h>

#include "kvi_fieldeditor.h"
#include "kvi_listview_qt.h"

KviListView::KviListView(QWidget *par, const char *name, bool editable)
	: QListView(par, name)
{
	m_pEditor          =  0;
	m_iColumnCount     =  0;
	m_pCurEditedItem   =  0;
	m_iCurEditedColumn = -1;
	m_resizeToContents = true;
	m_numeric          = new QBitArray(columns());
	m_numeric->fill(0);
	connect(this, SIGNAL(currentChanged(QListViewItem *)),   this, SLOT(slot_currentChanged(QListViewItem *)));
	connect(this, SIGNAL(doubleClicked(QListViewItem *)),    this, SLOT(slot_doubleClicked(QListViewItem *)));
	connect(this, SIGNAL(selectionChanged(QListViewItem *)), this, SLOT(slot_selectionChanged(QListViewItem *)));
	connect(
		this,    SIGNAL(rightButtonPressed(QListViewItem *, const QPoint &, int)),
		this, SLOT(slot_rightButtonPressed(QListViewItem *, const QPoint &, int))
	);
	setFrameStyle(QFrame::WinPanel | QFrame::Sunken);
	setAllColumnsShowFocus(true);
	setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Preferred);

	if( parent() ) {
		QWidget *w = (QWidget *) parent();
		while( w ) {
			w->installEventFilter(this);
			w = (QWidget *) w->parent();
		}
	}
	if( editable )
		setFieldEditor(new KviFieldEditor(viewport()));
}

KviListView::~KviListView()
{
	if( m_numeric ) {
		delete m_numeric;
		m_numeric = 0;
	}
}

void KviListView::setFieldEditor(KviFieldEditor *editor)
{
	if( !editor ) return;

	m_pEditor = editor;
	connect(this, SIGNAL(doubleClicked(KviListViewItem *)), this, SLOT(itemDoubleClicked(KviListViewItem *)));
	connect(m_pEditor, SIGNAL(editFinished(const QString &)), this, SLOT(editTerminated(const QString &)));
	connect(m_pEditor, SIGNAL(keyUpPressed()),                this, SLOT(editorKeyUp()));
	connect(m_pEditor, SIGNAL(keyDownPressed()),              this, SLOT(editorKeyDown()));
	connect(m_pEditor, SIGNAL(keyLeftPressed()),              this, SLOT(editorKeyLeft()));
	connect(m_pEditor, SIGNAL(keyRightPressed()),             this, SLOT(editorKeyRight()));
}

QSize KviListView::sizeHint() const
{
	QSize hint = QListView::sizeHint();
	if( m_resizeToContents ) {
		int width = contentsWidth() + (frameWidth() << 1);
		QScrollBar *sb = verticalScrollBar();
		if( sb->isVisible() )
			width += sb->width();
		hint = hint.boundedTo(QSize(width, QWIDGETSIZE_MAX));
	}
	return hint;
}

void KviListView::setAutoResize(const bool resize)
{
	m_resizeToContents = resize;
	if( !m_resizeToContents )
		setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred);
}

int KviListView::addColumn(const QString &label, bool isNumeric, int width)
{
	m_iColumnCount++;
	m_numeric->resize(m_iColumnCount);
	int col = QListView::addColumn(label, width);
	m_numeric->setBit(col, isNumeric);
	if( isNumeric )
		setColumnAlignment(col, Qt::AlignRight);
	return col;
}

void KviListView::removeColumn(int col)
{
	m_iColumnCount--;
	QListView::removeColumn(col);
}

void KviListView::itemDoubleClicked(KviListViewItem *it)
{
	if( !it ) return;
	QPoint pnt = QCursor::pos();
	pnt = viewport()->mapFromGlobal(pnt);

	int col   = 0;
	int xLeft = -contentsX();
	while( col < m_iColumnCount ) {
		xLeft += columnWidth(col);
		if( pnt.x() < xLeft ) {
			editField(it, col);
			return;
		}
		col++;
	}
}

void KviListView::editField(KviListViewItem *it, int column, bool bEnd)
{
	if( !m_pEditor ) return;

	QRect rct        = itemRect(it);
	int   fieldWidth = columnWidth(column);
	int   fieldX     = -contentsX();
	for( int i = 0; i < column; i++ )
		fieldX += columnWidth(i);
	m_pEditor->move(fieldX - 1, rct.top() - 1);
	m_pEditor->resize(fieldWidth + 2, rct.height() + 2);
	m_pCurEditedItem   = it;
	m_iCurEditedColumn = column;
	m_pEditor->edit(it->text(column), bEnd);
}

void KviListView::editTerminated(const QString &text)
{
	if( m_pCurEditedItem && (m_iCurEditedColumn > -1) && (m_iCurEditedColumn < m_iColumnCount) )
		m_pCurEditedItem->setText(m_iCurEditedColumn, text);
	m_pCurEditedItem   = 0;
	m_iCurEditedColumn = -1;
	setFocus();
}

bool KviListView::eventFilter(QObject *o, QEvent *e)
{
	if( m_pEditor && m_pEditor->isVisible() ) {
		if( (e->type() == QEvent::Accel) || (e->type() == QEvent::KeyPress) ) {
			if( ((QKeyEvent *) e)->key() != Qt::Key_Tab )
				m_pEditor->keyPressEvent((QKeyEvent *) e);
			else
				editorKeyRight();
			((QKeyEvent *) e)->accept();
			return true;
		}
	}
	return QListView::eventFilter(o, e);
}

void KviListView::editorKeyLeft()
{
	if( !m_pEditor ) return;

	// Accept the previous changes...
	KviListViewItem *i = m_pCurEditedItem;
	int idx = m_iCurEditedColumn;
	m_pEditor->terminateEdit(true);
	// And edit the next field
	--idx;
	if( idx >= 0 )
		editField(i, idx);
	else
		editField(i, m_iColumnCount - 1);
}

void KviListView::editorKeyRight()
{
	if( !m_pEditor ) return;

	// Accept the previous changes...
	KviListViewItem *i = m_pCurEditedItem;
	int idx = m_iCurEditedColumn;
	m_pEditor->terminateEdit(true);
	// And edit the next field
	++idx;
	if( idx < m_iColumnCount )
		editField(i, idx, false);
	else
		editField(i, 0, false);
}

void KviListView::editorKeyUp()
{
	if( !m_pEditor ) return;

	KviListViewItem *i = m_pCurEditedItem ? m_pCurEditedItem->itemAbove() : 0;
	if( !i ) return;
	int idx = m_iCurEditedColumn;
	m_pEditor->terminateEdit(true);
	// And edit the next field
	editField(i, idx);
}

void KviListView::editorKeyDown()
{
	if( !m_pEditor ) return;

	KviListViewItem *i = m_pCurEditedItem ? m_pCurEditedItem->itemBelow() : 0;
	if( !i ) return;
	int idx = m_iCurEditedColumn;
	m_pEditor->terminateEdit(true);
	// And edit the next field
	editField(i, idx);
}

KviListViewItem *KviListView::firstChild()
{
	return (KviListViewItem *) QListView::firstChild();
}

KviListViewItem *KviListView::itemAt(const QPoint &viewPos) const
{
	return (KviListViewItem *) QListView::itemAt(viewPos);
}

KviListViewItem *KviListView::selectedItem()
{
	return (KviListViewItem *) QListView::selectedItem();
}

void KviListView::slot_currentChanged(QListViewItem *item)
{
	emit currentChanged((KviListViewItem *) item);
}

void KviListView::slot_doubleClicked(QListViewItem *item)
{
	emit doubleClicked((KviListViewItem *) item);
}

void KviListView::slot_selectionChanged(QListViewItem *item)
{
	emit selectionChanged((KviListViewItem *) item);
}

void KviListView::slot_rightButtonPressed(QListViewItem *item, const QPoint &point, int col)
{
	emit rightButtonPressed((KviListViewItem *) item, point, col);
}

bool KviListView::isNumeric(int col) const
{
	return m_numeric->testBit(col);
}

/*
 * ======================================================================
 */
KviListViewItem::KviListViewItem(KviListView *parent)
	: QListViewItem(parent)
{
	// Nothing here
}

KviListViewItem::KviListViewItem(
	KviListView *parent, QString l1, QString l2, QString l3, QString l4, QString l5, QString l6, QString l7, QString l8)
	: QListViewItem(parent, l1, l2, l3, l4, l5, l6, l7, l8)
{
	// Nothing here
}

KviListViewItem::KviListViewItem(
	KviListView *parent, KviListViewItem *after, QString l1, QString l2,
	QString l3, QString l4, QString l5, QString l6, QString l7, QString l8)
	: QListViewItem(parent, after, l1, l2, l3, l4, l5, l6, l7, l8)
{
	// Nothing here
}

KviListViewItem::KviListViewItem(
	KviListViewItem *parent, QString l1, QString l2, QString l3, QString l4, QString l5, QString l6, QString l7, QString l8)
	: QListViewItem(parent, l1, l2, l3, l4, l5, l6, l7, l8)
{
	// Nothing here
}

KviListViewItem::KviListViewItem(
	KviListViewItem *parent, KviListViewItem *after, QString l1, QString l2,
	QString l3, QString l4, QString l5, QString l6, QString l7, QString l8)
	: QListViewItem(parent, after, l1, l2, l3, l4, l5, l6, l7, l8)
{
	// Nothing here
}

KviListViewItem::~KviListViewItem()
{
	// Nothing here
}

KviListViewItem *KviListViewItem::firstChild()
{
	return (KviListViewItem *) QListViewItem::firstChild();
}

KviListViewItem *KviListViewItem::itemAbove()
{
	return (KviListViewItem *) QListViewItem::itemAbove();
}

KviListViewItem *KviListViewItem::itemBelow()
{
	return (KviListViewItem *) QListViewItem::itemBelow();
}

KviListViewItem *KviListViewItem::nextSibling()
{
	return (KviListViewItem *) QListViewItem::nextSibling();
}

KviListViewItem *KviListViewItem::parent()
{
	return (KviListViewItem *) QListViewItem::parent();
}

int KviListViewItem::compare(QListViewItem *it, int col, bool ascending) const
{
	KviListView *lv = (KviListView *) listView();
	if( !lv->isNumeric(col) )
		return QListViewItem::compare(it, col, ascending);
	bool bOk = false;
	float val1, val2;
	val1 = key(col, ascending).toFloat(&bOk);
	if( bOk )
		val2 = it->key(col, ascending).toFloat(&bOk);
	if( !bOk )
		return QListViewItem::compare(it, col, ascending);
	if( val1 < val2 ) return -1;
	if( val1 > val2 ) return  1;
	return 0;
}

/*
 * ======================================================================
 */

KviListViewItemIterator::KviListViewItemIterator(KviListView *parent)
	: QListViewItemIterator(parent)
{
	// Nothing here
}

KviListViewItem *KviListViewItemIterator::current()
{
	return (KviListViewItem *) QListViewItemIterator::current();
}

#include "m_kvi_listview_qt.moc"
