// =============================================================================
//
//      --- kvi_systray.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviSysTray"

#include <qiconset.h>
#include <qpainter.h>
#include <qtooltip.h>

#include "kvi_debug.h"
#include "kvi_frame.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_systray.h"
#include "kvi_systray_bar.h"
#include "kvi_systray_clock.h"
#include "kvi_systray_icondisplay.h"
#include "kvi_systray_ioled.h"
#include "kvi_systray_onlinetimer.h"
#include "kvi_systray_utcclock.h"
#include "kvi_systray_widget.h"

/*
	@quickhelp: KviSysTray
	@widget: System display
		This is the system display.<br>
		It can act as a clock, a GMT clock, an Online timer or an I/O Led.<br>
		By clicking in the left corner of this window you will cyclically
		switch between off and these four modes.<br>
		<br>
		Position 1 : Off<br>
		Position 2 : Local time clock<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Should be in sync with your system clock.<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Shows the current date and time.<br>
		&nbsp;&nbsp;&nbsp;&nbsp;If you see three letters (DST) on the right side<br>
		&nbsp;&nbsp;&nbsp;&nbsp;that means that the daylight saving time is in effect.<br>
		Position 3 : GMT clock<br>
		&nbsp;&nbsp;&nbsp;&nbsp;If your timezone is set correctly<br>
		&nbsp;&nbsp;&nbsp;&nbsp;this clock shows the current Universal Coordinated Time and date.<br>
		Position 4 : Online timer<br>
		&nbsp;&nbsp;&nbsp;&nbsp;This one works only when you are connected to a server.<br>
		&nbsp;&nbsp;&nbsp;&nbsp;It shows the time that the connection started and the<br>
		&nbsp;&nbsp;&nbsp;&nbsp;duration.<br>
		Position 5 : I/O Led<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Input - Output "main socket traffic".<br>
		&nbsp;&nbsp;&nbsp;&nbsp;The upper indicator show the received packets of data, <br>
		&nbsp;&nbsp;&nbsp;&nbsp;and the lower the sent ones.<br>
		<br>
		Other modes can be added by plugins.<br>
*/
KviSysTray::KviSysTray(QWidget *parent)
	: QWidget(parent, "systray")
{
	setBackgroundMode(NoBackground);
	m_pIconDisplay = new KviSysTrayIconDisplay(this);

	m_pWidgetList = new QPtrList<KviSysTrayWidget>;
	m_pWidgetList->setAutoDelete(true);

	m_pWidgetList->append(new KviSysTrayClock(this));
	m_pWidgetList->append(new KviSysTrayUtcClock(this));
	m_pWidgetList->append(new KviSysTrayOnLineTimer(this));
	m_pWidgetList->append(new KviSysTrayIoLed(this));

	m_pCurrentWidget = findSysTrayWidget(g_pOptions->m_szSysTrayMode.ptr());
	if( !m_pCurrentWidget ) {
		m_pCurrentWidget = findSysTrayWidget("KviSysTrayIoLed");
		__range_valid(m_pCurrentWidget);
		g_pOptions->m_szSysTrayMode = "KviSysTrayIoLed";
	}

	m_pCurrentWidget->show();
	m_pIconDisplay->setIcon(m_pCurrentWidget->icon());
	setMinimumSize(QSize(180, 20));
	m_pMemBuffer = new QPixmap(width(), height());
}

KviSysTray::~KviSysTray()
{
	delete m_pWidgetList; m_pWidgetList = 0;
	delete m_pMemBuffer;  m_pMemBuffer  = 0;
}

QSize KviSysTray::sizeHint()
{
	return QSize(minimumSize().height() + 180, minimumSize().height());
}

void KviSysTray::resizeEvent(QResizeEvent *e)
{
	m_pIconDisplay->setGeometry(0, 0, height(), height());
	m_pCurrentWidget->setGeometry(height(), 0, width() - height(), height());
	m_pMemBuffer->resize(width(), height());
	QWidget::resizeEvent(e);
}

void KviSysTray::addPluginWidget(KviSysTrayWidget *w, bool bShow)
{
	m_pWidgetList->append(w);
	if( bShow )
		setCurrentWidget(w);
}

bool KviSysTray::setCurrentWidget(KviSysTrayWidget *w)
{
	if( m_pCurrentWidget == w )
		return true; // Already current

	for( KviSysTrayWidget *wdg = m_pWidgetList->first(); wdg; wdg = m_pWidgetList->next() ) {
		if( wdg == w ) {
			if( m_pCurrentWidget )
				m_pCurrentWidget->hide();
			m_pCurrentWidget = w;
			w->setGeometry(height(), 0, width() - height(), height());
			w->raise();
			w->show();
			m_pIconDisplay->setIcon(w->icon());
			g_pOptions->m_szSysTrayMode = w->className();
			return true;
		}
	}
	return false; // Not found!
}

void KviSysTray::removeWidget(KviSysTrayWidget *w, bool bDelete)
{
	m_pWidgetList->setAutoDelete(bDelete);
	if( w == m_pCurrentWidget )
		rotateMode();
	if( w == m_pCurrentWidget )
		m_pCurrentWidget = 0;
	m_pWidgetList->removeRef(w);
	m_pWidgetList->setAutoDelete(true);
}

void KviSysTray::killPluginWidgets(void *plugin_handle)
{
	QPtrList<KviSysTrayWidget> list;
	list.setAutoDelete(false);
	for( KviSysTrayWidget *w = m_pWidgetList->first(); w; w = m_pWidgetList->next() ) {
		if( w->m_plugin_handle == plugin_handle )
			list.append(w);
	}
	for( KviSysTrayWidget *w = list.first(); w; w = list.next() ) {
		// autoDelete is true
		if( w == m_pCurrentWidget )
			rotateMode();
		if( w == m_pCurrentWidget )
			m_pCurrentWidget = 0;
		m_pWidgetList->removeRef(w);
	}
}

KviSysTrayWidget *KviSysTray::currentWidget()
{
	return m_pCurrentWidget;
}

void KviSysTray::rotateMode()
{
	if( m_pWidgetList->count() <= 1 )
		return;
	for( KviSysTrayWidget *w = m_pWidgetList->first(); w; w = m_pWidgetList->next() ) {
		if( w == m_pCurrentWidget ) {
			w = m_pWidgetList->next();
			if( !w )
				w = m_pWidgetList->first();
			__range_valid(w);
			m_pCurrentWidget->hide();
			m_pCurrentWidget = w;
			m_pIconDisplay->setIcon(w->icon());
			w->setGeometry(height(), 0, width() - height(), height());
			w->raise();
			w->show();
			g_pOptions->m_szSysTrayMode = w->className();
			return;
		}
	}
}

QColor KviSysTray::getBackground()
{
	return g_pOptions->m_clrSysTrayBack;
}

QColor KviSysTray::getForeground()
{
	return g_pOptions->m_clrSysTrayFore;
}

KviSysTrayWidget *KviSysTray::findSysTrayWidget(const char *classname)
{
	for( KviSysTrayWidget *w = m_pWidgetList->first(); w; w = m_pWidgetList->next() ) {
		KviStr tmp = w->className();
		if( kvi_strEqualCI(tmp.ptr(), classname) )
			return w;
	}
	return 0;
}

#include "m_kvi_systray.moc"
