// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptCheckBox"

#include <qcheckbox.h>

#include "kvi_error.h"
#include "kvi_script_checkbox.h"
#include "kvi_script_objectclassdefinition.h"

/*
	@class: checkbox
	@short:
		A checkbox widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $text()
		Returns the text displayed on the check box.<br>

		!fn: $setText(&lt;text&gt;)
		Sets the text to be displayed on the check box.<br>

		!fn: $isChecked()
		Returns '1' if the check box is in the checked state, '0' otherwise.<br>

		!fn: $setChecked(&lt;bChecked&gt;)
		Sets the check state of the check box.<br>
		If &lt;bChecked&gt; is 0 (false) the check box is unchecked, otherwise
		it is put into checked state.<br>

	@events:
		!ev: OnToggle($1 = &lt;bChecked&gt;)
		Triggered when the user clicks the check box and changes its state.<br>
		&lt;bChecked&gt; is '1' if the new state is "checked", '0' otherwise.<br>
		The default event handler emits the [classsignal:checkbox]toggled[/classsignal] signal.<br>
		If you provide a new handler for this event, and still want the signal
		to be emitted, you must emit it by yourself:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:checkbox]toggled[/classsignal], $1)
		</example>

	@signals:
		!sg: toggled($1 = &lt;bChecked&gt;)
		Emitted when the box has been clicked by the user and its state has changed.<br>
		The first parameter is the new state of the checkbox : '1' if "checked", '0' if not.<br>
		<b>This signal is emitted by the default [classevent:checkbox]OnToggle[/classevent] event handler!<br>
		If you define a new event handler, and still want the signal to be emitted,
		you have to emit it by yourself!

	@description:
		A checkable box widget : useful in implementing boolean option choices for the user.<br>
		It contains a little button that has a check mark if the widget is in checked state.<br>
		A small text label is displayed on the right of the button.<br>
		The constructor for this object class accepts an additional parameter:
		the text to be displayed.<br>

	@examples:

	@seealso:
		class [class]object[/class], <br>
		class [class]widget[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * CHECKBOX class
 */
void KviScriptCheckBox::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText",    (scriptObjectFunction) &KviScriptCheckBox::builtinFunction_SETTEXT);
	d->addBuiltinFunction("text",       (scriptObjectFunction) &KviScriptCheckBox::builtinFunction_TEXT);
	d->addBuiltinFunction("isChecked",  (scriptObjectFunction) &KviScriptCheckBox::builtinFunction_ISCHECKED);
	d->addBuiltinFunction("setChecked", (scriptObjectFunction) &KviScriptCheckBox::builtinFunction_SETCHECKED);
	KviScriptEventStruct *s = new KviScriptEventStruct();
	s->szName   = "OnToggle";
	s->szBuffer = "$this->$emit(toggled, $1)";
	d->addDefaultEvent(s);
}

KviScriptCheckBox::KviScriptCheckBox(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	// Nothing here
}

KviScriptCheckBox::~KviScriptCheckBox()
{
	// Nothing here
}

bool KviScriptCheckBox::init(QPtrList<KviStr> *params)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new QCheckBox(((KviScriptWidget *) parent())->m_pWidget, name());
		}
	}
	if( !m_pWidget )
		m_pWidget = new QCheckBox(0, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()),   this, SLOT(widgetDestroyed()));
	connect(m_pWidget, SIGNAL(toggled(bool)), this, SLOT(boxToggled(bool)));
	if( params ) {
		KviStr *pS = params->first();
		if( pS )
			((QCheckBox *) m_pWidget)->setText(pS->ptr());
	}
	return true;
}

void KviScriptCheckBox::boxToggled(bool bChecked)
{
	KviStr params = bChecked ? '1' : '0';
	triggerEvent("OnToggle", params);
}

int KviScriptCheckBox::builtinFunction_TEXT(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(((QCheckBox *) m_pWidget)->text());
	return KVI_ERROR_Success;
}

int KviScriptCheckBox::builtinFunction_SETTEXT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			((QCheckBox *) m_pWidget)->setText(pS->ptr());
			return KVI_ERROR_Success;
		}
	}
	((QCheckBox *) m_pWidget)->setText("");
	return KVI_ERROR_Success;
}

int KviScriptCheckBox::builtinFunction_ISCHECKED(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(((QCheckBox *) m_pWidget)->isChecked() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptCheckBox::builtinFunction_SETCHECKED(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			((QCheckBox *) m_pWidget)->setChecked(!kvi_strEqualCI(pS->ptr(), "0"));
			return KVI_ERROR_Success;
		}
	}
	((QCheckBox *) m_pWidget)->setChecked(true);
	return KVI_ERROR_Success;
}

#include "m_kvi_script_checkbox.moc"
