//
//   SPEAK plugin for KVirc 1.1 Phoenix.
//
//   This plugin enables you to use IBM's ViaVoiceOutloud to speak
//   messages to you.  Messages that are directed to you, over the
//   Relay Chat, can be spoken outloud or sentences spoken by specific
//   people can also heard outloud.
//
//   To use the plugin, one needs to have a working IBM ViaVoice
//   Outloud, which can be retrieved from IBM's site.  For information
//   about IBM ViaVoice http://www-4.ibm.com/software/speech/dev/
//
//   Copyright (C) rn E. Hansen (oe.hansen@gamma.telenordia.se)
//   Copyright (C) Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#define __KVIRC_PLUGIN__

#include "libkvispeak.h"
#include "kvirc_plugin.h"

#include <dlfcn.h>

#include "kvi_process.h"
#include "kvi_fileutils.h"

extern "C" {

# include <stdlib.h>

};

#include <qlist.h>
#include <qfile.h>
#include <qlistview.h>
#include <qfiledialog.h>
#include <qtextstream.h>
#include <qmessagebox.h>

#include <kvi_string.h>
#include <kvi_locale.h>

#define KVI_XMMS_UPDATE_TIME 500

#define SEP_CHAR             char(1)

static void  *libibmeci50_handle = 0;

static void * (*libibmeci50_eciNew)()                        = 0;
static void   (*libibmeci50_eciSetOutputDevice)(void *,int)  = 0;
static void   (*libibmeci50_eciSetParam)(void *,int,int)     = 0;
static void   (*libibmeci50_eciDelete)(void *)               = 0;
static void   (*libibmeci50_eciSynthesize)(void *)           = 0;
static void   (*libibmeci50_eciAddText)(void *,const char *) = 0;
static void * hECI = 0;

static  bool                            eci_speak2me = true;
static  QList<struct eci_user>          eci_users;
static  smileyClass                     eci_smileyEdit;
static  BasicTree                      *smileyTree   = 0;

static  Voice                           defVoices[] = {
                                          {0,0,0,0,0,0,0,0},
                                          {0,50,50,50,50,50,80,0},
                                          {1,50,65,50,50,50,80,40},
					  {0,20,80,20,20,50,80,0},
					  {0,80,50,50,50,50,80,0},
					  {0,50,50,50,50,70,80,0},
					  {0,50,50,50,50,70,80,40},
					  {1,45,65,50,30,50,80,40},
					  {0,50,50,50,80,50,80,0},
					  {0,50,50,50,50,50,80,50},
                                        };

smileyNode::smileyNode(const QString& p_str,const QString& p_txt,bool p_cs)
{
  _key       = p_str.latin1();
  _english   = p_txt.latin1();
  _caseSense = p_cs;
}

smileyNode::smileyNode(const KviStr& p_str,const KviStr& p_txt, bool p_cs)
{
  _key       = p_str.ptr();
  _english   = p_txt.ptr();
  _caseSense = p_cs;
}

smileyNode::~smileyNode()
{
}

int smileyNode::Compare(void *p_ref)
{
  if ( _caseSense )
    return kvi_strcmpCI( (const char *)p_ref,_key.ptr() );
  return kvi_strcmpCS( (const char *)p_ref,_key.ptr() );
}

int smileyNode::Compare(const KviStr& p_str)
{
  if ( _caseSense )
    return kvi_strcmpCI( p_str.ptr(),_key.ptr() );
  return kvi_strcmpCS( p_str.ptr(),_key.ptr() );
}

smileyClass::smileyClass(QWidget* parent,const char* name)
 : smileyClassData( parent, name )
{
  _smileys_loaded = false;
  hide();
}

smileyClass::~smileyClass()
{
}

void smileyClass::init()
{
  if ( _smileys_loaded == false ) {
    KviStr s;

    if ( kvirc_plugin_get_read_only_config_file_path( s,"speak" ) )
      eci_smileyEdit.load(s);
    _smileys_loaded = true;
  }
}

void smileyClass::insertSmileys(QListView *p_view, smileyNode *p_ref = 0)
{
  QListViewItem *item;
  if ( p_ref == 0 )
    return;
  item = new QListViewItem(p_view,p_ref->Key().ptr());
  item->setText(2,p_ref->English().ptr());
  item->setText(1,QString("%1").arg((int)p_ref->CaseSense()));
  insertSmileys(p_view,(smileyNode *)p_ref->Left());
  insertSmileys(p_view,(smileyNode *)p_ref->Right());
}

void smileyClass::editSmileys()
{
  m_pShortHands->clear();
  if ( smileyTree )
    insertSmileys(m_pShortHands,(smileyNode *)smileyTree->Head());
  show();
}

void smileyClass::load(KviStr& p_fileName)
{
  KviStr entry;
  QString str,speak_text;
//  QListViewItem *item;
  bool sensation;
  int count;

  if ( kvi_fileExists(p_fileName.ptr()) ) {
    KviConfig cfg( p_fileName.ptr() );

    m_pShortHands->clear();
    cfg.setGroup("Smileys");
    count = cfg.readIntEntry("NSmileys",0);
    while( count-- ) {
      entry.sprintf("smileyData%d",count);
      str = cfg.readEntry(entry.ptr());
      entry.sprintf("smileyCase%d",count);
      sensation = QString(cfg.readEntry(entry.ptr())) ==  "1";
      entry.sprintf("smileyText%d",count);
      speak_text = cfg.readEntry(entry.ptr());
      if ( str.isEmpty() == false )
	new QListViewItem(m_pShortHands,str,QString("%1").arg(sensation?"1":"0"),speak_text);
    }
    if ( isVisible() == false )
      okPressed();
  }
}

void smileyClass::loadBase()
{
  KviStr path;
  KviStr fileName;

  kvirc_plugin_get_config_file_path( path,"speak" );
  fileName = QFileDialog::getOpenFileName( path.ptr(),"Config (*.conf)",this);
  if ( fileName.isEmpty() == false )
    load( fileName );
}

void smileyClass::insertItem()
{
  QListViewItem *item;

  if ( m_pShortText->text() == "" || m_pSpokenText->text() == "" )
    return;
  QListViewItemIterator it( m_pShortHands );
  for( ;(item = it.current());++it )
    if ( kvi_strEqualCI(item->text(0).latin1(),m_pShortText->text().latin1()) )
      break;
  if ( item == 0 )
    item = new QListViewItem( m_pShortHands,m_pShortText->text() );
  item->setText( 1,QString("%1").arg((int)(m_pCaseSensation->isChecked())));
  item->setText( 2,m_pSpokenText->text() );
}

void smileyClass::removeItem()
{
  QListViewItem *item = m_pShortHands->selectedItem();

  if ( item ) {
    m_pShortText->setText( "" );
    m_pSpokenText->setText( "" );
    delete item;
  }
}

void smileyClass::okPressed()
{
  QListViewItem *item;

  hide();
  if ( smileyTree )
    delete smileyTree;
  smileyTree = new BasicTree;
  QListViewItemIterator it( m_pShortHands );
  for( ;(item = it.current());++it )
    smileyTree->Insert( new smileyNode(item->text(0),item->text(2),item->text(1)=="1") );
}

void smileyClass::updateItem()
{
  QListViewItem *item = m_pShortHands->selectedItem();

  if ( item == 0 || m_pSpokenText->text() == "" )
    return;
  item->setText(1,QString("%1").arg((int)(m_pCaseSensation->isChecked())));
  item->setText(2,m_pSpokenText->text());
}

void smileyClass::saveBaseAs()
{
  KviStr path;
  QString fileName;
  QListViewItem *item;

  kvirc_plugin_get_config_file_path( path,"speak" );
  fileName = QFileDialog::getSaveFileName( path.ptr(),"Config (*.conf)",this );
  if ( !fileName.isNull() ) {
    if ( kvi_fileExists(fileName.latin1()) )
      if ( QMessageBox::warning( this,__tr("File exists"),__tr("Do you want to overwrite the file?"),__tr("No"),__tr("Yes")) == 0 )
	return;
    KviConfig cfg( fileName.latin1() );
    KviStr entry, str;
    int n;

    cfg.setGroup("Smileys");
    cfg.writeEntry("NSmileys",m_pShortHands->childCount());
    cfg.sync();
    QListViewItemIterator it( m_pShortHands );

    for( n=0;(item = it.current());it++,n++ ) {
      entry.sprintf("smileyData%d",n);
      cfg.writeEntry(entry.ptr(),item->text(0).latin1());
      entry.sprintf("smileyCase%d",n);
      cfg.writeEntry(entry.ptr(),item->text(1).latin1());
      entry.sprintf("smileyText%d",n);
      cfg.writeEntry(entry.ptr(),item->text(2).latin1());
    }
  }
}

void smileyClass::selectShortHand(QListViewItem *p_item)
{
  if ( p_item == 0 )
    return;
  m_pShortText->setText( p_item->text(0) );
  m_pCaseSensation->setChecked( p_item->text(1) == "1" );
  m_pSpokenText->setText( p_item->text(2) );
}

eciVoiceSetup::eciVoiceSetup(eci_user *p_user,
			     QWidget *p_parent, 
			     const char *p_name)
  : eciSetupData(p_parent,p_name)
{
  setup( p_user );
  show();
}

eciVoiceSetup::~eciVoiceSetup()
{
}

void eciVoiceSetup::okPressed()
{
  eci_user *user;

  if ( _user ) {
    _user->voice = buildVoice();
    if ( eci_users.find(_user) == -1 ) {
      for( user=eci_users.first();user;user=eci_users.next() )
	if ( kvi_strEqualCI(user->name.ptr(),_user->name.ptr()) )
	  break;
      if ( user )
	user->voice = _user->voice;
      else
	eci_users.append( _user );
    }
  }
  delete this;
}

void eciVoiceSetup::cancelPressed()
{
  if ( _user && eci_users.find(_user) == -1 )
    delete _user;
  delete this;
}

KviStr eciVoiceSetup::buildVoice()
{
  KviStr s,t;
  int i,n=0;

  s = "";
  for( i=0;i<8;i++ ) {
    t = "";
    switch(i) {
    case 0:
      if ( m_pGender->isChecked() != defVoices[_index].s_gender )
	t.sprintf("`vg%d",m_pGender->isChecked());
      break;
    case 1:
      if ( m_pPitchbase->value() != defVoices[_index].s_pitch_base )
	t.sprintf("`vb%d",m_pPitchbase->value());
      break;
    case 2:
      if ( m_pHeadsize->value() != defVoices[_index].s_headsize )
	t.sprintf("`vh%d",m_pHeadsize->value());
      break;
    case 3:
      if ( m_pRoughness->value() != defVoices[_index].s_roughness )
	t.sprintf("`vr%d",m_pRoughness->value());
      break;
    case 4:
      if ( m_pBreathiness->value() != defVoices[_index].s_breathy )
	t.sprintf("`vy%d",m_pBreathiness->value());
      break;
    case 5:
      if ( m_pPitchrange->value() != defVoices[_index].s_pitch_fluc )
	t.sprintf("`vf%d",m_pPitchrange->value());
      break;
    case 6:
      if ( m_pSpeed->value() != defVoices[_index].s_speed )
	t.sprintf("`vs%d",m_pSpeed->value());
      break;
    case 7:
      if ( m_pVolume->value() != defVoices[_index].s_volume )
	t.sprintf("`vv%d",m_pVolume->value());
      break;
    }
    if ( t.hasData() )
      s+= ' ' + t,n++;
  }
  if ( n < 8 ) {
    t.sprintf("`v%d ",_index);
    s.prepend(t);
  }
  debug( s.ptr() );
  return s;
}

void eciVoiceSetup::testPressed()
{
  KviStr s = buildVoice();

  s += " Spoken outloud by IBM's Via Voice speech library.";
  libibmeci50_eciAddText(hECI,s.ptr());
  libibmeci50_eciSynthesize(hECI);
}

void eciVoiceSetup::showAnnotations()
{
  m_pAnnotations->setText( buildVoice().ptr() );
}

void eciVoiceSetup::genderValue(bool p_set)
{
  m_pGender->setChecked( p_set );  
  showAnnotations();
}

void eciVoiceSetup::volumeValue(int p_val)
{
  m_pVolume->setValue( p_val );
  showAnnotations();
}

void eciVoiceSetup::pitchBaseValue(int p_val)
{
  m_pPitchbase->setValue(p_val);
  showAnnotations();
}

void eciVoiceSetup::speedValue(int p_val)
{
  m_pSpeed->setValue(p_val);
  showAnnotations();
}

void eciVoiceSetup::headValue(int p_val)
{
  m_pHeadsize->setValue(p_val);
  showAnnotations();
}

void eciVoiceSetup::breathinessValue(int p_val)
{
  m_pBreathiness->setValue(p_val);
  showAnnotations();
}

void eciVoiceSetup::pitchRangeValue(int p_val)
{
  m_pPitchrange->setValue(p_val);
  showAnnotations();
}

void eciVoiceSetup::roughnessValue(int p_val)
{
  m_pRoughness->setValue(p_val);
  showAnnotations();
}

void eciVoiceSetup::setVoice(int p_ix)
{
  _index  = p_ix;
  genderValue(defVoices[p_ix].s_gender);
  volumeValue(defVoices[p_ix].s_volume);
  pitchBaseValue(defVoices[p_ix].s_pitch_base);
  speedValue(defVoices[p_ix].s_speed);
  headValue(defVoices[p_ix].s_headsize);
  breathinessValue(defVoices[p_ix].s_breathy);
  pitchRangeValue(defVoices[p_ix].s_pitch_fluc);
  roughnessValue(defVoices[p_ix].s_roughness);
}

void eciVoiceSetup::male1()
{
  setVoice(1);
}

void eciVoiceSetup::male2()
{
  setVoice(4);
}

void eciVoiceSetup::female()
{
  setVoice(2);
}

void eciVoiceSetup::child()
{
  setVoice(3);
}

void eciVoiceSetup::grandma()
{
  setVoice(7);
}

void eciVoiceSetup::grandpa()
{
  setVoice(8);
}

void eciVoiceSetup::setup(eci_user *p_user)
{
  _user = p_user;
  if ( _user && _user->voice.hasData() == false )
    _user->voice = "`v1";
  male1();
  show();
}

bool speak_plugin_command_speak(KviPluginCommandStruct *cmd)
{
  KviStr tmp,*s;

  if(!cmd->params)return false;
  if(cmd->params->count() <= 1)return false;

  for( s=cmd->params->at(1);s;s=cmd->params->next())
    {
      if(tmp.hasData())
	tmp.append(' ');
      tmp.append( *s );
    }
  libibmeci50_eciAddText(hECI,tmp.ptr());
  libibmeci50_eciSynthesize(hECI);
  //  eciSynchronize(hECI);
  /*
    cmd->error = KVI_ERROR_InvalidParameter;
    cmd->errorstr = "An unsigned volume value is expected";
    return false;
  */
  return true;
}

bool speak_plugin_command_speaker(KviPluginCommandStruct *cmd)
{
  KviStr *s;
  KviStr mynick;
  struct eci_user *user;

  if(!cmd->params)return false;
  if(cmd->params->count() <= 1 || cmd->params->count() > 3)
    return false;

  // If the speaker is ME, then shout all messages
  // where my nick is on the line.
  mynick      = cmd->frame->m_global.szCurrentNick.ptr();
  s           = cmd->params->at(1);
  if ( kvi_strEqualCI(s->ptr(),mynick.ptr()) )
    return (eci_speak2me = true);

  // See if we the user is already listed as a speaker.
  for( user=eci_users.first();user;user=eci_users.next() )
    if ( kvi_strEqualCI(s->ptr(),user->name.ptr()) )
      break;
  if ( user == 0 ) {
    user = new eci_user;
    user->name  = *s;
    user->voice = "";
  }
  if ( (s = cmd->params->next()) != 0 ) {
    user->voice = *s;
    while( (s = cmd->params->next()) != 0 )
      user->voice += ' ' + *s;
    eci_users.append( user );
  } else
    new eciVoiceSetup( user );

  return true;
}

bool speak_plugin_command_typer(KviPluginCommandStruct *cmd)
{
  KviStr *s;
  KviStr mynick;
  struct eci_user *user = 0;

  if(!cmd->params)return false;
  if(cmd->params->count() <= 1)return false;

  mynick = cmd->frame->m_global.szCurrentNick.ptr();
  for( s=cmd->params->at(1);s;s=cmd->params->next() )
    if ( kvi_strEqualCI(s->ptr(),mynick.ptr()) )
      eci_speak2me = false;
    else {
      for( user=eci_users.first();user;user=eci_users.next() )
	if ( kvi_strEqualCI(user->name.ptr(),s->ptr()) )
	  break;
      if ( user )
	eci_users.remove( user );
    }

  return true;
}

bool speak_plugin_command_smiley(KviPluginCommandStruct *cmd)
{
  if(!cmd->params)
    return false;
  if(cmd->params->count() > 1)
    return false;          // accept no parameters.
  eci_smileyEdit.editSmileys();  // Show smiley editor
	return true;
}

bool speak_plugin_event(KviPluginCommandStruct *cmd)
{
  smileyNode *np;
  KviStr tmp,*s;
  KviStr nick, mask, mynick;
  struct eci_user *user;
  bool speak_it = false;
//  int j;

	if(!cmd->params)return true;

  nick   = cmd->params->at(1)->ptr();
  mask   = cmd->params->next()->ptr();
  mynick = cmd->frame->m_global.szCurrentNick.ptr();

  s = cmd->params->next();
  // Take care of "mynick: ..."
  if ( kvi_strEqualCIN(mynick.ptr(),s->ptr(),mynick.len()) )
    speak_it = eci_speak2me;
  for( ;s;s=cmd->params->next() ) {
    if ( tmp.hasData() )
      tmp.append(' ');
    if ( smileyTree ) {
      np = (smileyNode *)smileyTree->Lookup(s->ptr());
      if ( np )
	*s = np->English();
    }
    tmp.append( *s );
    // If my nick is on the line, speak the text.
    if ( !speak_it && kvi_strEqualCI(mynick.ptr(),s->ptr()) )
      speak_it = eci_speak2me;
  }

  // If the text is not to be spoken, check if it comes from
  // a nick, that I have registered as a speaker.
  for( user=eci_users.first();user;user=eci_users.next() )
    if ( kvi_strEqualCI(user->name.ptr(),nick.ptr()) )
      break;
  speak_it = ( speak_it || user );
  // Prepend any voice specific text.
  if ( user &&  user->voice.hasData() )
    tmp.prepend( user->voice + ' ' );

  if ( speak_it && tmp.hasData() ) {
    libibmeci50_eciAddText(hECI,tmp.ptr());
    libibmeci50_eciSynthesize(hECI);
  }    
  return false;
}

bool speak_plugin_DccMessageEvent(KviPluginCommandStruct *cmd)
{
  smileyNode *np;
  KviStr tmp,*s;
  bool speak_it = eci_speak2me;
//  int j;

  for(s = cmd->params->at(5);s;s=cmd->params->next() )
    {
      if ( tmp.hasData() )
	tmp.append(' ');
      if ( smileyTree ) {
	np = (smileyNode *)smileyTree->Lookup(s->ptr());
	if ( np )
	  *s = np->English();
      }
      tmp.append( *s );
    }
  debug( tmp.ptr() );
  if ( speak_it && tmp.hasData() ) {
    libibmeci50_eciAddText(hECI,tmp.ptr());
    libibmeci50_eciSynthesize(hECI);
  }    
  return false;
}

bool speak_plugin_nickEvent(KviPluginCommandStruct *cmd)
{
  eci_user *user;
  KviStr nick, username, host, newnick, mynick;

  nick     = cmd->params->at(1)->ptr();
  username = cmd->params->next()->ptr();
  host     = cmd->params->next()->ptr();
  newnick  = cmd->params->next()->ptr();
  mynick   = cmd->frame->m_global.szCurrentNick.ptr();

  // Hey, if it's me don't bother... just to be sure :-)
  if ( kvi_strEqualCI(nick.ptr(),mynick.ptr()) )
    return false;
  // Find the reference to the old nick.
  for( user=eci_users.first();user;user=eci_users.next() )
    if ( kvi_strEqualCI(user->name.ptr(),nick.ptr()) )
      break;
  // If old nick was a speaker, rename the speaker to the new nick.
  if ( user )
    user->name = newnick;

  return false;
}

bool speak_plugin_function_isSpeaker(KviPluginCommandStruct *cmd,
				     KviStr *buffer)
{
  eci_user *user;
  KviStr *nick, mynick;

  nick    = cmd->params->at(1);
  mynick  = cmd->frame->m_global.szCurrentNick.ptr();

  if ( nick == 0 || kvi_strEqualCI(nick->ptr(),mynick.ptr()) )
    buffer->setNum( (int)eci_speak2me );
  else {
    for( user=eci_users.first();user;user=eci_users.next() )
      if ( kvi_strEqualCI(nick->ptr(),user->name.ptr()) )
	break;
    buffer->setNum( (int)(user != 0) );
  }

  return true;
}

bool speak_plugin_init(KviPluginCommandStruct *cmd)
{
  libibmeci50_handle = dlopen("libibmeci50.so",RTLD_GLOBAL|RTLD_NOW);
  if(!libibmeci50_handle){
    debug("[libkvispeak] : Could not load libibmeci50.so : %s",dlerror());
    cmd->errorstr.sprintf("[libkvispeak] : Could not load libibmeci50.so : %s",dlerror());
    return false;
  }
  libibmeci50_eciNew = (void * (*)()) dlsym(libibmeci50_handle,"eciNew");
  if(!libibmeci50_eciNew){
    debug("[libkvispeak] : Could not find symbol eciNew : %s",dlerror());
    cmd->errorstr.sprintf("[libkvispeak] : Could not find symbol eciNew : %s",dlerror());
    dlclose(libibmeci50_handle);
    return false;
  }
  libibmeci50_eciSetOutputDevice = (void (*)(void *,int)) dlsym(libibmeci50_handle,"eciSetOutputDevice");
  if(!libibmeci50_eciSetOutputDevice){
    debug("[libkvispeak] : Could not find symbol eciSetOutputDevice : %s",dlerror());
    cmd->errorstr.sprintf("[libkvispeak] : Could not find symbol eciSetOutputDevice : %s",dlerror());
    dlclose(libibmeci50_handle);
    return false;
  }
  libibmeci50_eciSetParam = (void (*)(void *,int,int)) dlsym(libibmeci50_handle,"eciSetParam");
  if(!libibmeci50_eciSetParam){
    debug("[libkvispeak] : Could not find symbol eciSetParam : %s",dlerror());
    cmd->errorstr.sprintf("[libkvispeak] : Could not find symbol eciSetParam : %s",dlerror());
    dlclose(libibmeci50_handle);
    return false;
  }
  libibmeci50_eciDelete = (void (*)(void *)) dlsym(libibmeci50_handle,"eciDelete");
  if(!libibmeci50_eciDelete){
    debug("[libkvispeak] : Could not find symbol eciDelete : %s",dlerror());
    cmd->errorstr.sprintf("[libkvispeak] : Could not find symbol eciDelete : %s",dlerror());
    dlclose(libibmeci50_handle);
    return false;
  }
  libibmeci50_eciAddText = (void (*)(void *,const char *)) dlsym(libibmeci50_handle,"eciAddText");
  if(!libibmeci50_eciAddText){
    debug("[libkvispeak] : Could not find symbol eciAddText : %s",dlerror());
    cmd->errorstr.sprintf("[libkvispeak] : Could not find symbol eciAddText : %s",dlerror());
    dlclose(libibmeci50_handle);
    return false;
  }
  libibmeci50_eciSynthesize = (void (*)(void *)) dlsym(libibmeci50_handle,"eciSynthesize");
  if(!libibmeci50_eciSynthesize){
    debug("[libkvispeak] : Could not find symbol eciSynthesize : %s",dlerror());
    cmd->errorstr.sprintf("[libkvispeak] : Could not find symbol eciSynthesize : %s",dlerror());
    dlclose(libibmeci50_handle);
    return false;
  }
  char *eci_ini = getenv("ECIINI");
  if ( eci_ini == 0 ) {
    eci_ini = "eci.ini";
    if ( !kvi_fileExists(eci_ini) )
      system("/usr/lib/ViaVoiceOutloud/bin/inigen /usr/lib/enu50.so .");
  }
  if(!kvi_fileExists(eci_ini))
    {
      debug("[libkvispeak] : eci.ini is missing in the current directory ,\nplease read the plugin help page");
      cmd->errorstr.sprintf("[libkvispeak] : eci.ini is missing in the current directory ,\nplease read the plugin help page");
      dlclose(libibmeci50_handle);
      return false;
    }
  
  hECI = libibmeci50_eciNew();
  if(!hECI)
    {
      debug("[libkvispeak] : Can't initialize speak engine");
      cmd->errorstr.sprintf("[libkvispeak] : Can't initialize speak engine");
      dlclose(libibmeci50_handle);
      return false;
    }
  libibmeci50_eciSetOutputDevice(hECI,1); 
  // the first 1 here is eciInputType (enum in eci.h)
  libibmeci50_eciSetParam(hECI,1,1);
  
  kvirc_plugin_register_command(cmd->handle,"SPEAK",speak_plugin_command_speak);
  kvirc_plugin_register_command(cmd->handle,"SPEAKER",speak_plugin_command_speaker);
  kvirc_plugin_register_command(cmd->handle,"TYPER",speak_plugin_command_typer);
  kvirc_plugin_register_command(cmd->handle,"SMILEYEDIT",speak_plugin_command_smiley);
  kvirc_plugin_register_function(cmd->handle,"isSpeaker",speak_plugin_function_isSpeaker);
  kvirc_plugin_add_hook(cmd->handle,KviEvent_OnChannelMessage,speak_plugin_event);
  kvirc_plugin_add_hook(cmd->handle,KviEvent_OnDccChatMessage,speak_plugin_DccMessageEvent);
  kvirc_plugin_add_hook(cmd->handle,KviEvent_OnNick,speak_plugin_nickEvent);
  eci_users.setAutoDelete( true );
  eci_users.clear();
  eci_smileyEdit.init();
  return true; // success
}

void speak_plugin_cleanup()
{
  if(hECI)libibmeci50_eciDelete(hECI);
  if(libibmeci50_handle)dlclose(libibmeci50_handle);
  //	kvirc_plugin_unregister_all_commands(cmd->handle);
  //	kvirc_plugin_remove_all_hooks(cmd->handle);
	kvirc_plugin_unregister_meta_object("eciSetupData");
	kvirc_plugin_unregister_meta_object("smileyClassData");
	kvirc_plugin_unregister_meta_object("eciVoiceSetup");
	kvirc_plugin_unregister_meta_object("smileyClass");
}

/*
	@document: doc_plugin_speak.kvihelp
	@title: The IBM ViaVoice Outloud interface plugin
		The "speak" plugin is a simple interface to the IBM ViaVoice Outloud
		speech synthesis kit.<br>
		This plugin relies on the IBM ViaVoice Outloud kit
		available for "free testing" at the IBM site.<br>
		The plugin loads dynamically the library libibmeci50.so
		provided with the kit.<br>
		If you have installed the kit and you have trouble in
		loading the plugin check that the library is visible by ldconfig.<br>
		(This means : add the library path to /etc/ld.so.conf and run ldconfig).<br>
		<br>
		The synthesis interface has been implemented by
		Orn Einar Hansen &lt;oe.hansen@gamma.telenordia.se&gt;<br>
		The plugin exports several commands:<br>
		<b>speak &lt;text&gt;</b><br>
		It sends the &lt;text&gt; to the speech synthesis engine.<br>
		<b>speaker &lt;nick&gt; &lt;voice code&gt;</b><br>
		Which sets the flag for &lt;nick&gt; so that all text that
		that person writes, is sent through the speech synthesis engine
		for vocal expression.  If the &lt;voice code&rt; is omitted then
		a GUI will popup, that allows the user to specify the characteristics
		of the speakers voice.<br>
		<b>typer &lt;nick&gt; &lt;nick&gt; ...</b><br>
		That removes the flags on &lt;nick&gt; so that the text they
		write will not be directed through the synthesis engine, but
		will appear as normal text input.<br>
		<b>smileyedit</b>
		This command opens up a dialog, where short hands, like smileys
		can be edited and made to speak a text, instead of gibberish.
		For instance, the smiley :-) can be made to speak out as
		smile.<br>
		And one function:<br>
		<b>isSpeaker(&lt;nick&gt;)</b><br>
		It returns a true value, if the nick referenced is registered as a<br>
		speaker, and a false otherwise.<br>
		<docsubtitle>IMPORTANT NOTE</docsubtitle><br>
		For some reason the engine wants an eci.ini file to be present
		in the CURRENT DIRECTORY,<br>
		or the ECIINI environment variable to be set to its exact
		path.<br>
		Usually it is the directory from which you start kvirc.<br>
		If you start kvirc from the KDE panel, the CURRENT DIRECTORY
		is your home directory.<br>
		A suitable eci.ini file is usually installed
		in /usr/lib/ViaVoiceOutloud/samples/cmdlinespeak/eci.ini
		by the ViaVoice kit rpm.<br>
		You have either to copy this file to the current directory
		or <a href="cd.kvihelp">CD</a> to that directory before loading the plugin.<br>
		A missing eci.ini file causes the engine to produce strange results
		including randomized segfaults.<br>
		For this reason it is NOT a good idea to set the autoload flag on this plugin
		unless you always start kvirc from the same directory and have an eci.ini file there,<br>
		or if you always have the ECIINI set prior to starting the kvirc program.<br>
		<br><br>
*/

void speak_plugin_help()
{
	kvirc_plugin_show_help_page("doc_plugin_speak.kvihelp");
}

KviPlugin kvirc_plugin =
{
	"speak" ,
	"Interface to the\nIBM ViaVoice\nspeak synthesis library" ,
	"0.1" ,
	"Orn Einar Hansen <oe.hansen@gamma.telenordia.se>\n" \
	"Szymon Stefanek <stefanek@tin.it>" ,
	"Exports the /SPEAK /SPEAKER /TYPER and \n" \
	"/SMILEYEDIT commands" ,
	speak_plugin_init ,
	speak_plugin_cleanup ,
	0 ,
	speak_plugin_help
};


