//
//   File : libkvfserve.cpp (/usr/build/KVIrc/kvirc/src/kvilib/libkvifserve.cpp)
//   Last major modification : Wed Jul 21 1999 16:41:14 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC_PLUGIN__
#include "kvirc_plugin.h"

#include "libkvifserve.h"

#include "kvi_dcc_chat.h"
#include "kvi_dcc_manager.h"

#include "kvi_locale.h"

#include <qdir.h>
#include <qfileinfo.h>
#include <qpushbutton.h>

// Maximum number of saved user sessions
#define KVIFSERVE_MAX_SAVED_SESSIONS 32
#define KVIFSERVE_MAX_PENDING_SESSIONS 32

// A running fserve session structure
typedef struct _KviFServeSession
{
	KviStr      nick;                       // nickname of the user
	KviStr      address;                    // ip address (it is the ip address of the remote end of the dcc chat)
	KviStr      credit;                     // current credit in bytes or the string "unlimited"
	KviStr      currentDir;                 // current directory , relative to the fserve root
	KviWindow * pWnd;                       // dcc chat window pointer
} KviFServeSession;

// A pending dcc fservice
typedef struct _KviPendingSession
{
	KviWindow * pWnd;                       // an opened dcc chat window waiting for the connection
	KviStr      password;                   // a password (if it has been specified in PRIVMSG)
} KviPendingSession;

// A saved fserve session
// Used to save the credit
typedef struct _KviSavedSession
{
	KviStr nick;                            // nickname
	KviStr address;                         // ip address
	KviStr credit;                          // last credit (never "unlimited")
} KviSavedSession;


// Global variables
static KviStr g_szFServeRoot("/");          // root directory of the fileserver (must be an absolute path)
static KviStr g_szMotd("No MOTD yet");      // message of the day (multiline text)
static KviStr g_szFServePass("");           // password for the public access
static KviStr g_szInitialCredit("0");       // initial credit for the public access
static int g_iRatioTake = 1;                // ratio : give g_iRatioGive bytes of credit for each
static int g_iRatioGive = 1;                //         packet of g_iRatioTake bytes
static bool g_bServerActive = false;        // global activation flag (if this is false the fserver is dead)
static bool g_bShowMotdAtLogin = true;      // show message of the day at user login ?
static QList<KviStr> * g_pBannedIpList = 0; // banned ip addresses
static bool g_bListenToPrivmsg = false;     // can be run by /msg <nick> !fserve <pass> ?
static unsigned int g_uMaxRunningSessions = 10;      // maximum number of running sessions

static QList<KviFServeSession> * g_pSessions = 0;         // list of active sessions
static QList<KviSavedSession> * g_pSavedSessions = 0;     // list of saved sessions
static KviFServeConfigDialog * g_pConfigDialog = 0;       // global plugin configuration dialog
static QList<KviPendingSession> * g_pPendingSessions = 0; // list of pending sessions

void * g_handle = 0;                        // global plugin handle (set by the init routine)

static bool fserve_checkRoot()
{
	//
	// Check the existence and validity of the fserver root directory
	//

	g_szFServeRoot.stripWhiteSpace();
	if(g_szFServeRoot.lastCharIs('/')){
		// We need no slash at the end (unless the root is '/')
		if(g_szFServeRoot.len() > 1)g_szFServeRoot.cutRight(1);
	}
	// The root must be an absolute path!
	if(*(g_szFServeRoot.ptr()) != '/'){
		debug("FSERVE: File server root path is relative");
		return false;
	} else {
		// Check if we can list the contents of the dir
		QDir d(g_szFServeRoot.ptr());
		if(!d.entryInfoList()){
			debug("FSERVE: Could not read root directory contents");
			return false;
		}
	}
	return true;
}


static void fserve_pluginSaveState()
{
	KviStr fName;
	kvirc_plugin_get_config_file_path(fName,"fserve");

	KviConfig cfg(fName.ptr());

	KviStr tmp = g_szMotd;
	tmp.replaceAll('\n',"{NewLine}");
	cfg.writeEntry("Motd",tmp.ptr());
	cfg.writeEntry("ShowMotdAtLogin",g_bShowMotdAtLogin);
	cfg.writeEntry("ServerActive",g_bServerActive);
	cfg.writeEntry("RatioGive",g_iRatioGive);
	cfg.writeEntry("RatioTake",g_iRatioTake);
	cfg.writeEntry("InitialCredit",g_szInitialCredit.ptr());
	cfg.writeEntry("FServePass",g_szFServePass.ptr());
	cfg.writeEntry("FServeRoot",g_szFServeRoot.ptr());
	cfg.writeEntry("ListenToPrivmsg",g_bListenToPrivmsg);
	cfg.writeEntry("MaxRunningSessions",g_uMaxRunningSessions);

	tmp = "";
	for(KviStr * s = g_pBannedIpList->first();s;s=g_pBannedIpList->next()){
		if(tmp.hasData())tmp.append(',');
		tmp.append(s->ptr());
	}
	cfg.writeEntry("BannedIpList",tmp.ptr());
}




static void fserve_pluginLoadState()
{
	//
	// Load the options of the plugin
	// called from the init_routine
	//

	// Find the config file path...
	// We just have to supply the plugin name
	// KVIrc will tell us how the file is named and
	// where it is.
	KviStr fName;
	kvirc_plugin_get_config_file_path(fName,"fserve");
	// Create a KviConfig object
	// KviConfig is a class specifically designed
	// for dealing with config files
	// Its declaration is in kvilib/kvi_config.h
	KviConfig cfg(fName.ptr());
	// Finally load the options
	g_szFServeRoot     = cfg.readEntry("FServeRoot","/usr/local");
	g_szFServePass     = cfg.readEntry("FServePass","");
	g_szInitialCredit  = cfg.readEntry("InitialCredit","0");
	g_iRatioTake       = cfg.readIntEntry("RatioTake",1);
	if(g_iRatioTake < 1)g_iRatioTake = 1;
	g_iRatioGive       = cfg.readIntEntry("RatioGive",1);
	if(g_iRatioGive < 0)g_iRatioGive = 1;
	g_bServerActive    = cfg.readBoolEntry("ServerActive",false);
	g_bListenToPrivmsg = cfg.readBoolEntry("ListenToPrivmsg",false);
	g_uMaxRunningSessions = cfg.readUIntEntry("MaxRunningSessions",10);
	// The message of the day
	g_bShowMotdAtLogin = cfg.readBoolEntry("ShowMotdAtLogin",true);
	g_szMotd           = cfg.readEntry("Motd","No MOTD yet...");
	// the KviConfig class can not save multiline text options
	// here is the workaround
	g_szMotd.replaceAll("{NewLine}","\n");
	// Banned Ip addresses
	KviStr tmp         = cfg.readEntry("BannedIpList","");
	KviStr ip;
	while(tmp.hasData()){
		tmp.getToken(ip,',');
		ip.stripWhiteSpace();
		if(ip.hasData()){
			// TODO : check if it is a valid IP address mask ??
			g_pBannedIpList->append(new KviStr(ip.ptr()));
		}
	}
	// Check the server root directory
	g_bServerActive = (fserve_checkRoot() && g_bServerActive);
}

static KviFServeSession * fserve_findRunningSession(KviWindow * wnd)
{
	//
	// Find a running fserve session
	//

	for(KviFServeSession *s=g_pSessions->first();s;s=g_pSessions->next()){
		if(wnd == s->pWnd)return s;
	}
	return 0;
}

static KviFServeSession * fserve_findRunningSessionByNickAndIp(const char *nick,const char *ip)
{
	//
	// Find a running fserve session
	//

	for(KviFServeSession *s=g_pSessions->first();s;s=g_pSessions->next()){
		if(kvi_strEqualCI(nick,s->nick.ptr())){
			if(kvi_strEqualCI(s->address.ptr(),ip))return s;
		}
	}
	return 0;
}

static KviPendingSession * fserve_findPendingSession(KviWindow * wnd)
{
	//
	// Find a pending fserve session
	//

	for(KviPendingSession *s=g_pPendingSessions->first();s;s=g_pPendingSessions->next()){
		if(wnd == s->pWnd)return s;
	}
	return 0;
}

static KviSavedSession * fserve_findSavedSession(const char *nick,const char *ip)
{
	//
	// Find a saved fserve session
	//

	for(KviSavedSession *s=g_pSavedSessions->first();s;s=g_pSavedSessions->next()){
		if(kvi_strEqualCI(s->nick.ptr(),nick)){
			if(kvi_strEqualCI(s->address.ptr(),ip))return s;
		}
	}
	return 0;
}

static void fserve_cleanupPendingSessions(KviFrame * frm)
{
	QList<KviPendingSession> l;
	l.setAutoDelete(false);
	for(KviPendingSession *s=g_pPendingSessions->first();s;s=g_pPendingSessions->next()){
		if(!frm->windowExists(s->pWnd))l.append(s);
	}
	for(KviPendingSession *ps=l.first();ps;ps=l.next())g_pPendingSessions->removeRef(ps);
}

static void fserve_chatOutputAndSend(KviPluginCommandStruct *cmd,const char *nick,const KviStr& data)
{
	cmd->window->output(KVI_OUT_OWN,"[fserve >> %s] %s",nick,data.ptr());
	KviStr tmp="[fserve] ";
	tmp.append(data);
	if(!((KviDccChat *)(cmd->window))->sendData(tmp.ptr()))debug("oops! dcc chat not connected?");
}

static void fserve_parseCmdStats(KviPluginCommandStruct *cmd,KviFServeSession *s)
{
	KviStr tmp(KviStr::Format,"Stats for user %s@%s",s->nick.ptr(),s->address.ptr());
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);
	tmp.sprintf("Directory : %s",s->currentDir.ptr());
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);
	tmp.sprintf("Credit : %s bytes",s->credit.ptr());
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);
	if(s->credit.isUnsignedNum()){
		tmp.sprintf("Ratio : %d:%d",g_iRatioTake,g_iRatioGive);
		fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);
	}
}

static void fserve_parseCmdUnknown(KviPluginCommandStruct *cmd,KviFServeSession *s)
{
	cmd->window->output(KVI_OUT_OWN,"[fserve >> %s] %s",s->nick.ptr(),"Unknown command");
	((KviDccChat *)(cmd->window))->sendData("[fserve] Unknown command");
}

static void fserve_parseCmdLs(KviPluginCommandStruct *cmd,KviFServeSession *s)
{
	KviStr tmp = g_szFServeRoot;
	tmp.append('/');
	tmp.append(s->currentDir);

	QString nameFilter = ((cmd->params->count() > 6) ? QString(kvirc_plugin_param(cmd,6)) : QString::null);

//	if(cmd->params->count() > 6)nameFilter = kvirc_plugin_param(cmd,6);

	QDir d(QString(tmp.ptr()),nameFilter,QDir::Name|QDir::IgnoreCase|QDir::DirsFirst,QDir::All|QDir::Readable|QDir::Hidden);

	const QFileInfoList * l = d.entryInfoList();
	if(!l){
		fserve_chatOutputAndSend(cmd,s->nick.ptr(),KviStr("Unable to list directory contents."));
		fserve_chatOutputAndSend(cmd,s->nick.ptr(),KviStr("Probably the file server root directory was reconfigured"));
		fserve_chatOutputAndSend(cmd,s->nick.ptr(),KviStr("Warping you to /"));
		s->currentDir = "/";
		return;
	}

	QFileInfoListIterator it(*l);
	QFileInfo *fi;

	int total = 0;
	bool bUnlimitedCredit = kvi_strEqualCI(s->credit.ptr(),"unlimited");
	unsigned long credit = 0;

	if(!bUnlimitedCredit){
		bool bOk = false;
		credit = s->credit.toUInt(&bOk);
		if(!bOk)s->credit = "0";
		credit = 0;
	}

	KviStr num;

	while((fi = it.current())){
		if(fi->isDir()){
			tmp = "D           ";
			tmp.append(KVI_TEXT_BOLD);
			tmp.append(fi->fileName());
			tmp.append(KVI_TEXT_BOLD);
		} else if(fi->isSymLink()){
			tmp = "L ";
			num.setNum(fi->size());
			if((fi->size() > credit) && (!bUnlimitedCredit)){
				tmp.append(KVI_TEXT_COLOR);
				tmp.append("4 ");
				tmp.append(num);
				tmp.append(KVI_TEXT_RESET);
				tmp.append(' ');
				while(tmp.len() < 15)tmp.append(' ');
			} else {
				tmp.append(' ');
				tmp.append(num);
				tmp.append(' ');
				while(tmp.len() < 12)tmp.append(' ');
			}
			tmp.append(fi->fileName());
		} else {
			tmp = "F ";
			num.setNum(fi->size());
			if((fi->size() > credit) && (!bUnlimitedCredit)){
				tmp.append(KVI_TEXT_COLOR);
				tmp.append("4 ");
				tmp.append(num);
				tmp.append(KVI_TEXT_RESET);
				tmp.append(' ');
				while(tmp.len() < 15)tmp.append(' ');
			} else {
				tmp.append(' ');
				tmp.append(num);
				tmp.append(' ');
				while(tmp.len() < 12)tmp.append(' ');
			}
			tmp.append(fi->fileName());
		}

		fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);
		
		++it;
		++total;
	}
	tmp.setNum(total);
	tmp.prepend("total ");
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);
}

static void fserve_parseCmdCd(KviPluginCommandStruct *cmd,KviFServeSession *s)
{
	// get the directory , other parameters are ignored
	if(cmd->params->count() < 7)return;

	KviStr dir = kvirc_plugin_param(cmd,6);
	if(dir.isEmpty())return;

	KviStr token;
	const char *aux = dir.ptr();
	// save the last directory
	KviStr saveDir = s->currentDir;
	// if it is absolute path , start from the root
	while(*aux == '/'){
		s->currentDir = "/";
		aux++;
	}

	// ensure that our s->currentDir is valid at all
	if(*(s->currentDir.ptr()) != '/')s->currentDir = '/';

	bool bError = false;

	// cd one dir at a time
	while((*aux) && (!bError)){
		// get the next path component
		aux = kvi_extractToken(token,aux,'/');

		if(token.hasData()){
			if(kvi_strEqualCI(token.ptr() , "..")){
				// cdup
				if(s->currentDir.len() > 1){
					// can cdup
					if(s->currentDir.lastCharIs('/'))s->currentDir.cutRight(1);
					int idx = s->currentDir.findLastIdx('/');
					if(idx > 0)s->currentDir = s->currentDir.left(idx);
					else {
						if(idx == 0)s->currentDir = '/';
					}
					// double check
					if(*(s->currentDir.ptr()) != '/')s->currentDir = "/";
				}
			} else if(!kvi_strEqualCI(token.ptr(), ".")){
				// cd directory
				KviStr tmp = g_szFServeRoot;
				tmp.append(s->currentDir);
				if(!s->currentDir.lastCharIs('/'))tmp.append('/');
				tmp.append(token);
				if(kvi_directoryExists(tmp.ptr())){
					if(!s->currentDir.lastCharIs('/'))s->currentDir.append('/');
					s->currentDir.append(token);
				} else bError = true;
			}
		}
	}

	if(bError){
		s->currentDir = saveDir;
		token.sprintf("Can not cd to %s : no such directory",dir.ptr());
	} else token.sprintf("Directory changed to %s",s->currentDir.ptr());

	fserve_chatOutputAndSend(cmd,s->nick.ptr(),token);
}

static void fserve_parseCmdGet(KviPluginCommandStruct *cmd,KviFServeSession *s)
{
	if(cmd->params->count() < 7)return;
	KviStr filename = kvirc_plugin_param(cmd,6);
	if(filename.isEmpty())return;

	if(filename.findFirstIdx('/') == -1){

		KviStr fName = g_szFServeRoot;
		if(!fName.lastCharIs('/'))fName.append('/');
		fName.append(s->currentDir);
		if(!fName.lastCharIs('/'))fName.append('/');
		fName.append(filename);

		QFileInfo fi(fName.ptr());
		if(fi.exists()){
			if(fi.isReadable()){
				bool bOk = false;
				unsigned int credit = s->credit.toUInt(&bOk);
				if(bOk){
					if(credit < fi.size()){
						KviStr tmp(KviStr::Format,"%s: Credit too low , please upload first",filename.ptr());
						fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);
						return;
					} else {
						credit -= fi.size();
						s->credit.setNum(credit);
					}
				}
				// start the DCC
				cmd->frame->m_pDccManager->requestDccSend(s->nick.ptr(),fName.ptr());
				filename.prepend('/');
				filename.prepend(s->currentDir);
				KviStr tmp(KviStr::Format,"Sending file %s",filename.ptr());
				fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);
				if(bOk){
					tmp.sprintf("Your credit is now %s bytes",s->credit.ptr());
					fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);
				}
			} else {
				KviStr tmp(KviStr::Format,"%s: Access forbidden",filename.ptr());
				fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);
			}
		} else {
			KviStr tmp(KviStr::Format,"%s: No such file",filename.ptr());
			fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);
		}
	} else {
		KviStr tmp(KviStr::Format,"Invalid file name %s (no path elements allowed)",filename.ptr());
		fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);
	}
}

static void fserve_parseCmdHelp(KviPluginCommandStruct *cmd,KviFServeSession *s)
{
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),KviStr("Available commands:"));
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),KviStr(".cd <directory> : changes the current working directory"));
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),KviStr(".ls : lists the contents of the current directory"));
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),KviStr(".get <filename> : starts a dcc get session"));
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),KviStr(".stats : shows your current credit"));
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),KviStr(".motd : shows the message of the day"));
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),KviStr(".pwd : shows the current working directory"));
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),KviStr(".quit: exits fserve mode"));
}

static void fserve_parseCmdMotd(KviPluginCommandStruct *cmd,KviFServeSession *s)
{
	KviStr tmp = g_szMotd;
	KviStr line;
	while(tmp.getLine(line))fserve_chatOutputAndSend(cmd,s->nick.ptr(),line);
}

static void fserve_parseCmdPwd(KviPluginCommandStruct *cmd,KviFServeSession *s)
{
	KviStr tmp(KviStr::Format,"Current directory is %s",s->currentDir.ptr());
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),tmp);	
}

static void fserve_closeSession(KviFServeSession *s)
{
	if(!kvi_strEqualCI(s->credit.ptr(),"unlimited")){
		// limited credit....need to save the session
		KviSavedSession * ss = new KviSavedSession;
		ss->nick = s->nick;
		ss->address = s->address;
		ss->credit = s->credit;
		g_pSavedSessions->append(ss);
		if(g_pSavedSessions->count() > KVIFSERVE_MAX_SAVED_SESSIONS)g_pSavedSessions->removeFirst();
	}
	// close the session
	g_pSessions->removeRef(s);
}

static void fserve_parseCmdQuit(KviPluginCommandStruct *cmd,KviFServeSession *s)
{
	fserve_chatOutputAndSend(cmd,s->nick.ptr(),KviStr("File server session closed."));	
	fserve_closeSession(s);
}

static bool fserve_isBannedIp(const char *ip)
{
	for(KviStr * s=g_pBannedIpList->first();s;s=g_pBannedIpList->next()){
		if(kvi_matchWildExpr(ip,s->ptr()))return true;
	}
	return false;
}

static void fserve_startSession(KviPluginCommandStruct *cmd,const char *nick,
	const char *ip,const char *credit)
{
	KviFServeSession *s = new KviFServeSession;
	s->nick = nick;
	s->address = ip;
	s->currentDir = "/";
	s->credit = credit;
	s->pWnd = cmd->window;
	g_pSessions->append(s);
	cmd->window->output(KVI_OUT_INTERNAL,"[fserve] Opening fserve session for user %s@%s",s->nick.ptr(),s->address.ptr());
	if(g_bShowMotdAtLogin)fserve_parseCmdMotd(cmd,s);
	fserve_parseCmdStats(cmd,s);
//	fserve_parseCmdPwd(cmd,s);
}

static void fserve_startNormalSession(KviPluginCommandStruct *cmd,const char *nick,const char *ip)
{
	KviSavedSession * ss= fserve_findSavedSession(nick,ip);
	if(ss){
		// normal session
		fserve_startSession(cmd,nick,ip,ss->credit.ptr());
		g_pSavedSessions->removeRef(ss);
	} else {
		// new session
		fserve_startSession(cmd,nick,ip,g_szInitialCredit.ptr());
	}
}

static void fserve_doLogin(KviPluginCommandStruct * cmd,KviStr &nick,KviStr &user,KviStr &host,KviStr &ip,KviStr &pass)
{
	// user & host is known
	if(g_pSessions->count() >= g_uMaxRunningSessions){
		fserve_chatOutputAndSend(cmd,nick.ptr(),KviStr("Sorry , the server is full: try again later"));
		return;
	}

	KviStr mask(KviStr::Format,"%s!%s@%s",nick.ptr(),user.ptr(),host.ptr());
	KviRegisteredUser * u = kvirc_plugin_find_registered_user_with_flag(mask.ptr(),'f');
	if(u){
		// user is registered with the fserve flag
		// check if has a password set
		u->passwd.stripWhiteSpace(); // no spaces allowed
		if(u->passwd.hasData()){
			// check if the user specified a password
			if(pass.isEmpty()){
				// failed...no pass specified
				// if the fserve has no global pass , allow a normal login
				if(g_szFServePass.isEmpty()){
					// the fserve doesn't need a global pass...ok
					fserve_startNormalSession(cmd,nick.ptr(),ip.ptr());
				} else {
					// the fserve needs also a global pass...failed
					fserve_chatOutputAndSend(cmd,nick.ptr(),KviStr("Use '.fserve <password>' to login"));
				}
			} else {
				if(kvi_strEqualCI(pass.ptr(),u->passwd.ptr())){
					// matched his password...ok..leech account
					fserve_startSession(cmd,nick.ptr(),ip.ptr(),"unlimited");
				} else if(kvi_strEqualCI(pass.ptr(),g_szFServePass.ptr())){
					// seems that he wants a normal session
					fserve_startNormalSession(cmd,nick.ptr(),ip.ptr());
				} else {
					// failed at all...
					fserve_chatOutputAndSend(cmd,nick.ptr(),KviStr("Access denied: invalid password"));
				}	
			}
		} else {
			// no password needed: start the session
			fserve_startSession(cmd,nick.ptr(),ip.ptr(),"unlimited");
		}
	} else {
		// user is not registered
		if(g_szFServePass.hasData()){
			// a password is needed
			if(pass.isEmpty()){
				// failed...no pass specified
				fserve_chatOutputAndSend(cmd,nick.ptr(),KviStr("Use '.fserve <password>' to login"));
				return;
			} else {
				if(!kvi_strEqualCI(pass.ptr(),g_szFServePass.ptr())){
					fserve_chatOutputAndSend(cmd,nick.ptr(),KviStr("Access denied: invalid password"));
					return;
				}	
			}
		}	
		// no password needed or password ok
		fserve_startNormalSession(cmd,nick.ptr(),ip.ptr());
	}
}

static void fserve_parseCmdFServe(KviPluginCommandStruct *cmd)
{
	// first of all , build a mask
	// .fserve [<user> <host>] [password]
	KviStr nick = kvirc_plugin_param(cmd,1);
	KviStr user = kvirc_plugin_param(cmd,2);
	KviStr host = kvirc_plugin_param(cmd,3);
	KviStr ip   = kvirc_plugin_param(cmd,4);
	KviStr pass;
	g_szFServePass.stripWhiteSpace();

	if(g_pSessions->count() >= g_uMaxRunningSessions){
		fserve_chatOutputAndSend(cmd,nick.ptr(),KviStr("Sorry , the server is full: try again later"));
		return;
	}

	if(fserve_isBannedIp(ip.ptr())){
		fserve_chatOutputAndSend(cmd,nick.ptr(),KviStr("Sorry , you're banned from this file server"));
		return;
	}

	switch(cmd->params->count()){
		case 6:
			// Ok... :)
		break;
		case 7:
			pass = kvirc_plugin_param(cmd,6);
		break;
		case 8:
			user = kvirc_plugin_param(cmd,6);
			host = kvirc_plugin_param(cmd,7);
		break;
		case 9:
			user = kvirc_plugin_param(cmd,6);
			host = kvirc_plugin_param(cmd,7);
			pass = kvirc_plugin_param(cmd,8);
		break;
		default:
			fserve_chatOutputAndSend(cmd,nick.ptr(),KviStr("Invalid number of parameters."));
			fserve_chatOutputAndSend(cmd,nick.ptr(),KviStr("Syntax: .fserve [<user> <host>] [password]"));
			return;	
		break;
	}

	if(kvi_strEqualCS("*",host.ptr()) || kvi_strEqualCS("*",user.ptr())){
		// user & host unknown
		fserve_chatOutputAndSend(cmd,nick.ptr(),KviStr("Sorry , your user@hostname is unknown."));
		fserve_chatOutputAndSend(cmd,nick.ptr(),KviStr("Please login with '.fserve <user> <host> [password]'"));
		return;
	}
	fserve_doLogin(cmd,nick,user,host,ip,pass);
}

bool fserve_plugin_hook_onMePrivateMessage(KviPluginCommandStruct *cmd)
{
	if(!g_bServerActive || !g_bListenToPrivmsg)debug("WARNING : Unexpected hook on private message");

	if(kvi_strEqualCIN(kvirc_plugin_param(cmd,3),"!fserve",7)){

		if(g_pSessions->count() >= g_uMaxRunningSessions)return false; //server full
		if(g_szFServePass.hasData()){
			KviStr mask(KviStr::Format,"%s!%s",kvirc_plugin_param(cmd,1),kvirc_plugin_param(cmd,2));
			KviRegisteredUser * u = kvirc_plugin_find_registered_user_with_flag(mask.ptr(),'f');
		
			if(cmd->params->count() == 4){
				// must be registered with f flag and have no password set
				if(!u)return false;
				if(u->passwd.hasData())return false;
			} else {
				// specified a password
				if(!kvi_strEqualCI(g_szFServePass.ptr(),kvirc_plugin_param(cmd,4))){
					// invalid fserve global pass
					if(!u)return false;
					if(u->passwd.isEmpty())return false;
					if(!kvi_strEqualCI(u->passwd.ptr(),kvirc_plugin_param(cmd,4)))return false;
				}
			}
		}

		cmd->window->output(KVI_OUT_INTERNAL,"[fserve] Received a valid file service request from %s!%s: %s %s",
			kvirc_plugin_param(cmd,1),kvirc_plugin_param(cmd,2),kvirc_plugin_param(cmd,3),
			(cmd->params->count() > 4 ? kvirc_plugin_param(cmd,4) : ""));

		KviPendingSession * s = new KviPendingSession;
		s->pWnd = cmd->frame->m_pDccManager->requestDccChat(kvirc_plugin_param(cmd,1),kvirc_plugin_param(cmd,2));
		if(cmd->params->count() > 4)s->password = kvirc_plugin_param(cmd,4);

		fserve_cleanupPendingSessions(cmd->frame);
		g_pPendingSessions->append(s);
		return true; // do not output , do not create the query if it is not there
	}
	return false;
}

bool fserve_plugin_hook_onDccChatConnected(KviPluginCommandStruct *cmd)
{
	if(!g_bServerActive)return false;
	KviPendingSession * s = fserve_findPendingSession(cmd->window);
	if(!s)return false;
	KviStr nick = kvirc_plugin_param(cmd,1);
	KviStr user = kvirc_plugin_param(cmd,2);
	KviStr host = kvirc_plugin_param(cmd,3);
	KviStr ip   = kvirc_plugin_param(cmd,4);
	KviStr pass = s->password;
	fserve_doLogin(cmd,nick,user,host,ip,pass);
	g_pPendingSessions->removeRef(s);
	fserve_cleanupPendingSessions(cmd->frame);
	return false; // do not block any output
}

bool fserve_plugin_hook_onDccChatMessage(KviPluginCommandStruct *cmd)
{
	if(!g_bServerActive)return false;

	KviStr * firstWord = cmd->params->at(5);

	if(!firstWord)return false; // not a fserve command at all
	if(*(firstWord->ptr()) != '.')return false; // same as above


	KviFServeSession * s = fserve_findRunningSession(cmd->window);
	if(s){
		// A running service
		KviStr out = firstWord->ptr();
		for(unsigned int i=6;i < cmd->params->count();i++){
			out.append(' ');
			out.append(kvirc_plugin_param(cmd,i));
		}
		cmd->window->output(KVI_OUT_NONE,"[%s >> fserve] %s",s->nick.ptr(),out.ptr());

		if(kvi_strEqualCI(firstWord->ptr(),".stats"))fserve_parseCmdStats(cmd,s);
		else if(kvi_strEqualCI(firstWord->ptr(),".ls") || kvi_strEqualCI(firstWord->ptr(),".dir"))fserve_parseCmdLs(cmd,s);
		else if(kvi_strEqualCI(firstWord->ptr(),".cd") && (cmd->params->count() > 6))fserve_parseCmdCd(cmd,s);
		else if(kvi_strEqualCI(firstWord->ptr(),".get") && (cmd->params->count() > 6))fserve_parseCmdGet(cmd,s);
		else if(kvi_strEqualCI(firstWord->ptr(),".help"))fserve_parseCmdHelp(cmd,s);
		else if(kvi_strEqualCI(firstWord->ptr(),".motd"))fserve_parseCmdMotd(cmd,s);
		else if(kvi_strEqualCI(firstWord->ptr(),".quit"))fserve_parseCmdQuit(cmd,s);
		else if(kvi_strEqualCI(firstWord->ptr(),".pwd"))fserve_parseCmdPwd(cmd,s);
		else fserve_parseCmdUnknown(cmd,s);
	} else {
		// Not a running service
		if(kvi_strEqualCI(firstWord->ptr(),".fserve")){
			// A login attempt
			KviStr out = firstWord->ptr();
			for(unsigned int i=6;i < cmd->params->count();i++){
				out.append(' ');
				out.append(cmd->params->at(i)->ptr());
			}
			cmd->window->output(KVI_OUT_NONE,"[%s >> fserve] %s",kvirc_plugin_param(cmd,1),out.ptr());
			fserve_parseCmdFServe(cmd);
		} else return false;
	}

	return true;
}

bool fserve_plugin_hook_onDccChatTerminated(KviPluginCommandStruct *cmd)
{
	KviFServeSession * s = fserve_findRunningSession(cmd->window);
	if(s)fserve_closeSession(s);
	fserve_cleanupPendingSessions(cmd->frame); //not needed...but do it...just in case
	return false; // do not block any output
}

bool fserve_plugin_hook_onDccGetTransferComplete(KviPluginCommandStruct *cmd)
{
	bool bOk = false;
	unsigned int crAdd = cmd->params->at(6)->toUInt(&bOk);
	if(!bOk){
		debug("oops...can't calculate the credit to give to %s , giving (100.000 bytes * ratio)",cmd->params->at(1)->ptr());
		crAdd = 100000;
	}

	KviFServeSession * s = fserve_findRunningSessionByNickAndIp(kvirc_plugin_param(cmd,1),kvirc_plugin_param(cmd,4));
	if(s){
		if(s->credit.isUnsignedNum()){
			unsigned int crOld = s->credit.toUInt();
			crOld += ((crAdd / g_iRatioTake) * g_iRatioGive);
			s->credit.setNum(crOld);
			KviStr fName = kvirc_plugin_param(cmd,5);
			int idx = fName.findLastIdx('/');
			if(idx >= 0)fName.cutLeft(idx + 1);
			KviStr tmp(KviStr::Format,"I have received succesfully the file '%s' , %s bytes long",cmd->params->at(5)->ptr(),cmd->params->at(6)->ptr());
			s->pWnd->output(KVI_OUT_OWN,"[fserve >> %s] %s",s->nick.ptr(),tmp.ptr());
			tmp.prepend("[fserve] ");
			((KviDccChat *)(s->pWnd))->sendData(tmp.ptr());
			tmp.sprintf("Your credit is now %s bytes",s->credit.ptr());
			s->pWnd->output(KVI_OUT_OWN,"[fserve >> %s] %s",s->nick.ptr(),tmp.ptr());
			tmp.prepend("[fserve] ");
			((KviDccChat *)(s->pWnd))->sendData(tmp.ptr());
		}
	} else {
		KviSavedSession * ss = fserve_findSavedSession(kvirc_plugin_param(cmd,1),kvirc_plugin_param(cmd,4));
		if(ss){
			unsigned int crOld = ss->credit.toUInt(&bOk);
			if(!bOk)crOld = 0;
			crOld += ((crAdd / g_iRatioTake) * g_iRatioGive);
			ss->credit.setNum(crOld);
		}
		// We could also ack the dcc send completion to that user...
	}
	return false;
}

void fserve_plugin_config()
{
	if(!g_pConfigDialog){
		g_pConfigDialog = new KviFServeConfigDialog();
		g_pConfigDialog->show();
	}
}

bool fserve_plugin_command_fserve(KviPluginCommandStruct *cmd)
{
	if(!g_bServerActive){
		cmd->window->output(KVI_OUT_INTERNAL,__tr("[fserve] The file server is not active"));
		return true;
	}
	if(cmd->params->count() < 2){
		cmd->error = KVI_ERROR_MissingParameter;
		return false;
	}
	if(kvi_strEqualCI(kvirc_plugin_param(cmd,1),"list")){
		int total = 0;
		for(KviFServeSession * s = g_pSessions->first();s;s=g_pSessions->next()){
			cmd->window->output(KVI_OUT_INTERNAL,__tr("%cSession: %s@%s"),KVI_TEXT_BOLD,s->nick.ptr(),s->address.ptr());
			cmd->window->output(KVI_OUT_INTERNAL,__tr("Current credit: %s"),s->credit.ptr());
			cmd->window->output(KVI_OUT_INTERNAL,__tr("Current directory: %s"),s->currentDir.ptr());
			total++;
		}
		cmd->window->output(KVI_OUT_INTERNAL,__tr("[fserve] Total : %d running sessions"),total);
	} else if(kvi_strEqualCI(kvirc_plugin_param(cmd,1),"boot")){
		if(cmd->params->count() < 3){
			cmd->error = KVI_ERROR_MissingParameter;
			cmd->errorstr = __tr("The 'boot' operation requires a target (<nick>@<address>)");
			return false;
		}
		KviStr address = kvirc_plugin_param(cmd,2);
		KviStr nick = address.getToken('@');
		KviFServeSession * s = fserve_findRunningSessionByNickAndIp(nick.ptr(),address.ptr());
		if(!s){
			cmd->error = KVI_ERROR_InvalidParameter;
			cmd->errorstr = __tr("Session not found");
			return false;
		}
		s->pWnd->output(KVI_OUT_OWN,"[fserve >> %s] Your session has been closed by the system administrator.",s->nick.ptr());
		((KviDccChat *)(s->pWnd))->sendData("[fserve] Your session has been closed by the system administrator.");
		fserve_closeSession(s);
		cmd->window->output(KVI_OUT_INTERNAL,__tr("[fserve] Session closed (%s@%s)"),nick.ptr(),address.ptr());
	} else if(kvi_strEqualCI(kvirc_plugin_param(cmd,1),"credit")){
		if(cmd->params->count() < 4){
			cmd->error = KVI_ERROR_MissingParameter;
			cmd->errorstr = __tr("The 'credit' operation requires a target (<nick>@<address>) and a credit value");
			return false;
		}
		KviStr address = kvirc_plugin_param(cmd,2);
		KviStr nick = address.getToken('@');
		KviFServeSession * s = fserve_findRunningSessionByNickAndIp(nick.ptr(),address.ptr());
		if(!s){
			cmd->error = KVI_ERROR_InvalidParameter;
			cmd->errorstr = __tr("Session not found");
			return false;
		}
		KviStr credit = kvirc_plugin_param(cmd,3);
		if(!credit.isUnsignedNum()){
			if(!kvi_strEqualCI(credit.ptr(),"unlimited")){
				cmd->error = KVI_ERROR_InvalidParameter;
				cmd->errorstr = __tr("The credit must be an unsigned integer or the word 'unlimited'");
				return false;
			}
		}
		s->credit = credit;
		KviStr tmp(KviStr::Format,"Your credit was reset by the system administrator to %s bytes",credit.ptr());
		s->pWnd->output(KVI_OUT_OWN,"[fserve >> %s] %s",s->nick.ptr(),tmp.ptr());
		tmp.prepend("[fserve] ");
		((KviDccChat *)(s->pWnd))->sendData(tmp.ptr());
		cmd->window->output(KVI_OUT_INTERNAL,__tr("[fserve] Credit for session %s@%s succesfully set to %s"),nick.ptr(),address.ptr(),credit.ptr());
	} else if(kvi_strEqualCI(kvirc_plugin_param(cmd,1),"config")){
		fserve_plugin_config();
		return true;
	} else {
		cmd->error = KVI_ERROR_InvalidOperation;
		cmd->errorstr = __tr("Available operations are : 'list' , 'boot' and 'credit'");
		return false;
	}
	return true;
}

// init routine
bool fserve_plugin_init(KviPluginCommandStruct * cmd)
{
	// Initialize globals
	g_pSessions = new QList<KviFServeSession>;
	g_pSessions->setAutoDelete(true);
	g_pBannedIpList = new QList<KviStr>;
	g_pBannedIpList->setAutoDelete(true);
	g_pSavedSessions = new QList<KviSavedSession>;
	g_pSavedSessions->setAutoDelete(true);
	g_pPendingSessions = new QList<KviPendingSession>;
	g_pPendingSessions->setAutoDelete(true);
	g_handle = cmd->handle;
	// Load configuration
	fserve_pluginLoadState();

	kvirc_plugin_register_command(cmd->handle,"FSERVE",fserve_plugin_command_fserve);

	if(g_bServerActive){
		kvirc_plugin_add_hook(g_handle,KviEvent_OnDccChatConnected,fserve_plugin_hook_onDccChatConnected);
		kvirc_plugin_add_hook(g_handle,KviEvent_OnDccChatTerminated,fserve_plugin_hook_onDccChatTerminated);
		kvirc_plugin_add_hook(g_handle,KviEvent_OnDccChatMessage,fserve_plugin_hook_onDccChatMessage);
		kvirc_plugin_add_hook(g_handle,KviEvent_OnDccGetTransferComplete,fserve_plugin_hook_onDccGetTransferComplete);
		if(g_bListenToPrivmsg){
			kvirc_plugin_add_hook(g_handle,KviEvent_OnMePrivateMessage,fserve_plugin_hook_onMePrivateMessage);
		}
	}

	return true;
}

void fserve_plugin_cleanup()
{
	fserve_pluginSaveState();

	if(g_pConfigDialog)delete g_pConfigDialog; //we're unloaded!
	delete g_pSessions;
	delete g_pBannedIpList;
	delete g_pSavedSessions;
	delete g_pPendingSessions;

	kvirc_plugin_unregister_meta_object("KviFServeConfigDialog");
}

void fserve_configFinished(bool bCommit)
{
	if(bCommit){
		// Ensure that the pass is correctly assigned
		g_szFServePass.stripWhiteSpace();
		// check the root directory , and if some check fails , disable the server
		g_bServerActive = fserve_checkRoot() && g_bServerActive;

		g_szInitialCredit.stripWhiteSpace();
		if(!g_szInitialCredit.isUnsignedNum()){
			if(!kvi_strEqualCI(g_szInitialCredit.ptr(),"unlimited")){
				debug("Initial credit has a syntax error inside...setting to 0");
				g_szInitialCredit = "0";
			}
		}

		if(!g_bServerActive){
			// The server has been disactivated...kill all the sessions
			for(KviFServeSession *s=g_pSessions->first();s;s=g_pSessions->next()){
				s->pWnd->output(KVI_OUT_OWN,"[fserve >> %s] The file service has been disactivated: closing your session.",s->nick.ptr());
				((KviDccChat *)(s->pWnd))->sendData("[fserve] The file service has been disactivated: closing your session.");
			}
			while(g_pSessions->first())fserve_closeSession(g_pSessions->first());
			while(g_pPendingSessions->first())g_pPendingSessions->removeFirst();
			// remove the hooks (avoid unnecessary lag to KVIrc)
			kvirc_plugin_remove_all_hooks(g_handle);
		} else {
			// The root may be changed...warp all sessions to '/'
			for(KviFServeSession *s=g_pSessions->first();s;s=g_pSessions->next()){
				s->pWnd->output(KVI_OUT_OWN,"[fserve >> %s] The file service has been reconfigured: warping you to '/'.",s->nick.ptr());
				((KviDccChat *)(s->pWnd))->sendData("[fserve] The file service has been reconfigured: warping you to '/'.");
				s->currentDir = "/";
			}
			// If the server was previously disabled , reinstall the hooks
			if(!kvirc_plugin_is_hook_registered(g_handle,KviEvent_OnDccChatConnected)){
				// no previous hooks (the server was disabled)
				kvirc_plugin_add_hook(g_handle,KviEvent_OnDccChatConnected,fserve_plugin_hook_onDccChatConnected);
				kvirc_plugin_add_hook(g_handle,KviEvent_OnDccChatTerminated,fserve_plugin_hook_onDccChatTerminated);
				kvirc_plugin_add_hook(g_handle,KviEvent_OnDccChatMessage,fserve_plugin_hook_onDccChatMessage);
				kvirc_plugin_add_hook(g_handle,KviEvent_OnDccGetTransferComplete,fserve_plugin_hook_onDccGetTransferComplete);
				if(g_bListenToPrivmsg){
					kvirc_plugin_add_hook(g_handle,KviEvent_OnMePrivateMessage,fserve_plugin_hook_onMePrivateMessage);
				}
			} else {
				// the main hooks are already installed
				if(g_bListenToPrivmsg){
					// Check if we have to install the hook or it is already done
					if(!kvirc_plugin_is_hook_registered(g_handle,KviEvent_OnMePrivateMessage)){
						// This hook was not previously registered
						kvirc_plugin_add_hook(g_handle,KviEvent_OnMePrivateMessage,fserve_plugin_hook_onMePrivateMessage);
					}
				} else {
					// Check if we have to remove the hook
					if(kvirc_plugin_is_hook_registered(g_handle,KviEvent_OnMePrivateMessage)){
						// This hook was not previously registered
						kvirc_plugin_remove_hook(g_handle,KviEvent_OnMePrivateMessage);
					}
				}
			}
		}
	}
	delete g_pConfigDialog;
	g_pConfigDialog = 0;
}

/*
	@document: doc_plugin_fserve.kvihelp
	@title: The KVIrc FServe plugin
		The KVIrc FServe plugin is a simple implementation of a file transfer
		service over a DCC connnection.<br>
		<docsubtitle>Directory tree</docsubtitle>
		The file server works in its own directory tree , which root
		directory path can be specified in the settings dialog.<br>
		Any user that access the server will see the root directory as '/'
		and will be able to discend the entire underlying tree by using the '.cd' command.<br>
		He will be also able to list the directory contents with the '.ls' command.<br>
		Obviously all this assuming that the user that is running the fserve
		has access to that directory tree.<br>
		<docsubtitle>Access</docsubtitle>
		There are two kind of accesses:<br>
		<b>- Public access</b><br>
		It is optionally protected by a global fserve password.<br>
		Any remote user can access the file service by typing '.fserve [password]'
		in a running DCC CHAT connection.<br>
		If no global password has been set , the service will be accessible by simply
		typing '.fserve'.<br>
		The user will obtain the global initial credit (specified in the settings dialog)
		that can be optionally 'unlimited'.<br>
		The user will be able to download files from the fserve directory tree
		by typing '.get &lt;filename&gt;' , where &lt;filename&gt; is a file contained in
		its current working directory (i.e. the file name can NOT contain a path prefix).<br>
		The download will be allowed only if the user's credit is greater or equal
		than the requested file size.<br>
		At the transfer start moment , the size of the file (in bytes) will be subtracted
		from the user's credit (unless the credit is 'unlimited').<br>
		Users can also upload files by simply sending it by DCC.<br>
		At the end of the upload , the transfered file size multiplied for the fserve ratio
		(configurable in the settings dialog) will be added to the user's credit;<br>
		either if the user is still using the fserver or not.<br>
		The last 32 'Limited credit' sessions will be saved in memory
		until the file service is stopped (i.e. the plugin is unloaded, the file service
		disactivated or kvirc application is terminated).<br>
		<b>- Registered access</b><br>
		Users registered in the "KVIrc users database" with the 'f' flag can access
		the file service and obtain 'unlimited' credit access by typing '.fserve [user_password]'.<br>
		If the user has no password set , he will be able to access the file server by simply
		typing '.fserve'.<br>
		<docsubtitle>Restricting access</docsubtitle>
		- The file server supports 'bans' on ip addresses.<br>
		You can add a list of ip address masks (like 127.0.*.*) that will be rejected
		- There is a configurable limit on the number of running sessions.<br>
		When the server is 'full' any login will be rejected.<br>
		<docsubtitle>Other features</docsubtitle>
		- Optionally the file server can react to requests through private messages.<br>
		In that case any previously described access can be obtained by typing
		'/msg &lt;your_nickname&gt; !fserve [password]'.<br>
		The fserve will open a DCC CHAT connection and enter the file server mode automatically.<br>
		To enable this feature you must check the appropriate option in the plugin configuration dialog.<br>
		- A configurable 'message of the day' can be displayed to the user at login time
		and can be later accessed by using the '.motd' command.<br>
		- The user is able to list available commands by using the '.help' command.<br>
		<docsubtitle>Local commands</docsubtitle>
		<br><b>/fserve list</b><br><br>
		Lists active user sessions.<br>
		<br><b>/fserve boot &lt;nickname@ipaddress&gt;</b><br><br>
		Terminates a running file service.<br>
		A similar result can be achieved by simply closing the DCC chat connection.<br>
		NOTE : /fserve boot will NOT close the DCC connection , it will just close the fserve session.<br>
		<br><b>/fserve credit &lt;nickname@ipaddress&gt; &lt;creditvalue&gt;</b><br><br>
		Sets the credit of a specified session to &lt;creditvalue&gt; that can be a positive
		numeric value or the string 'unlimited'.<br>
		<br><b>/fserve config</b><br>
		Opens the plugin configuration dialog.<br>
*/

void fserve_plugin_help()
{
	kvirc_plugin_show_help_page("doc_plugin_fserve.kvihelp");
}

// plugin definition structure
// MUST be named 'kvirc_plugin'
KviPlugin kvirc_plugin =
{
	"fserve" ,                                    // plugin name
	"KVIrc DCC file server" ,                     // description
	"1.0" ,                                       // version
	"Szymon Stefanek <stefanek@tin.it>" ,         // author info
	"Allows remote users to browse a local " \
	"file archive in an ftp-like manner.\n" \
	"The users can download files and a " \
	"downloaded/uploaded bytes ratio can be set." ,
	fserve_plugin_init ,                          // init routine
	fserve_plugin_cleanup ,                       // cleanup routine
	fserve_plugin_config ,                        // configuration routine
	fserve_plugin_help
};


// =======================================
// Config dialog
// =======================================

#include <qlayout.h>
#include <qframe.h>

#include "kvi_selectors.h"

KviFServeConfigDialog::KviFServeConfigDialog()
:QTabDialog()
{
	setCaption(__tr("File server configuration"));

	QWidget *tab = new QWidget(this);
	QGridLayout *g = new QGridLayout(tab,5,1,10,4);

	KviNewBoolSelector *b=new KviNewBoolSelector(tab,__tr("Activate file server"),&(g_bServerActive),true);
	g->addWidget(b,0,0);

	QFrame *r = new QFrame(tab);
	r->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addWidget(r,1,0);

	QLabel *l = new QLabel(
		__tr("A misconfigured file server may allow remote users to download " \
		"unwanted files from your filesystem.\n" \
		"Please check the settings twice before enabling the service.\n" \
		"It is also not reccomended to leave the service " \
		"running \"alone\" while you are away.\nHave fun! :)"),tab);
	l->setAlignment(AlignCenter|WordBreak);
	l->setEnabled(false);
	g->addWidget(l,2,0);

	r = new QFrame(tab);
	r->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addWidget(r,3,0);

	KviNewStringSelector *s=new KviNewStringSelector(tab,__tr("File server root "),&(g_szFServeRoot),g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),s,SLOT(setEnabled(bool)));
	g->addWidget(s,4,0);

	g->setRowStretch(2,2);

	addTab(tab,__tr("Main"));

	tab = new QWidget(this);
	g = new QGridLayout(tab,7,1,10,4);

	s=new KviNewStringSelector(tab,__tr("Password"),&(g_szFServePass),g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),s,SLOT(setEnabled(bool)));
	g->addMultiCellWidget(s,0,0,0,4);

	s=new KviNewStringSelector(tab,__tr("Initial credit"),&(g_szInitialCredit),g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),s,SLOT(setEnabled(bool)));
	g->addMultiCellWidget(s,1,1,0,4);

	r = new QFrame(tab);
	r->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addMultiCellWidget(r,2,2,0,4);

	l = new QLabel(__tr("Give "),tab);
	l->setEnabled(g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),l,SLOT(setEnabled(bool)));
	g->addWidget(l,3,0);

	m_pRatioGiveEdit = new QLineEdit(tab);
	m_pRatioGiveEdit->setEnabled(g_bServerActive);
	KviStr tmp;
	tmp.setNum(g_iRatioGive);
	m_pRatioGiveEdit->setText(tmp.ptr());
	connect(b,SIGNAL(toggled(bool)),m_pRatioGiveEdit,SLOT(setEnabled(bool)));
	g->addWidget(m_pRatioGiveEdit,3,1);

	l = new QLabel(__tr(" byte(s) for each block of "),tab);
	l->setEnabled(g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),l,SLOT(setEnabled(bool)));
	l->setMinimumSize(l->sizeHint());
	g->addWidget(l,3,2);

	m_pRatioTakeEdit = new QLineEdit(tab);
	m_pRatioTakeEdit->setEnabled(g_bServerActive);
	tmp.setNum(g_iRatioTake);
	m_pRatioTakeEdit->setText(tmp.ptr());
	connect(b,SIGNAL(toggled(bool)),m_pRatioTakeEdit,SLOT(setEnabled(bool)));
	g->addWidget(m_pRatioTakeEdit,3,3);

	l = new QLabel(__tr(" byte(s) received"),tab);
	l->setEnabled(g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),l,SLOT(setEnabled(bool)));
	l->setMinimumSize(l->sizeHint());
	g->addWidget(l,3,4);

	r = new QFrame(tab);
	r->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	r->setMinimumHeight(r->frameWidth() * 2);
	g->addMultiCellWidget(r,4,4,0,4);

	KviNewBoolSelector * b1=new KviNewBoolSelector(tab,__tr("Listen to PRIVMGS <mynick> !fserve [password]"),&(g_bListenToPrivmsg),g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),b1,SLOT(setEnabled(bool)));
	g->addMultiCellWidget(b1,5,5,0,4);

	KviNewIntegerSelector * i1 = new KviNewIntegerSelector(tab,__tr("Max running sessions "),&(g_uMaxRunningSessions),KviNewIntegerSelector::UInt,
		0,65535,1,g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),i1,SLOT(setEnabled(bool)));
	g->addMultiCellWidget(i1,6,6,0,4);

	addTab(tab,__tr("Public access"));

	tab = new QWidget(this);
	g = new QGridLayout(tab,2,1,10,4);

	b1=new KviNewBoolSelector(tab,__tr("Show motd after login"),&(g_bShowMotdAtLogin),g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),b1,SLOT(setEnabled(bool)));
	g->addWidget(b1,0,0);

	m_pMotdEdit = new QMultiLineEdit(tab);
	m_pMotdEdit->setEnabled(g_bServerActive);
	m_pMotdEdit->setText(g_szMotd.ptr());
	connect(b,SIGNAL(toggled(bool)),m_pMotdEdit,SLOT(setEnabled(bool)));
	g->addWidget(m_pMotdEdit,1,0);
	g->setRowStretch(1,2);

	addTab(tab,__tr("Motd"));

	tab = new QWidget(this);
	g = new QGridLayout(tab,1,3,10,4);

	m_pBannedIpEdit = new QLineEdit(tab);
	m_pBannedIpEdit->setEnabled(g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),m_pBannedIpEdit,SLOT(setEnabled(bool)));
	g->addWidget(m_pBannedIpEdit,0,0);

	QPushButton * btn = new QPushButton(__tr("<-- Add"),tab);
	btn->setEnabled(g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),btn,SLOT(setEnabled(bool)));
	connect(btn,SIGNAL(clicked()),this,SLOT(addBannedIp()));
	g->addWidget(btn,0,1);

	btn = new QPushButton(__tr("Remove selected"),tab);
	btn->setEnabled(g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),btn,SLOT(setEnabled(bool)));
	connect(btn,SIGNAL(clicked()),this,SLOT(removeBannedIp()));
	g->addWidget(btn,0,2);

	m_pBannedIpList = new QListBox(tab);
	m_pBannedIpList->setEnabled(g_bServerActive);
	connect(b,SIGNAL(toggled(bool)),m_pBannedIpList,SLOT(setEnabled(bool)));
	g->addMultiCellWidget(m_pBannedIpList,1,1,0,2);

	for(KviStr * s=g_pBannedIpList->first();s;s=g_pBannedIpList->next()){
		m_pBannedIpList->insertItem(s->ptr());
	}

	g->setRowStretch(1,2);

	addTab(tab,__tr("Banned IP addresses"));

	setCancelButton(__tr("Cancel"));

}

KviFServeConfigDialog::~KviFServeConfigDialog()
{
}

void KviFServeConfigDialog::done(int r)
{
	QTabDialog::done(r);
	if(r == Accepted){
		KviNewBoolSelector::commitAll(this);
		KviNewStringSelector::commitAll(this);
		KviNewIntegerSelector::commitAll(this);
		KviStr tmp = m_pRatioTakeEdit->text();
		tmp.stripWhiteSpace();
		bool bOk = false;
		g_iRatioTake = tmp.toInt(&bOk);
		if((!bOk) || (g_iRatioTake < 1))g_iRatioTake = 1;
		tmp = m_pRatioGiveEdit->text();
		tmp.stripWhiteSpace();
		bOk = false;
		g_iRatioGive = tmp.toInt(&bOk);
		if((!bOk) || (g_iRatioGive < 0))g_iRatioGive = 1;
		g_szMotd = m_pMotdEdit->text();

		while(g_pBannedIpList->first())g_pBannedIpList->removeFirst();
		unsigned int count = m_pBannedIpList->count();
		for(unsigned int i=0;i<count;i++){
			tmp = m_pBannedIpList->text(i);
			tmp.stripWhiteSpace();
			if(tmp.hasData())g_pBannedIpList->append(new KviStr(tmp.ptr()));
		}
	}
	fserve_configFinished((r == Accepted));
}

void KviFServeConfigDialog::closeEvent(QCloseEvent *)
{
	// Finished without committing
	fserve_configFinished(false);
}

void KviFServeConfigDialog::addBannedIp()
{
	KviStr tmp = m_pBannedIpEdit->text();
	tmp.stripWhiteSpace();
	if(tmp.hasData())m_pBannedIpList->insertItem(tmp.ptr());
}

void KviFServeConfigDialog::removeBannedIp()
{
	int idx = m_pBannedIpList->currentItem();
	if(idx != -1)m_pBannedIpList->removeItem(idx);
}

#include "m_libkvifserve.moc"
