#include "editor.h"

#undef Unsorted

#include <qframe.h>
#include <qpainter.h>
#include <qfontmetrics.h>
#include <qfile.h>
#include <qnamespace.h>
#include <qcursor.h>
#include <qclipboard.h>
#include <qapplication.h>
#include <qfiledialog.h>
#include <qfileinfo.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <qtoolbutton.h>
#include <qlabel.h>
#include <qaccel.h>

#include <string.h> //for memmove()
#include <ctype.h>

#define __DEBUG


#ifdef __DEBUG
	#define __assert(_expr) if(! (_expr) )if(!(_expr))debug("RANGE ASSERT : \"%s\" is false in %s (%d)",#_expr,__FILE__,__LINE__);
#else
	#define __assert(_expr)
#endif

//#include <X11/X.h>
#ifdef COMPILE_USE_AA_FONTS
	extern XftFont        * g_pXftFont;
	extern XftDraw        * g_pXftDraw;
	extern int qt_use_xft (void); // qpainter_x11.cpp
	extern void *qt_ft_font (const QFont *f); // qfont_x11.cpp
	extern XftDraw * qt_lookup_ft_draw (Drawable draw, bool paintEventClipOn, QRegion *crgn);
#endif

SSEXFindWidget::SSEXFindWidget(SSEXEditor * parent)
:QFrame(parent)
{
	setFrameStyle(QFrame::WinPanel | QFrame::Raised);
	m_pEditor = parent;
	setCursor(arrowCursor);

	QGridLayout *g = new QGridLayout(this,13,3,4,0);
	QToolButton *tb = new QToolButton(DownArrow,this);
	tb->setFixedSize(12,12);
	tb->setAutoRepeat(false);
	connect(tb,SIGNAL(clicked()),m_pEditor,SLOT(toggleFindWidget()));
	g->addWidget(tb,0,2);

	QFrame *f = new QFrame(this);
	f->setFrameStyle(QFrame::HLine | QFrame::Sunken);
	g->addMultiCellWidget(f,1,1,0,2);

	QLabel *l = new QLabel("String to find:",this);
	g->addMultiCellWidget(l,2,2,0,2);

	m_pFindStringEdit = new QLineEdit(this);
	g->addMultiCellWidget(m_pFindStringEdit,3,3,0,2);
	
	setFocusProxy(m_pFindStringEdit);

	m_pRegExpCheckBox = new QCheckBox("Regular expression",this);
//	connect(m_pRegExpCheckBox,SIGNAL(toggled(bool)),this,SLOT(regExpCheckBoxToggled(bool)));
	g->addMultiCellWidget(m_pRegExpCheckBox,4,4,0,2);

	m_pCaseSensitiveCheckBox = new QCheckBox("Case sensitive",this);
	m_pCaseSensitiveCheckBox->setChecked(true);
//	connect(m_pCaseSensitiveCheckBox,SIGNAL(toggled(bool)),this,SLOT(regExpCheckBoxToggled(bool)));
	g->addMultiCellWidget(m_pCaseSensitiveCheckBox,5,5,0,2);


	QPushButton *b = new QPushButton("Find &Next",this);
	b->setDefault(true);
	g->addWidget(b,6,0);
	connect(b,SIGNAL(clicked()),this,SLOT(findNextClicked()));

	b = new QPushButton("Find &Prev",this);
	g->addMultiCellWidget(b,6,6,1,2);
	connect(b,SIGNAL(clicked()),this,SLOT(findPrevClicked()));

	f = new QFrame(this);
	f->setFrameStyle(QFrame::HLine | QFrame::Sunken);
	g->addMultiCellWidget(f,7,7,0,2);

	l = new QLabel("Replacement text:",this);
	g->addMultiCellWidget(l,8,8,0,2);

	m_pReplaceStringEdit = new QLineEdit(this);
	g->addMultiCellWidget(m_pReplaceStringEdit,9,9,0,2);

	m_pReplace = new QPushButton("&Replace",this);
	connect(m_pReplace,SIGNAL(clicked()),m_pEditor,SLOT(replace()));
	g->addWidget(m_pReplace,10,0);
	m_pReplace->setEnabled(m_pEditor->hasSelection());

	b = new QPushButton("Replace &All",this);
	connect(b,SIGNAL(clicked()),this,SLOT(replaceAllClicked()));
	g->addMultiCellWidget(b,10,10,1,2);

	m_pReplaceAndFindNext = new QPushButton("R&eplace Next",this);
	connect(m_pReplaceAndFindNext,SIGNAL(clicked()),this,SLOT(replaceAndFindNextClicked()));
	g->addMultiCellWidget(m_pReplaceAndFindNext,11,11,0,2);
	m_pReplaceAndFindNext->setEnabled(m_pEditor->hasSelection());

	m_pReplaceAllInSelection = new QPushButton("Replace in &Selection",this);
	connect(m_pReplaceAllInSelection,SIGNAL(clicked()),this,SLOT(replaceAllInSelectionClicked()));
	g->addMultiCellWidget(m_pReplaceAllInSelection,12,12,0,2);
	m_pReplaceAllInSelection->setEnabled(m_pEditor->hasSelection());


	g->setResizeMode(QGridLayout::Fixed);
}

SSEXFindWidget::~SSEXFindWidget()
{
}

void SSEXFindWidget::findNextClicked()
{
	if(m_pRegExpCheckBox->isChecked())m_pEditor->findNextRegExp();
	else m_pEditor->findNext();
}

void SSEXFindWidget::findPrevClicked()
{
	if(m_pRegExpCheckBox->isChecked())m_pEditor->findPrevRegExp();
	else m_pEditor->findPrev();
}

void SSEXFindWidget::replaceAllClicked()
{
	if(m_pRegExpCheckBox->isChecked())m_pEditor->replaceAllRegExp();
	else m_pEditor->replaceAll();
}

void SSEXFindWidget::replaceAndFindNextClicked()
{
	if(m_pRegExpCheckBox->isChecked())m_pEditor->replaceAndFindNextRegExp();
	else m_pEditor->replaceAndFindNext();
}

void SSEXFindWidget::replaceAllInSelectionClicked()
{
	if(m_pRegExpCheckBox->isChecked())m_pEditor->replaceAllInSelectionRegExp();
	else m_pEditor->replaceAllInSelection();
}

void SSEXFindWidget::mousePressEvent(QMouseEvent *e)
{
	m_pressPoint = e->pos();
}

void SSEXFindWidget::mouseMoveEvent(QMouseEvent *)
{
	QPoint p=m_pEditor->mapFromGlobal(QCursor::pos());
	p-=m_pressPoint;
	if(p.x() < 0)p.setX(0);
	else if((p.x() + width()) > m_pEditor->width())p.setX(m_pEditor->width() - width());
	if(p.y() < 0)p.setY(0);
	else if((p.y() + height()) > m_pEditor->height())p.setY(m_pEditor->height() - height());
	move(p);
}


#define SSEX_EDITOR_BORDER 5
#define SSEX_EDITOR_BLINK_TIME 500
#define SSEX_EDITOR_DRAG_TIMEOUT 150

SSEXEditor::SSEXEditor(QWidget * parent)
:QTableView(parent)
{
	setFrameStyle(QFrame::Panel|QFrame::Sunken);
	setNumRows(1);
	setNumCols(1);

	setCursor(ibeamCursor);

	m_pColors = new SSEXEditorColors;

	m_pColors->background = QColor(0,0,0);
	m_pColors->extBackground = QColor(40,40,40);
	m_pColors->normalText = QColor(50,255,0);
	m_pColors->cursor     = QColor(255,0,0);

	m_pColors->font = QFont("helvetica");

	m_pColors->cppBackground = QColor(0,0,0);
	m_pColors->cppExtBackground = QColor(40,40,40);
	m_pColors->cppNormalText = QColor(80,255,0);
	m_pColors->cppCursor     = QColor(255,0,0);
	m_pColors->cppBrace      = QColor(255,0,0);
	m_pColors->cppLineComment    = QColor(40,150,0);
	m_pColors->cppMultilineComment = QColor(20,180,0);
	m_pColors->cppParenthesis = QColor(170,130,30);
	m_pColors->cppOperator   = QColor(150,150,40);
	m_pColors->cppEscape = QColor(50,130,240);
	m_pColors->cppChar = QColor(100,140,250);
	m_pColors->cppString     = QColor(80,170,250);
	m_pColors->cppKeyword    = QColor(130,130,130);
	m_pColors->cppType       = QColor(160,160,160);
	m_pColors->cppNumber     = QColor(190,200,255);
	m_pColors->cppPunctuation = QColor(180,180,50);
	m_pColors->cppSystemIdentifier = QColor(255,0,255);
	m_pColors->cppPreprocessor = QColor(255,255,255);
	m_pColors->cppInclude = QColor(200,200,200);
	m_pColors->cppMemberVariable = QColor(190,170,80);
	m_pColors->cppGlobalVariable = QColor(230,200,110);

	m_pColors->cppQClass = QColor(255,255,50);
	m_pColors->cppQSignals = QColor(255,150,0);

	m_pColors->cppKClass = QColor(255,255,0);
	m_pColors->cppXStuff = QColor(255,255,90);
//	m_pColors->cppXStuff = QColor(255,207,0);

	m_pColors->cppGtkStruct = QColor(255,255,50);
	m_pColors->cppGdkStruct = QColor(255,205,90);
	m_pColors->cppGType = QColor(190,190,190);
	m_pColors->cppGtkCall = QColor(150,150,180);
	m_pColors->cppGdkCall = QColor(150,120,180);
	m_pColors->cppGtkMacro = QColor(220,170,180);
	m_pColors->cppGdkMacro = QColor(230,170,160);
	m_pColors->cppSpecial = QColor(150,150,150);
//	m_pColors->cppSysCall = QColor(255,50,0);
//	m_pColors->cppSignals = QColor(110,120,180);


	m_pColors->htmlBackground = QColor(0,0,0);
	m_pColors->htmlExtBackground = QColor(40,40,40);
	m_pColors->htmlNormalText = QColor(20,255,20);
	m_pColors->htmlCursor = QColor(255,0,0);
	m_pColors->htmlComment = QColor(35,180,0);
	m_pColors->htmlTag = QColor(180,100,30);
	m_pColors->htmlString = QColor(40,180,255);
	m_pColors->htmlTagInternal = QColor(180,150,20);

	m_pColors->htmlFont = QFont("clean");

	m_pColors->cppFont = QFont("clean");

	m_pLines = new QList<SSEXEditorTextLine>;
	m_pLines->setAutoDelete(true);

	m_mode = Normal;
	m_iTabsNumSpaces = 4;

	m_iCursorRow              = 0;
	m_iCursorPosition         = 0;
	m_iCursorPositionInPixels = 0;
	m_iLastCursorRow          = 0;
	m_iLastCursorPosition     = 0;

	m_iMaxTextWidth = 0;
	m_pMemBuffer = new QPixmap(cellWidth() + 1,cellHeight() + 1);
	m_pBlinkTimer = new QTimer();
	m_bCursorOn = false;
	m_bOverwrite = false;

	m_bRecordingKeystrokes = false;
	m_pKeystrokes = new QList<SSEXEditorKeystroke>;
	m_pKeystrokes->setAutoDelete(true);

	m_bHasSelection = false;

	m_pContextPopup = new QPopupMenu();
	m_pFindPopup = new QPopupMenu();
//	m_pFindEdit = 0;
//	m_szTextToFind = "";


	connect(m_pBlinkTimer,SIGNAL(timeout()),this,SLOT(blinkTimer()));

	m_bModified = false;
	m_pDragTimer = new QTimer();
	connect(m_pDragTimer,SIGNAL(timeout()),this,SLOT(dragTimer()));

	setTableFlags(Tbl_autoScrollBars | Tbl_smoothScrolling);

	updateFontDependantVariables();
	updateCellSize();

	setBackgroundMode(NoBackground);

	setFocusPolicy(QWidget::StrongFocus);

	m_iBlinkTime = SSEX_EDITOR_BLINK_TIME;
	m_szFileName = "";

	m_pFindWidget = new SSEXFindWidget(this);
	m_pFindWidget->hide();

	m_lastFindWidgetPosition = QPoint(20,20);

	initializeCurrentMode();
	setText("");

}

SSEXEditor::~SSEXEditor()
{
	closeFile(); // cancel does nothing here

	delete m_pFindWidget;
	delete m_pColors;
	delete m_pLines;
	delete m_pMemBuffer;
	delete m_pBlinkTimer;
	delete m_pDragTimer;
	delete m_pKeystrokes;
	delete m_pContextPopup;
	delete m_pFindPopup;
}

bool SSEXEditor::closeFile()
{
	if(m_bModified){
		QString msg;
		msg.sprintf("The file %s has been modified\nDo you wish to save your changes?",m_szFileName.isEmpty() ? "Untitled" : m_szFileName.data());
		int ret = QMessageBox::warning(this,"Warning",msg,"&Save","&Don't Save","&Cancel");
		if(ret == 0){
			if(!saveFile())return closeFile(); //cancelled after this...repeat plz...
			else return true; //saved
		} else if(ret == 1){
			m_bModified = false; // don't call closeFile in destructor
			return true;
		} else return false; //cancelled;
	}
	return true; // safe to close
}

void SSEXEditor::setColors(SSEXEditorColors *clrs)
{
	if(clrs)
	{
		if(clrs != m_pColors)
		{
			delete m_pColors;
			m_pColors = clrs;
			initializeCurrentMode();
			update();
		}
	}
}

void SSEXEditor::toggleFindWidget()
{
	if(m_pFindWidget->isVisible())
	{
		m_lastFindWidgetPosition = m_pFindWidget->pos();
		m_pFindWidget->hide();
		setFocus();
	} else {
		if((m_lastFindWidgetPosition.x() + m_pFindWidget->width()) > width())m_lastFindWidgetPosition.setX(width() - m_pFindWidget->width());
		if((m_lastFindWidgetPosition.y() + m_pFindWidget->height()) > height())m_lastFindWidgetPosition.setY(height() - m_pFindWidget->height());
		m_pFindWidget->move(m_lastFindWidgetPosition);
		m_pFindWidget->show();
		m_pFindWidget->m_pFindStringEdit->setFocus();
	}
}

void SSEXEditor::setModified(bool bModified)
{
	if(bModified != m_bModified)
	{
		m_bModified = bModified;
		emit modifyStateChanged(this,m_bModified);
	}
}

void SSEXEditor::setHasSelection(bool bHasSelection)
{
	m_bHasSelection = bHasSelection;
	if(m_pFindWidget)
	{
		m_pFindWidget->m_pReplace->setEnabled(bHasSelection);
		m_pFindWidget->m_pReplaceAndFindNext->setEnabled(bHasSelection);
		m_pFindWidget->m_pReplaceAllInSelection->setEnabled(bHasSelection);
	}
}

void SSEXEditor::blinkTimer()
{
	m_bCursorOn = !m_bCursorOn;
	updateCell(m_iCursorRow,0,false);
}

void SSEXEditor::setBlinkTime(int blinkTime)
{
	if(blinkTime < 100)blinkTime = 100;
	if(blinkTime > 10000)blinkTime = 10000;
	m_iBlinkTime = 100;
	if(hasFocus())startBlinking();
}

void SSEXEditor::startBlinking()
{
	if(m_pBlinkTimer->isActive())m_pBlinkTimer->stop();
	m_bCursorOn = true;
	m_pBlinkTimer->start(m_iBlinkTime);	
}


void SSEXEditor::stopBlinking()
{
	if(m_pBlinkTimer->isActive())m_pBlinkTimer->stop();
	if(m_bCursorOn){
		m_bCursorOn = false;
		updateCell(m_iCursorRow,false);
	}
}

void SSEXEditor::focusInEvent(QFocusEvent *e)
{
	startBlinking();
	QTableView::focusInEvent(e);
	emit gainedFocus(this);
}

void SSEXEditor::focusOutEvent(QFocusEvent *e)
{
	stopBlinking();
	QTableView::focusOutEvent(e);
}

void SSEXEditor::setTabsNumSpaces(int spaces)
{
	if(spaces < 1)spaces = 1;
	if(spaces > 64)spaces = 64;
	m_iTabsNumSpaces = spaces;
	updateFontDependantVariables();
	updateMaxTextWidth();
	updateCellSize();
	recalcCursorPosition(m_pLines->at(m_iCursorRow));
	update();
}

void SSEXEditor::updateMaxTextWidth()
{
	m_iMaxTextWidth = 0;
	int line = 0;
	for(SSEXEditorTextLine * l=m_pLines->first();l;l=m_pLines->next())
	{
		if(l->width > m_iMaxTextWidth)
		{
			m_iMaxTextWidth = l->width;
			m_iMaxTextWidthLine = line;
		}
		line++;
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////
//
// PAINT
//
//////////////////////////////////////////////////////////////////////////////////////////////////


void SSEXEditor::paintCell(QPainter * p,int row,int)
{
	SSEXEditorTextLine * l = m_pLines->at(row);
	if(!l)return;

	__assert(((unsigned int)(l->text.length())) == ((unsigned int)(l->length)));

	switch(m_mode)
	{
		case Normal:
			paintCellNormal(p,l,row);
			break;
		case Cpp:
			paintCellCpp(p,l,row);
			break;
		case Html:
			paintCellHtml(p,l,row);
			break;
	}
}

void SSEXEditor::paintCellNormal(QPainter * p,SSEXEditorTextLine * l,int row)
{
	Display * dpy = x11Display();
	Drawable drw = m_pMemBuffer->handle();
	GC gc = XCreateGC(dpy,drw,0,0);
	QRect updateR = cellUpdateRect();

	XSetFillStyle(dpy,gc,FillSolid);

	XSetForeground(dpy,gc,m_pColors->background.pixel());
#ifdef COMPILE_USE_AA_FONTS
	if(qt_use_xft())
	{
		g_pXftFont = (XftFont *)qt_ft_font(&(font()));
		g_pXftDraw = qt_lookup_ft_draw (drw,false,0);
		if(!g_pXftDraw)
		{
			XSetFont(dpy,gc,font().handle());
			g_pXftFont = 0;
		}
		
	} else {
#endif
		XSetFont(dpy,gc,font().handle());
#ifdef COMPILE_USE_AA_FONTS
		g_pXftFont = 0;
		g_pXftDraw = 0;
	}
#endif

	XFillRectangle(dpy,drw,gc,updateR.x(),updateR.y(),updateR.width(),updateR.height());

	XSetForeground(dpy,gc,m_pColors->normalText.pixel());

	const char * c = l->text.data();
	int curXPos = SSEX_EDITOR_BORDER;
	int lastTabPos = SSEX_EDITOR_BORDER;

	while(*c)
	{
		if(*c == '\t')
		{
			// only a tab
			while(lastTabPos <= curXPos)lastTabPos += m_iTabsNumPixels;
			curXPos = lastTabPos;
			c++;
		} else {
			// text block
			const char * begin = c;
			int blockWidth = 0;
			while(*c && *c != '\t'){
				blockWidth += m_iCharWidth[(unsigned char)*c];
				c++;
			}
#ifdef COMPILE_USE_AA_FONTS
			if(g_pXftFont){
				XftColor color;
				QColor * clr = &(m_pColors->normalText);
				color.color.red = clr->red() | clr->red() << 8;
				color.color.green = clr->green() | clr->green() << 8;
				color.color.blue = clr->blue() | clr->blue() << 8;
				color.color.alpha = 0xffff;
				color.pixel = clr->pixel();
				XftDrawString8(g_pXftDraw,&color,g_pXftFont,curXPos,m_iFontAscent,(unsigned char *)begin,c - begin);
			} else
#endif
				XDrawString(dpy,drw,gc,curXPos,m_iFontAscent,begin,c - begin);
			curXPos += blockWidth;
		}
	}

	if(m_iCursorRow == row)paintCursor(dpy,drw,gc,&(m_pColors->cursor),l);
	if(m_bHasSelection)paintSelection(row,l,dpy,drw,gc,updateR);

	int col_xPos,row_yPos;

	p->worldMatrix().map(0,0,&col_xPos,&row_yPos);

	XCopyArea(dpy,drw,handle(),gc,updateR.x(),updateR.y(),updateR.width(),updateR.height(),col_xPos + updateR.x(),row_yPos + updateR.y());

	XFreeGC(dpy,gc);
}

void SSEXEditor::paintCursor(Display *dpy,Drawable drw,GC gc,QColor * pColor,SSEXEditorTextLine *l)
{
	// paint the cursor
	if((m_iCursorRow != m_iLastCursorRow) || (m_iCursorPosition != m_iLastCursorPosition))
	{
		m_iLastCursorRow = m_iCursorRow;
		m_iLastCursorPosition = m_iCursorPosition;
//		debug("Cursor pos = %d,%d",m_iCursorRow,m_iCursorPosition);
		emit cursorPositionChanged(this,m_iCursorRow,(m_iCursorPosition < l->length) ? m_iCursorPosition : l->length);
	}
	if(!m_bCursorOn)return;
	int curXPos = m_iCursorPositionInPixels + SSEX_EDITOR_BORDER - 1;
	XSetForeground(dpy,gc,pColor->pixel());
	int rowLowPos = cellHeight() - 1;
	XDrawLine(dpy,drw,gc,curXPos,0,curXPos,rowLowPos);
	XDrawLine(dpy,drw,gc,curXPos + 1,0,curXPos + 1,rowLowPos);

	XDrawLine(dpy,drw,gc,curXPos - 3,0,curXPos + 4,0);
	XDrawLine(dpy,drw,gc,curXPos - 3,rowLowPos,curXPos + 4,rowLowPos);
}

void SSEXEditor::paintSelection(int row,SSEXEditorTextLine *l,Display *dpy,Drawable drw,GC gc,QRect &updateR)
{
	// paint the selection

	if((m_selection1.y() <= row) && (m_selection2.y() >= row))
	{
		int selectionBeginX = SSEX_EDITOR_BORDER;
		int selectionEndX = SSEX_EDITOR_BORDER;
		if(m_selection1.y() == row)
		{
			// the selection begins in this row
			selectionBeginX += getTextWidthWithTabsForCursorPosition(l->text.data(),m_selection1.x());
		}
		if(m_selection2.y() == row)
		{
			// the selection ends in this row
			selectionEndX += getTextWidthWithTabsForCursorPosition(l->text.data(),m_selection2.x());
		} else {
			// selection ends below
			selectionEndX += getTextWidthWithTabsForCursorPosition(l->text.data(),l->length);
		}
		XSetFunction(dpy,gc,GXinvert);
		XFillRectangle(dpy,drw,gc,selectionBeginX,updateR.y(),selectionEndX - selectionBeginX,updateR.height());
		XSetFunction(dpy,gc,GXcopy);
	}
}

bool cmp(const char *token1,const char *token2,int len1,int len2)
{
	if(len1 != len2)return false;
	return (strncmp(token1,token2,len2) == 0);
}

bool cmpnocase(const char *token1,const char *token2,int len1,int len2)
{
	if(len1 != len2)return false;
	return (strncasecmp(token1,token2,len2) == 0);
}

bool cmpnolen(const char *token1,const char *token2,int len)
{
	return (strncmp(token1,token2,len) == 0);
}

void SSEXEditor::paintCellCpp(QPainter * p,SSEXEditorTextLine * l,int row)
{
	Display * dpy = x11Display();
	Drawable drw = m_pMemBuffer->handle();
	GC gc = XCreateGC(dpy,drw,0,0);
	QRect updateR = cellUpdateRect();

	XSetFillStyle(dpy,gc,FillSolid);

	XSetForeground(dpy,gc,m_pColors->cppBackground.pixel());
#ifdef COMPILE_USE_AA_FONTS
	if(qt_use_xft())
	{
		g_pXftFont = (XftFont *)qt_ft_font(&(font()));
		g_pXftDraw = qt_lookup_ft_draw (drw,false,0);
		if(!g_pXftDraw)
		{
			XSetFont(dpy,gc,font().handle());
			g_pXftFont = 0;
		}
		
	} else {
#endif
		XSetFont(dpy,gc,font().handle());
#ifdef COMPILE_USE_AA_FONTS
		g_pXftFont = 0;
		g_pXftDraw = 0;
	}
#endif

	XFillRectangle(dpy,drw,gc,updateR.x(),updateR.y(),updateR.width(),updateR.height());

	const char * c = l->text.data();
	int curXPos = SSEX_EDITOR_BORDER;
	int lastTabPos = SSEX_EDITOR_BORDER;

	const char * begin;
	QColor * pClr;
	int blockWidth;

	bool bInLineComment = false;
	bool bInString = false;
	bool bInChar = false;
	bool bInMultilineComment = l->flags & SSEX_EDITOR_BEGIN_IN_COMMENT;
	bool bPreprocessor = false;
	bool bGotInclude = false;

//	bool bLastTokenInScope;
//	bool bCurTokenInScope = false;

	while(*c)
	{
//		bLastObjectScope = false;
//		bLastTokenInScope = bCurTokenInScope;
//		bCurTokenInScope = false;
		switch(*c)
		{
			case '\t':
				// a tab character
				while(lastTabPos <= curXPos)lastTabPos += m_iTabsNumPixels;
				curXPos = lastTabPos;
				c++;
				break;
			case ' ':
				// a set of spaces
				begin = c;
				while(*c == ' ')c++;
				curXPos += (m_iCharWidth[(unsigned char)' '] * (c - begin));
//				debug("LIne %d, a set of %d spaces brings us to %d",row,c - begin,curXPos);
				break;
			default:
				begin      = c;
				blockWidth = 0;
				{
					//
					// EXTRACT TOKEN ####################################################################################
					//
					if(bInLineComment)
					{
						// We are in a line comment...up to the end of the line!
						while((*c) && (*c != ' ') && (*c != '\t'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(m_pColors->cppLineComment);

					} else if(bInMultilineComment)
					{
						// Multiline comment
						while((*c) && (*c != ' ') && (*c != '\t') && bInMultilineComment)
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							if(*c == '*')
							{
								c++;
								if(*c == '/'){
									// End of comment!!!
									blockWidth += m_iCharWidth[(unsigned char)'/'];
									c++;
									bInMultilineComment = false;
								}
							} else c++;
						}
						pClr = &(m_pColors->cppMultilineComment);

					} else if(bInString)
					{
						// Inside a string
						if(*c == '\\'){
							// escape sequence inside a string
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							if((*c) && (*c != ' ') && (*c != '\t'))
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
								pClr = &(m_pColors->cppEscape);
							} else pClr = &(m_pColors->cppString);
						} else {
							while((*c) && (*c != ' ') && (*c != '\t') && (*c != '\\') && bInString)
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								if(*c == '"')
								{
									c++;
									bInString = false;
								} else c++;
							}
							pClr = &(m_pColors->cppString);
						}
					} else if(bInChar)
					{
						// Inside a char constant
						if(*c == '\\'){
							// escape sequence inside a string
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							if((*c) && (*c != ' ') && (*c != '\t'))
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
								pClr = &(m_pColors->cppEscape);
							} else pClr = &(m_pColors->cppChar);
						} else {
							while((*c) && (*c != ' ') && (*c != '\t') && (*c != '\\') && bInChar)
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								if(*c == '\'')
								{
									c++;
									bInChar = false;
								} else c++;
							}
							pClr = &(m_pColors->cppChar);
						}
					} else if(bGotInclude){
						bGotInclude = false;
						while((*c) && (*c != ' ') && (*c != '\t'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(m_pColors->cppInclude);
					} else if(	((*c >= 'a') && (*c <= 'z')) ||
								((*c >= 'A') && (*c <= 'Z')) ||
								(*c == '_'))
					{
						// A text token
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						while(	((*c >= 'a') && (*c <= 'z')) ||
								((*c >= 'A') && (*c <= 'Z')) ||
								((*c >= '0') && (*c <= '9')) ||
								(*c == '_'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						if(bPreprocessor){
							pClr = &(m_pColors->cppPreprocessor);
							bPreprocessor = false;
							if(cmp(begin,"include",7,c - begin))bGotInclude = true;
						} else pClr = cppModeGetTokenColor(begin,c - begin);
					} else if(	(*c == '(') || (*c == ')') ||
								(*c == '[') || (*c == ']'))
					{
						// A parenthesis
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						while(	(*c == '(') || (*c == ')') ||
								(*c == '[') || (*c == ']'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(m_pColors->cppParenthesis);
					} else if(  (*c == ':'))
					{
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						if(*c == ':'){
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(m_pColors->cppPunctuation);
					} else if(	(*c == ';') || (*c == ',') || (*c == '.'))
					{
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						pClr = &(m_pColors->cppPunctuation);
					} else if(	(*c >= '0') && (*c <= '9'))
					{
						// A parenthesis
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						while(	((*c >= '0') && (*c <= '9')) ||
								((*c >= 'a') && (*c <= 'z')) ||
								((*c >= 'A') && (*c <= 'Z')))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(m_pColors->cppNumber);
					} else if(	*c == '#')
					{
						// A preprocessor code
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						bPreprocessor = true;
						pClr = &(m_pColors->cppPreprocessor);
					} else if(	(*c == '+') || (*c == '-') ||
								(*c == '*') || (*c == '?') ||
								(*c == '!') || (*c == '<') ||
								(*c == '>') || (*c == '^') ||
								(*c == '%') || (*c == '&') ||
								(*c == '|') || (*c == '=') ||
								(*c == '~') || (*c == '\\'))
					{
						// A parenthesis
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						while(	(*c == '+') || (*c == '-') ||
								(*c == '*') || (*c == '?') ||
								(*c == '!') || (*c == '<') ||
								(*c == '>') || (*c == '^') ||
								(*c == '%') || (*c == '&') ||
								(*c == '|') || (*c == '=') ||
								(*c == '~') || (*c == '\\'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(m_pColors->cppOperator);
					} else if( (*c == '{') || (*c == '}') )
					{
						// brace
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						pClr = &(m_pColors->cppBrace);
					} else if( (*c == '"'))
					{
						// string
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						bInString = true;
						pClr = &(m_pColors->cppString);
					} else if( (*c == '\''))
					{
						// string
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						bInChar = true;
						pClr = &(m_pColors->cppChar);
					} else if( (*c == '/'))
					{
						// Comment begin ?
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						if(*c == '/'){
							bInLineComment = true;
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							pClr = &(m_pColors->cppLineComment);
						} else if(*c == '*'){
							bInMultilineComment = true;
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							pClr = &(m_pColors->cppMultilineComment);
						} else pClr = &(m_pColors->cppOperator);	
					} else {
						while(	(*c) && (*c != ' ') && (*c != '\t') &&
								(*c != '{') && (*c != '}') && (*c != '/') &&
								(*c != '(') && (*c != ')') && (*c != '[') &&
								(*c != ']') && (*c != '"') && (*c != '\'') &&
								(*c != '+') && (*c != '-') && (*c != '*') &&
								(*c != '!') && (*c != '<') && (*c != '.') &&
								(*c != '>') && (*c != '^') && (*c != '%') &&
								(*c != '&') && (*c != '|') && (*c != '=') &&
								(*c != ',') && (*c != ';') && (*c != ':') &&
								(*c != '~') && (*c != '?') && (*c != '#') && (*c != '\\'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(m_pColors->cppNormalText);
					}
					//
					// EXTRACT TOKEN ####################################################################################
					//
				}
//				debug("LIne %d, block width: %d, starting at %d, of %d chars of (%s)",row,blockWidth,curXPos,c - begin,begin);
				XSetForeground(dpy,gc,pClr->pixel());
#ifdef COMPILE_USE_AA_FONTS
				if(g_pXftFont){
					XftColor color;
					QColor * clr = &(pClr);
					color.color.red = clr->red() | clr->red() << 8;
					color.color.green = clr->green() | clr->green() << 8;
					color.color.blue = clr->blue() | clr->blue() << 8;
					color.color.alpha = 0xffff;
					color.pixel = clr->pixel();
					XftDrawString8(g_pXftDraw,&color,g_pXftFont,curXPos,m_iFontAscent,(unsigned char *)begin,c - begin);
				} else
#endif
					XDrawString(dpy,drw,gc,curXPos,m_iFontAscent,begin,c - begin);
				curXPos += blockWidth;
				break;
		}
	}

	if(m_iCursorRow == row)paintCursor(dpy,drw,gc,&(m_pColors->cppCursor),l);
	if(m_bHasSelection)paintSelection(row,l,dpy,drw,gc,updateR);

	int col_xPos,row_yPos;

	p->worldMatrix().map(0,0,&col_xPos,&row_yPos);

	XCopyArea(dpy,drw,handle(),gc,updateR.x(),updateR.y(),updateR.width(),updateR.height(),col_xPos + updateR.x(),row_yPos + updateR.y());

	XFreeGC(dpy,gc);
}

QColor * SSEXEditor::cppModeGetTokenColor(const char * token,int len)
{
	if(*(token + len - 1) == 't')
	{
		if(len > 2)
		{
			if((*token != '_') && (*(token + len - 2) == '_'))return &(m_pColors->cppType);
		}
	}

	switch(*token)
	{
		case '_':
			return &(m_pColors->cppSystemIdentifier);
		break;
		case 'a':
			switch(len)
			{
				case 3:
					if(cmpnolen(token,"asm",3))return &(m_pColors->cppKeyword);
				break;
				case 4:
					if(cmpnolen(token,"auto",4))return &(m_pColors->cppKeyword);
				break;
			}
		break;
		case 'b':
			switch(len)
			{
				case 4:
					if(cmpnolen(token,"bool",4))return &(m_pColors->cppType);
				break;
				case 5:
					if(cmpnolen(token,"break",5))return &(m_pColors->cppKeyword);
				break;
			}
		break;
		case 'c':
			switch(len)
			{
				case 4:
					if(cmpnolen(token,"case",4))return &(m_pColors->cppKeyword);
					if(cmpnolen(token,"char",4))return &(m_pColors->cppType);
				break;
				case 5:
					if(cmpnolen(token,"const",5))return &(m_pColors->cppKeyword);
					if(cmpnolen(token,"class",5))return &(m_pColors->cppKeyword);
					if(cmpnolen(token,"catch",5))return &(m_pColors->cppKeyword);
					if(cmpnolen(token,"catch",5))return &(m_pColors->cppKeyword);
				case 8:
					if(cmpnolen(token,"continue",8))return &(m_pColors->cppKeyword);
				break;
				case 10:
					if(cmpnolen(token,"const_cast",10))return &(m_pColors->cppKeyword);
				break;

			}
		break;
		case 'd':
			if(	cmp(token,"default",7,len) ||
				cmp(token,"do",2,len) ||
				cmp(token,"delete",6,len) ||
				cmp(token,"dynamic_cast",11,len) ||
				cmp(token,"dec",3,len)
			) return &(m_pColors->cppKeyword);
			if(	cmp(token,"double",6,len)
			) return &(m_pColors->cppType);
			if( cmp(token,"defined",7,len)
			) return &(m_pColors->cppPreprocessor);
//			if(	cmp(token,"dup",3,len) ||
//				cmp(token,"dup2",4,len) ||
//				cmp(token,"domainname",10,len) ||
//				cmp(token,"daemon",6,len) ||
//				cmp(token,"dlopen",6,len) ||
//				cmp(token,"dlsym",5,len) ||
//				cmp(token,"dlclose",7,len) ||
//				cmp(token,"dlerror",7,len)
//			) return &(m_pColors->cppSysCall);
		break;
		case 'e':
			if(	cmp(token,"extern",6,len) ||
				cmp(token,"else",4,len) ||
				cmp(token,"enum",4,len)
			) return &(m_pColors->cppKeyword);
			if( cmp(token,"emit",4,len))return &(m_pColors->cppQSignals);
//			if(	cmp(token,"execl",5,len) ||
//				cmp(token,"execle",6,len) ||
//				cmp(token,"execlp",6,len) ||
//				cmp(token,"execv",5,len) ||
//				cmp(token,"execve",6,len) ||
//				cmp(token,"execvp",6,len) ||
//				cmp(token,"encrypt",7,len)
//			) return &(m_pColors->cppSysCall);
		break;
		case 'f':
			if(	cmp(token,"for",3,len) ||
				cmp(token,"fixed",5,len) ||
				cmp(token,"false",5,len) ||
				cmp(token,"friend",6,len)
			) return &(m_pColors->cppKeyword);
			if(	cmp(token,"float",5,len)
			) return &(m_pColors->cppType);
//			if(	cmp(token,"fork",4,len) ||
//				cmp(token,"fcntl",5,len) ||
//				cmp(token,"fclose",6,len) ||
//				cmp(token,"fexecve",7,len) ||
//				cmp(token,"fopen",5,len) ||
//				cmp(token,"freopen",7,len) ||
//				cmp(token,"fdopen",6,len) ||
//				cmp(token,"feof",4,len) ||
//				cmp(token,"ferror",6,len) ||
//				cmp(token,"fflush",6,len) ||
//				cmp(token,"free",4,len) ||
//				cmp(token,"fgetc",5,len) ||
//				cmp(token,"fputc",5,len) ||
//				cmp(token,"fgets",5,len) ||
//				cmp(token,"fgetc",5,len) ||
//				cmp(token,"fread",5,len) ||
//				cmp(token,"fprintf",7,len) ||
//				cmp(token,"fileno",6,len) ||
//				cmp(token,"fscanf",6,len) ||
//				cmp(token,"fseek",5,len) ||
//				cmp(token,"fwrite",6,len) ||
//				cmp(token,"ftime",5,len)
//			) return &(m_pColors->cppSysCall);
		break;
		case 'g':
			if(len > 1)
			{
				if(*(token + 1) == '_')return &(m_pColors->cppGlobalVariable);
				if(strncmp(token,"gtk_",4) == 0)return &(m_pColors->cppGtkCall);
				if(strncmp(token,"gdk_",4) == 0)return &(m_pColors->cppGdkCall);
				switch(len)
				{
					case 4:
						if(cmpnolen(token,"goto",4))return &(m_pColors->cppKeyword);
						if(cmpnolen(token,"gint",4))return &(m_pColors->cppGType);
					break;
					case 5:
						if(cmpnolen(token,"guint",5))return &(m_pColors->cppGType);
						if(cmpnolen(token,"glong",5))return &(m_pColors->cppGType);
						if(cmpnolen(token,"gchar",5))return &(m_pColors->cppGType);
						if(cmpnolen(token,"gsize",5))return &(m_pColors->cppGType);
						if(cmpnolen(token,"gint8",5))return &(m_pColors->cppGType);
					break;
					case 6:
						if(cmpnolen(token,"gulong",6))return &(m_pColors->cppGType);
						if(cmpnolen(token,"guchar",6))return &(m_pColors->cppGType);
						if(cmpnolen(token,"gshort",6))return &(m_pColors->cppGType);
						if(cmpnolen(token,"gfloat",6))return &(m_pColors->cppGType);
						if(cmpnolen(token,"gssize",6))return &(m_pColors->cppGType);
						if(cmpnolen(token,"guint8",6))return &(m_pColors->cppGType);
						if(cmpnolen(token,"gint16",6))return &(m_pColors->cppGType);
						if(cmpnolen(token,"gint32",6))return &(m_pColors->cppGType);
						if(cmpnolen(token,"gint64",6))return &(m_pColors->cppGType);
					break;
					case 7:
						if(cmpnolen(token,"gushort",7))return &(m_pColors->cppGType);
						if(cmpnolen(token,"gdouble",7))return &(m_pColors->cppGType);
						if(cmpnolen(token,"guint16",7))return &(m_pColors->cppGType);
						if(cmpnolen(token,"guint32",7))return &(m_pColors->cppGType);
						if(cmpnolen(token,"guint64",7))return &(m_pColors->cppGType);
					break;
					case 8:
						if(cmpnolen(token,"gboolean",8))return &(m_pColors->cppGType);
						if(cmpnolen(token,"gldouble",8))return &(m_pColors->cppGType);
						if(cmpnolen(token,"gpointer",8))return &(m_pColors->cppGType);
					break;
					case 13:
						if(cmpnolen(token,"gconstpointer",13))return &(m_pColors->cppGType);
					break;
				}
			}
		break;
		case 'i':
			if(	cmp(token,"if",2,len) ||
				cmp(token,"inline",6,len) ||
				cmp(token,"internal",8,len)
			) return &(m_pColors->cppKeyword);
			if(	cmp(token,"int",3,len)
			) return &(m_pColors->cppType);
		break;
		case 'k':
			if( strncmp(token,"kvi_",4) == 0)return &(m_pColors->cppSpecial);
		break;
		case 'l':
			if(	cmp(token,"long",4,len)
			) return &(m_pColors->cppType);
		break;
		case 'm':
			if( (strncmp(token,"m_",2) == 0)
			) return &(m_pColors->cppMemberVariable);
			if(	cmp(token,"mutable",7,len)
			) return &(m_pColors->cppKeyword);
		break;
		case 'n':
			if(	cmp(token,"namespace",9,len) ||
				cmp(token,"new",3,len)
			) return &(m_pColors->cppKeyword);
		break;
		case 'o':
			if(	cmp(token,"overload",8,len) ||
				cmp(token,"operator",8,len)
			) return &(m_pColors->cppKeyword);
		break;
		case 'p':
			if(	cmp(token,"public",6,len) ||
				cmp(token,"protected",9,len) ||
				cmp(token,"private",7,len)
			) return &(m_pColors->cppKeyword);
		break;
//		case 'q':
//			if(	cmp(token,"public",6,len) ||
//				cmp(token,"protected",9,len) ||
//				cmp(token,"private",7,len)
//			) return &(m_pColors->cppKeyword);
//		break;
		case 'r':
			if(	cmp(token,"return",6,len) ||
				cmp(token,"register",8,len) ||
				cmp(token,"reinterpret_cast",16,len)
			) return &(m_pColors->cppKeyword);
		break;
		case 's':
			if(	cmp(token,"switch",6,len) ||
				cmp(token,"struct",6,len) ||
				cmp(token,"static",6,len) ||
				cmp(token,"sizeof",6,len) ||
				cmp(token,"static_cast",11,len)
			) return &(m_pColors->cppKeyword);
			if(	cmp(token,"signed",6,len) ||
				cmp(token,"short",5,len)
			) return &(m_pColors->cppType);
			if(	cmp(token,"signals",7,len) ||
				cmp(token,"slots",5,len)
			) return &(m_pColors->cppQSignals);
		break;
		case 't':
			if(	cmp(token,"typedef",7,len) ||
				cmp(token,"template",8,len) ||
				cmp(token,"this",4,len) ||
				cmp(token,"throw",5,len) ||
				cmp(token,"try",3,len) ||
				cmp(token,"true",4,len) ||
				cmp(token,"typeid",6,len)
			) return &(m_pColors->cppKeyword);
			
		break;
		case 'u':
			if(	cmp(token,"union",5,len)
			) return &(m_pColors->cppKeyword);
			if(	cmp(token,"uint",4,len) ||
				cmp(token,"uchar",5,len) ||
				cmp(token,"ulong",5,len) ||
				cmp(token,"ushort",6,len) ||
				cmp(token,"unsigned",8,len) ||
				cmp(token,"u_int",5,len) ||
				cmp(token,"u_char",6,len) ||
				cmp(token,"u_long",6,len) ||
				cmp(token,"u_short",7,len)
			) return &(m_pColors->cppType);
		break;
		case 'v':
			if(	cmp(token,"volatile",8,len) ||
				cmp(token,"virtual",7,len)
			) return &(m_pColors->cppKeyword);
			if(	cmp(token,"void",4,len)
			) return &(m_pColors->cppType);
		break;
		case 'w':
			if(	cmp(token,"while",5,len)
			) return &(m_pColors->cppKeyword);
		break;
		case 'A':
			if(	cmp(token,"Atom",4,len)
			) return &(m_pColors->cppXStuff);
		break;
		case 'B':
			if(	cmp(token,"Bool",4,len)
			) return &(m_pColors->cppXStuff);
		break;
		case 'C':
			if(	cmp(token,"Colormap",8,len) ||
				cmp(token,"Cursor",6,len) ||
				cmp(token,"Cardinal",8,len)
			) return &(m_pColors->cppXStuff);
		break;
		case 'D':
			if(	cmp(token,"Depth",5,len) ||
				cmp(token,"Display",7,len) ||
				cmp(token,"Drawable",8,len)
			) return &(m_pColors->cppXStuff);
			if(	cmp(token,"DIR",3,len))return &(m_pColors->cppType);
		break;
		case 'F':
			if(	cmpnocase(token,"FALSE",5,len)
			) return &(m_pColors->cppKeyword);
			if(	cmp(token,"FILE",4,len)
			) return &(m_pColors->cppType);
			if(	cmp(token,"Font",4,len)
			) return &(m_pColors->cppXStuff);
		break;
		case 'G':
			if(len > 2)
			{
				if(*(token + 1)=='_')return &(m_pColors->cppGMacro);
				if(len > 4)
				{
					if(strncmp(token,"GTK_",4) == 0)return &(m_pColors->cppGtkMacro);
					if(strncmp(token,"GDK_",4) == 0)return &(m_pColors->cppGdkMacro);
				}
				if(strncmp(token,"Gtk",3) == 0)return &(m_pColors->cppGtkStruct);
				if(strncmp(token,"Gdk",3) == 0)return &(m_pColors->cppGdkStruct);
			} else if(cmp(token,"GC",2,len))return &(m_pColors->cppXStuff);
//			return &(m_pColors->cppGStruct);
		break;
		case 'K':
			return &(m_pColors->cppKClass);
		break;
		case 'M':
			if(	cmp(token,"Mask",4,len)
			) return &(m_pColors->cppXStuff);
		break;
		case 'N':
			if(	cmp(token,"NULL",4,len)
			) return &(m_pColors->cppKeyword);
		break;
		case 'P':
			if(	cmp(token,"Pixmap",6,len)
			) return &(m_pColors->cppXStuff);
		break;
		case 'Q':
			return &(m_pColors->cppQClass);
		break;
		case 'S':
			if(	cmp(token,"SLOT",4,len) ||
				cmp(token,"SIGNAL",6,len)
			) return &(m_pColors->cppQSignals);
			if(	cmp(token,"Status",6,len) ||
				cmp(token,"Screen",6,len) ||
				cmp(token,"ScreenFormat",12,len)) return &(m_pColors->cppXStuff);
		break;
		case 'T':
			if(	cmpnocase(token,"TRUE",4,len)
			) return &(m_pColors->cppKeyword);
			if(	cmp(token,"Time",4,len)
			) return &(m_pColors->cppXStuff);
		break;
		case 'V':
			if(	cmp(token,"Visual",6,len) ||
				cmp(token,"VisualID",8,len)
			) return &(m_pColors->cppXStuff);
		break;
		case 'W':
			if(	cmp(token,"Window",6,len) ||
				cmp(token,"Widget",6,len)
			) return &(m_pColors->cppXStuff);
		break;
		case 'X':
			return &(m_pColors->cppXStuff);
		break;
	}
	return &(m_pColors->cppNormalText);
}

void SSEXEditor::paintCellHtml(QPainter * p,SSEXEditorTextLine * l,int row)
{
	Display * dpy = x11Display();
	Drawable drw = m_pMemBuffer->handle();
	GC gc = XCreateGC(dpy,drw,0,0);
	QRect updateR = cellUpdateRect();

	XSetFillStyle(dpy,gc,FillSolid);

	XSetForeground(dpy,gc,m_pColors->htmlBackground.pixel());
#ifdef COMPILE_USE_AA_FONTS
	if(qt_use_xft())
	{
		g_pXftFont = (XftFont *)qt_ft_font(&(font()));
		g_pXftDraw = qt_lookup_ft_draw (drw,false,0);
		if(!g_pXftDraw)
		{
			XSetFont(dpy,gc,font().handle());
			g_pXftFont = 0;
		}
		
	} else {
#endif
		XSetFont(dpy,gc,font().handle());
#ifdef COMPILE_USE_AA_FONTS
		g_pXftFont = 0;
		g_pXftDraw = 0;
	}
#endif

	XFillRectangle(dpy,drw,gc,updateR.x(),updateR.y(),updateR.width(),updateR.height());

	const char * c = l->text.data();
	int curXPos = SSEX_EDITOR_BORDER;
	int lastTabPos = SSEX_EDITOR_BORDER;

	const char * begin;
	QColor * pClr = 0;
	int blockWidth;

	bool bInString = false;
	bool bInComment = l->flags & SSEX_EDITOR_BEGIN_IN_COMMENT;
	bool bInTag = l->flags & SSEX_EDITOR_BEGIN_IN_TAG;

	while(*c)
	{
		switch(*c)
		{
			case '\t':
				// a tab character
				while(lastTabPos <= curXPos)lastTabPos += m_iTabsNumPixels;
				curXPos = lastTabPos;
				c++;
				break;
			case ' ':
				// a set of spaces
				begin = c;
				while(*c == ' ')c++;
				curXPos += (m_iCharWidth[(unsigned char)' '] * (c - begin));
//				debug("LIne %d, a set of %d spaces brings us to %d",row,c - begin,curXPos);
				break;
			default:
				begin      = c;
				blockWidth = 0;
				{
					//
					// EXTRACT TOKEN ####################################################################################
					//
					if(bInComment)
					{
						if(*c == '-')
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							if(*c == '-')
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
								if(*c == '>')
								{
									blockWidth += m_iCharWidth[(unsigned char)(*c)];
									c++;
									bInComment = false;
									bInTag = false;
									bInString = false;
								} else pClr = &(m_pColors->htmlComment);
							} else pClr = &(m_pColors->htmlComment);
						} else {
							while(*c && (*c != '-') && (*c != ' ') && (*c != '\t'))
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
							}
							pClr = &(m_pColors->htmlComment);
						}
					} else if(bInTag)
					{
						if(*c == '"')
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							bInString = !bInString;
							pClr = &(m_pColors->htmlString);
						} else if(*c == '>')
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							bInString = false;
							bInTag = false;
							pClr = &(m_pColors->htmlTag);
						} else {
							while(*c && (*c != '>') && (*c != '"'))
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
							}
							pClr = bInString ? &(m_pColors->htmlString) : &(m_pColors->htmlTagInternal);
						}
					} else {
						// normal text
						if(*c == '<')
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							bInString = false;
							bInTag = true;
							if(*c == '!')
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
								if(*c == '-')
								{
									blockWidth += m_iCharWidth[(unsigned char)(*c)];
									c++;
									if(*c == '-')
									{
										blockWidth += m_iCharWidth[(unsigned char)(*c)];
										c++;
										bInString = false;
										bInTag =false;
										bInComment = true;
										pClr = &(m_pColors->htmlComment);
									} else pClr = &(m_pColors->htmlTag);
								} else pClr = &(m_pColors->htmlTag);
							} else pClr = &(m_pColors->htmlTag);
						} else {
							while(*c && (*c != '<'))
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
							}
							pClr = &(m_pColors->htmlNormalText);
						}
					}
					//
					// EXTRACT TOKEN ####################################################################################
					//
				}
//				debug("LIne %d, block width: %d, starting at %d, of %d chars of (%s)",row,blockWidth,curXPos,c - begin,begin);
				XSetForeground(dpy,gc,pClr->pixel());
#ifdef COMPILE_USE_AA_FONTS
				if(g_pXftFont){
					XftColor color;
					QColor * clr = &(pClr);
					color.color.red = clr->red() | clr->red() << 8;
					color.color.green = clr->green() | clr->green() << 8;
					color.color.blue = clr->blue() | clr->blue() << 8;
					color.color.alpha = 0xffff;
					color.pixel = clr->pixel();
					XftDrawString8(g_pXftDraw,&color,g_pXftFont,curXPos,m_iFontAscent,(unsigned char *)begin,c - begin);
				} else
#endif
					XDrawString(dpy,drw,gc,curXPos,m_iFontAscent,begin,c - begin);
				curXPos += blockWidth;
				break;
		}
	}

	if(m_iCursorRow == row)paintCursor(dpy,drw,gc,&(m_pColors->htmlCursor),l);
	if(m_bHasSelection)paintSelection(row,l,dpy,drw,gc,updateR);

	int col_xPos,row_yPos;

	p->worldMatrix().map(0,0,&col_xPos,&row_yPos);

	XCopyArea(dpy,drw,handle(),gc,updateR.x(),updateR.y(),updateR.width(),updateR.height(),col_xPos + updateR.x(),row_yPos + updateR.y());

	XFreeGC(dpy,gc);
}


inline void SSEXEditor::recalcCursorPosition(SSEXEditorTextLine *l)
{
	m_iCursorPositionInPixels = getTextWidthWithTabsForCursorPosition(l->text.data(),m_iCursorPosition);
}

int SSEXEditor::cursorCol() const
{
	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
	if(!l)return 0;
	return (m_iCursorPosition < l->length) ? m_iCursorPosition : l->length;
}

/////////////////////////////////////////////////////////////////////////////////////////////////
//
// SELECTION HANDLING HELPERS
//
/////////////////////////////////////////////////////////////////////////////////////////////////

void SSEXEditor::ensureSelectionCoherent()
{
	if((m_selection2.y() < m_selection1.y()) || ((m_selection1.y() == m_selection2.y()) && (m_selection2.x() < m_selection1.x())))
	{
		QPoint aux = m_selection1;
		m_selection1 = m_selection2;
		m_selection2 = aux;
	}
}

void SSEXEditor::setSelectionCoords(int x1,int row1,int x2,int row2)
{
	setHasSelection(true);
	m_selection1 = QPoint(x1,row1);
	m_selection2 = QPoint(x2,row2);
}

void SSEXEditor::clearSelection(bool bUpdate)
{
	setHasSelection(false);
	if(bUpdate)update();	
}

void SSEXEditor::selectionCursorMovement(const QPoint &oldCursorPos,const QPoint &newCursorPos)
{
	if(m_bHasSelection)
	{
		if(m_selection1 == oldCursorPos)m_selection1 = newCursorPos;
		else if(m_selection2 == oldCursorPos)m_selection2 = newCursorPos;
		else setSelectionCoords(oldCursorPos.x(),oldCursorPos.y(),newCursorPos.x(),newCursorPos.y());
	} else setSelectionCoords(oldCursorPos.x(),oldCursorPos.y(),newCursorPos.x(),newCursorPos.y());

	ensureSelectionCoherent();
}
/////////////////////////////////////////////////////////////////////////////////////////////////
//
// CURSOR MOVEMENT
//
/////////////////////////////////////////////////////////////////////////////////////////////////

void SSEXEditor::cursorUp(bool bSelect)
{

	if(m_iCursorRow > 0)
	{
		if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(m_iCursorPosition,m_iCursorRow - 1));
		else if(m_bHasSelection)clearSelection(true);

		m_iCursorRow--;
		recalcCursorPosition(m_pLines->at(m_iCursorRow));
		ensureCursorVisible();
		if(bSelect)update();
		else {
			updateCell(m_iCursorRow + 1,0,false);
			updateCell(m_iCursorRow,0,false);
		}
	}
}

void SSEXEditor::cursorDown(bool bSelect)
{
	if(((int)m_iCursorRow) < ((int)(m_pLines->count() - 1)))
	{
		if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(m_iCursorPosition,m_iCursorRow + 1));
		else if(m_bHasSelection)clearSelection(true);

		m_iCursorRow++;

		recalcCursorPosition(m_pLines->at(m_iCursorRow));
		ensureCursorVisible();
		if(bSelect)update();
		else {
			updateCell(m_iCursorRow - 1,0,false);
			updateCell(m_iCursorRow,0,false);
		}
	}
}

void SSEXEditor::cursorPageUp(bool bSelect)
{
	if(m_iCursorRow > 0)
	{
		int oldCursorRow = m_iCursorRow;
		m_iCursorRow -=(viewHeight() / cellHeight());
		if(m_iCursorRow < 0)m_iCursorRow = 0;

		if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,oldCursorRow),QPoint(m_iCursorPosition,m_iCursorRow));
		else if(m_bHasSelection)clearSelection(true);

		recalcCursorPosition(m_pLines->at(m_iCursorRow));
		ensureCursorVisible();
		update();
	}
}

void SSEXEditor::cursorPageDown(bool bSelect)
{
	if(((int)m_iCursorRow) < ((int)(m_pLines->count() - 1)))
	{
		int oldCursorRow = m_iCursorRow;
		m_iCursorRow += (viewHeight() / cellHeight());
		if(((int)m_iCursorRow) >= ((int)m_pLines->count()))m_iCursorRow = m_pLines->count() - 1;

		if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,oldCursorRow),QPoint(m_iCursorPosition,m_iCursorRow));
		else if(m_bHasSelection)clearSelection(true);


		recalcCursorPosition(m_pLines->at(m_iCursorRow));
		ensureCursorVisible();
		update();
	}
}

void SSEXEditor::cursorLeft(bool bSelect)
{	
	SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
	if(m_iCursorPosition > l->length)m_iCursorPosition = l->length;
	if(m_iCursorPosition > 0)
	{
		if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(m_iCursorPosition - 1,m_iCursorRow));
		else if(m_bHasSelection)clearSelection(true);

		m_iCursorPosition--;
		recalcCursorPosition(l);
		ensureCursorVisible();
		if(bSelect)update();
		else updateCell(m_iCursorRow,0,false);
	} else {
		if(m_iCursorRow > 0)
		{
			QPoint old(m_iCursorPosition,m_iCursorRow);

			m_iCursorRow--;
			l = m_pLines->at(m_iCursorRow);
			m_iCursorPosition = l->length;

			if(bSelect)selectionCursorMovement(old,QPoint(m_iCursorPosition,m_iCursorRow));
			else if(m_bHasSelection)clearSelection(true);

			recalcCursorPosition(l);
			ensureCursorVisible();
			if(bSelect)update();
			else {
				updateCell(m_iCursorRow + 1,0,false);
				updateCell(m_iCursorRow,0,false);
			}
		}
	}
}

void SSEXEditor::cursorRight(bool bSelect)
{
	SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
	if(l->length > m_iCursorPosition)
	{
		if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(m_iCursorPosition + 1,m_iCursorRow));
		else if(m_bHasSelection)clearSelection(true);

		m_iCursorPosition++;
		recalcCursorPosition(l);
		ensureCursorVisible();
		if(bSelect)update();
		else updateCell(m_iCursorRow,0,false);
	} else {
		if(((int)m_iCursorRow) < ((int)(m_pLines->count() - 1)))
		{
			if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(0,m_iCursorRow));
			else if(m_bHasSelection)clearSelection(true);

			m_iCursorRow++;
			m_iCursorPosition = 0;
			m_iCursorPositionInPixels = 0;
			ensureCursorVisible();
			if(bSelect)update();
			else {
				updateCell(m_iCursorRow - 1,0,false);
				updateCell(m_iCursorRow,0,false);
			}
		}
	}
}

void SSEXEditor::cursorHome(bool bSelect)
{
	if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(0,m_iCursorRow));
	else if(m_bHasSelection)clearSelection(true);

	SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
	if(m_iCursorPosition > l->length)m_iCursorPosition = l->length;
	m_iCursorPosition = 0;
	recalcCursorPosition(l);
	ensureCursorVisible();
	if(bSelect)update();
	else updateCell(m_iCursorRow,0,false);

}

void SSEXEditor::cursorEnd(bool bSelect)
{	
	SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
	if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(l->length,m_iCursorRow));
	else if(m_bHasSelection)clearSelection(true);
	if(m_iCursorPosition > l->length)m_iCursorPosition = l->length;
	m_iCursorPosition = l->length;
	recalcCursorPosition(l);
	ensureCursorVisible();
	if(bSelect)update();
	else updateCell(m_iCursorRow,0,false);
}



void SSEXEditor::ensureCursorVisible()
{
	if(topCell() > m_iCursorRow){
		setTopCell(m_iCursorRow);
	} else if(topCell() == m_iCursorRow){
		// We want it to be totally visible
//		debug("TOP CELL IS HERE");
		setYOffset(topCell() * cellHeight());
	} else {
		if(lastRowVisible() <= m_iCursorRow)setBottomCell(m_iCursorRow);
	}

	if((m_iCursorPositionInPixels) < xOffset())setXOffset(m_iCursorPositionInPixels);
	else  if((m_iCursorPositionInPixels + SSEX_EDITOR_BORDER + 20) > (viewWidth() + xOffset()))
			setXOffset((m_iCursorPositionInPixels + SSEX_EDITOR_BORDER + 20) - viewWidth());
}

void SSEXEditor::setBottomCell(int row)
{
	int row_yPos = row * cellHeight();
	setYOffset(row_yPos - (viewHeight() - cellHeight()));
}

void SSEXEditor::indent()
{
	if(m_bHasSelection)clearSelection(true);

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
	l->text.insert(0,'\t');
	l->length++;
	l->width  = getTextWidthWithTabs(l->text.data());
	if(m_iMaxTextWidthLine == m_iCursorRow){
		m_iMaxTextWidth = l->width;
		updateCellSize();
	} else {
		if(m_iMaxTextWidth < l->width)
		{
			m_iMaxTextWidth = l->width;
			m_iMaxTextWidthLine = m_iCursorRow;
			updateCellSize();
		}
	}

	if(((int)m_iCursorRow) < ((int)(m_pLines->count() - 1)))m_iCursorRow++;
	recalcCursorPosition(m_pLines->at(m_iCursorRow));

	ensureCursorVisible();
	updateCell(m_iCursorRow,0,false);
	updateCell(m_iCursorRow - 1,0,false);
	setModified(true);
}

void SSEXEditor::unindent()
{
	if(m_bHasSelection)clearSelection(true);

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	if(*(l->text.data()) == '\t')
	{
		l->text.remove(0,1);
		l->length--;
		l->width = getTextWidthWithTabs(l->text.data());
		if(m_iMaxTextWidthLine == m_iCursorRow){
			updateMaxTextWidth();
			updateCellSize();
		}
	} else l=0;
	if(((int)m_iCursorRow) < ((int)(m_pLines->count() - 1)))m_iCursorRow++;
	recalcCursorPosition(m_pLines->at(m_iCursorRow));
	ensureCursorVisible();
	updateCell(m_iCursorRow,false);
	updateCell(m_iCursorRow - 1,0,false);
	if(l)setModified(true);
}

void SSEXEditor::commentOut(bool bAlternative)
{
	if(m_bHasSelection)clearSelection(true);

	if(m_mode == Cpp){
		SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
		if(bAlternative){
			l->text.insert(0,"/*");
			l->text.append("*/");
			l->length += 4;
		} else {
			l->text.insert(0,"//");
			l->length += 2;
		}
		l->width  = getTextWidthWithTabs(l->text.data());
		if(m_iMaxTextWidthLine == m_iCursorRow){
			m_iMaxTextWidth = l->width;
			updateCellSize();
		} else {
			if(m_iMaxTextWidth < l->width)
			{
				m_iMaxTextWidth = l->width;
				m_iMaxTextWidthLine = m_iCursorRow;
				updateCellSize();
			}
		}

		if(((int)m_iCursorRow) < ((int)(m_pLines->count() - 1)))m_iCursorRow++;
		recalcCursorPosition(m_pLines->at(m_iCursorRow));

		ensureCursorVisible();
		updateCell(m_iCursorRow,0,false);
		updateCell(m_iCursorRow - 1,0,false);
		setModified(true);
	}
}

void SSEXEditor::removeComment()
{
	if(m_bHasSelection)clearSelection(true);

	if(m_mode == Cpp)
	{

		SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

		if(strncmp(l->text.data(),"//",2) == 0)
		{
			l->text.remove(0,2);
			l->length -= 2;
			l->width = getTextWidthWithTabs(l->text.data());
			if(m_iMaxTextWidthLine == m_iCursorRow){
				updateMaxTextWidth();
				updateCellSize();
			}
		} else if(strncmp(l->text.data(),"/*",2) == 0)
		{
			if((strncmp(l->text.right(2).data(),"*/",2) == 0) && (l->length > 3))
			{
				l->text.remove(0,2);
				l->text.remove(l->text.length() - 2,2);
				l->length -= 4;
				l->width = getTextWidthWithTabs(l->text.data());

				if(m_iMaxTextWidthLine == m_iCursorRow){
					updateMaxTextWidth();
					updateCellSize();
				}
			} else l=0;
		} else l=0;
		if(((int)m_iCursorRow) < ((int)(m_pLines->count() - 1)))m_iCursorRow++;
		recalcCursorPosition(m_pLines->at(m_iCursorRow));
		ensureCursorVisible();
		updateCell(m_iCursorRow,false);
		updateCell(m_iCursorRow - 1,0,false);
		if(l)setModified(true);
	}
}

void SSEXEditor::insertChar(char c)
{
#ifdef __DEBUG
	if((((unsigned char)c) < 32) && (((unsigned char)c) != ((unsigned char)'\t')))debug("INSERTING A STRANGE CHAR! (%c:%d)",c,c);
#endif

	if(m_bHasSelection)killSelection(true); // trigger the total update too

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
	char deletedChar = 0;
	if(m_iCursorPosition <= l->length){
		if(m_bOverwrite && (m_iCursorPosition < l->length)){
			deletedChar = *(l->text.data() + m_iCursorPosition);
			l->text.remove(m_iCursorPosition,1);
		} else l->length++;
		l->text.insert(m_iCursorPosition,c);
		m_iCursorPosition++;
	} else {
		l->text.insert(l->length,c);
		l->length++;
		m_iCursorPosition = l->length;
	}
	l->width  = getTextWidthWithTabs(l->text.data());
	recalcCursorPosition(l);
	if(m_iMaxTextWidthLine == m_iCursorRow){
		if(m_bOverwrite)updateMaxTextWidth();
		else m_iMaxTextWidth = l->width;
		updateCellSize();
	} else {
		if(m_iMaxTextWidth < l->width)
		{
			m_iMaxTextWidth = l->width;
			m_iMaxTextWidthLine = m_iCursorRow;
			updateCellSize();
		}
	}

//	__assert(l->text.length() == l->length);


	if((!m_bOverwrite) || (c != deletedChar))
	{
		if(m_mode == Cpp)
		{
			// In cpp mode update the comments state if needed
			if((c == '/')||(deletedChar == '/')){
				if((*(l->text.data() + m_iCursorPosition) == '*') || (*(l->text.data() + m_iCursorPosition) == '/') || (l->flags & SSEX_EDITOR_END_IN_COMMENT)){
					cppModeComputeCommentState(l);
					update();
				} else if(m_iCursorPosition > 1){
					if(*(l->text.data() + m_iCursorPosition - 2) == '*'){
						cppModeComputeCommentState(l);
						update();
					}
				}
			} else {
				if(*(l->text.data() + m_iCursorPosition) == '/'){
					cppModeComputeCommentState(l);
					update();
				} else if(m_iCursorPosition > 1){
					if(*(l->text.data() + m_iCursorPosition - 2) == '/'){
						cppModeComputeCommentState(l);
						update();
					}
				}
			}
		} else if(m_mode == Html)
		{
			if((c == '<') ||
			(deletedChar == '<') ||
			(c == '>') ||
			(deletedChar == '>') ||
			(c == '!') ||
			(deletedChar == '!') ||
			(c == '-') ||
			(deletedChar == '-')
			){
				htmlModeComputeTagState(l);
				update();
			}
		}
	}
	ensureCursorVisible();
	updateCell(m_iCursorRow,0,false);
	setModified(true);
}

void SSEXEditor::newLine()
{
	if(m_bHasSelection)clearSelection(false);

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
	int pos = ((m_iCursorPosition <= l->length) ? m_iCursorPosition : l->length);
	SSEXEditorTextLine *n = new SSEXEditorTextLine;
	n->text = (l->text.data() + pos);
	n->length = n->text.length();
	l->length -= n->length;
	l->text.truncate(l->length);
	l->width  = getTextWidthWithTabs(l->text.data());
	n->width  = getTextWidthWithTabs(n->text.data());
	m_pLines->insert(m_iCursorRow + 1,n);
	setNumRows(m_pLines->count());
	if(m_iMaxTextWidthLine == m_iCursorRow){
		updateMaxTextWidth();
		updateCellSize();
	} else {
		// MaxTextWidthLine is scrolled down!
		if(m_iMaxTextWidthLine > m_iCursorRow)m_iMaxTextWidthLine++;
	}
	m_iCursorRow++;
	m_iCursorPosition = 0;
	recalcCursorPosition(l);

	if(m_mode == Cpp)
	{
		// In cpp mode comments state if needed
		if((l->flags & SSEX_EDITOR_END_IN_COMMENT) && (*(n->text.data()) != '*'))
		{
			n->flags = SSEX_EDITOR_BEGIN_IN_COMMENT | SSEX_EDITOR_END_IN_COMMENT;
		} else cppModeComputeCommentState(l);
	} else if(m_mode == Html)
	{
		if((l->flags & SSEX_EDITOR_END_IN_COMMENT) && (*(n->text.data()) != '-') &&(*(n->text.data()) != '>'))
		{
			n->flags = SSEX_EDITOR_BEGIN_IN_COMMENT | SSEX_EDITOR_END_IN_COMMENT;
		} else	if(l->flags & SSEX_EDITOR_END_IN_TAG)
		{
			n->flags = SSEX_EDITOR_BEGIN_IN_TAG | SSEX_EDITOR_END_IN_TAG;
		} else htmlModeComputeTagState(l);
	}
	ensureCursorVisible();
	update();
	setModified(true);
}

void SSEXEditor::backspace()
{
	if(m_bHasSelection){
		killSelection(true);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
	int pos = (m_iCursorPosition <= l->length) ? m_iCursorPosition : l->length;

	if(pos == 0){
		// join lines
		if(m_iCursorRow > 0){
			m_iCursorRow--;
			SSEXEditorTextLine *prev = m_pLines->at(m_iCursorRow);
			prev->text.append(l->text);
			m_iCursorPosition = prev->length;
			recalcCursorPosition(prev);
			prev->length += l->length;
			prev->width = getTextWidthWithTabs(prev->text.data());
			if(m_mode == Cpp){
				prev->flags = ((prev->flags & SSEX_EDITOR_BEGIN_IN_COMMENT) | (l->flags & SSEX_EDITOR_END_IN_COMMENT));
				m_pLines->setAutoDelete(false);
				m_pLines->removeRef(l);
				m_pLines->setAutoDelete(true);
				if((*(l->text.data()) == '*') || (*(l->text.data()) == '/'))cppModeComputeCommentState(prev);
				delete l;
			} else if(m_mode == Html)
			{
				prev->flags = ((prev->flags & SSEX_EDITOR_BEGIN_IN_COMMENT) | (l->flags & SSEX_EDITOR_END_IN_COMMENT)) | ((prev->flags & SSEX_EDITOR_BEGIN_IN_TAG) | (l->flags & SSEX_EDITOR_END_IN_TAG));
				m_pLines->setAutoDelete(false);
				m_pLines->removeRef(l);
				m_pLines->setAutoDelete(true);
				if((*(l->text.data()) == '>') || (*(l->text.data()) == '<') || (*(l->text.data()) == '-') || (*(l->text.data()) == '!'))htmlModeComputeTagState(l);
				delete l;
			} else m_pLines->removeRef(l);
			setNumRows(m_pLines->count());

			if((m_iMaxTextWidthLine == m_iCursorRow)||(m_iMaxTextWidthLine == (m_iCursorRow + 1))){
				m_iMaxTextWidth = prev->width;
				m_iMaxTextWidthLine = m_iCursorRow;
				updateCellSize();
			} else {
				if(m_iMaxTextWidth < prev->width)
				{
					m_iMaxTextWidth = prev->width;
					m_iMaxTextWidthLine = m_iCursorRow;
					updateCellSize();
				} else {
					// The maxtextwidthline is scrolled up!
					if(m_iMaxTextWidthLine > (m_iCursorRow + 1))m_iMaxTextWidthLine--;
				}
			}
			update(); // we could calc the right rect here
		}
	} else {
		char c = *(l->text.data() + pos - 1);
		l->text.remove(pos - 1,1);
		l->length--;
		l->width = getTextWidthWithTabs(l->text.data());
		m_iCursorPosition = pos - 1;
		recalcCursorPosition(l);
		if(m_iMaxTextWidthLine == m_iCursorRow){
			updateMaxTextWidth();
			updateCellSize();
		}
		pos--;
		if(m_mode == Cpp)
		{
			// In cpp mode update the comments state if needed
			if(c == '/'){
				if((*(l->text.data() + pos) == '*') || (*(l->text.data() + pos) == '/') || (l->flags & SSEX_EDITOR_END_IN_COMMENT)){
					cppModeComputeCommentState(l);
					update();
				} else if(pos > 0){
					if(*(l->text.data() + pos - 1) == '*'){
						cppModeComputeCommentState(l);
						update();
					}
				}
			} else {
				if(*(l->text.data() + pos) == '/'){
					cppModeComputeCommentState(l);
					update();
				} else if(pos > 0){
					if(*(l->text.data() + pos - 1) == '/'){
						cppModeComputeCommentState(l);
						update();
					}
				}
			}
		} else if(m_mode == Html)
		{
			// In html mode update the tag state if needed
			if(	(c == '<') ||
				(c == '>') ||
				(c == '!')||
				(c == '-')
			){
				htmlModeComputeTagState(l);
				update();
			}
		}
		ensureCursorVisible();
		updateCell(m_iCursorRow,false);
	}
	setModified(true);
}

void SSEXEditor::del()
{
	if(m_bHasSelection){
		killSelection(true);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
	if(m_iCursorPosition > l->length)m_iCursorPosition = l->length;

	if(m_iCursorPosition == l->length){
		// join lines
		if(((int)m_iCursorRow) < ((int)(m_pLines->count() - 1))){
			SSEXEditorTextLine *next = m_pLines->at(m_iCursorRow + 1);
			l->text.append(next->text);
			l->length += next->length;
			l->width = getTextWidthWithTabs(l->text.data());
			if(m_mode == Cpp){
				l->flags = ((l->flags & SSEX_EDITOR_BEGIN_IN_COMMENT) | (next->flags & SSEX_EDITOR_END_IN_COMMENT));
				m_pLines->setAutoDelete(false);
				m_pLines->removeRef(next);
				m_pLines->setAutoDelete(true);
				if((*(next->text.data()) == '*') || (*(next->text.data()) == '/'))cppModeComputeCommentState(l);
				delete next;
			} else if(m_mode == Html)
			{
				l->flags = ((l->flags & SSEX_EDITOR_BEGIN_IN_COMMENT) | (next->flags & SSEX_EDITOR_END_IN_COMMENT)) | ((l->flags & SSEX_EDITOR_BEGIN_IN_TAG) | (next->flags & SSEX_EDITOR_END_IN_TAG));
				m_pLines->setAutoDelete(false);
				m_pLines->removeRef(next);
				m_pLines->setAutoDelete(true);
				if((*(next->text.data()) == '>') || (*(next->text.data()) == '<')|| (*(next->text.data()) == '-') || (*(next->text.data()) == '!'))htmlModeComputeTagState(l);
				delete next;
			} else m_pLines->removeRef(next);
			setNumRows(m_pLines->count());
			if((m_iMaxTextWidthLine == m_iCursorRow)||(m_iMaxTextWidthLine == (m_iCursorRow + 1))){
				m_iMaxTextWidth = l->width;
				m_iMaxTextWidthLine = m_iCursorRow;
				updateCellSize();
			} else {
				if(m_iMaxTextWidth < l->width)
				{
					m_iMaxTextWidth = l->width;
					m_iMaxTextWidthLine = m_iCursorRow;
					updateCellSize();
				}
			}
			update(); // we could calc the right rect here
		}
	} else {
		char c = *(l->text.data() + m_iCursorPosition);
		l->text.remove(m_iCursorPosition,1);
		l->length--;
		l->width = getTextWidthWithTabs(l->text.data());
		if(m_iMaxTextWidthLine == m_iCursorRow){
			updateMaxTextWidth();
			updateCellSize();
		}
		if(m_mode == Cpp)
		{
			// In cpp mode update the comments state if needed
			if(c == '/'){
				if((*(l->text.data() + m_iCursorPosition) == '*') || (*(l->text.data() + m_iCursorPosition) == '/') || (l->flags & SSEX_EDITOR_END_IN_COMMENT)){
					cppModeComputeCommentState(l);
					update();
				} else if(m_iCursorPosition > 0){
					if(*(l->text.data() + m_iCursorPosition - 1) == '*'){
						cppModeComputeCommentState(l);
						update();
					}
				}
			} else {
				if(*(l->text.data() + m_iCursorPosition) == '/'){
					cppModeComputeCommentState(l);
					update();
				} else if(m_iCursorPosition > 0){
					if(*(l->text.data() + m_iCursorPosition - 1) == '/'){
						cppModeComputeCommentState(l);
						update();
					}
				}
			}
		} else if(m_mode == Html)
		{
			// In html mode update the tag state if needed
			if(	(c == '<') ||
				(c == '>') ||
				(c == '!')||
				(c == '-')
			){
				htmlModeComputeTagState(l);
				update();
			}
		}
		ensureCursorVisible();
		updateCell(m_iCursorRow,false);
	}
	setModified(true);
}

void SSEXEditor::copy()
{
	if(m_bHasSelection)QApplication::clipboard()->setText(selectedText());
}

void SSEXEditor::paste()
{
	QCString cText = QApplication::clipboard()->text().ascii();
	if(!(cText.isNull() || cText.isEmpty()))insertText(cText);
}

void SSEXEditor::cut()
{
	if(!m_bHasSelection)return;
	copy();
	killSelection(true);
}

void SSEXEditor::killSelection(bool bUpdate,bool bRecalcWidths)
{
	if(!m_bHasSelection)return;
	SSEXEditorTextLine * first = m_pLines->at(m_selection1.y());
	if(!first)return;
	// Single line selection
	if(m_selection1.y() == m_selection2.y())
	{
		first->text.remove(m_selection1.x(),m_selection2.x() - m_selection1.x());
	} else {
		// Multiple lines selection
		first->text.remove(m_selection1.x(),first->length - m_selection1.x());
		int idx = m_selection1.y() + 1;
		SSEXEditorTextLine * l = m_pLines->next();
		QList<SSEXEditorTextLine> list;
		list.setAutoDelete(false);
		while(l && (idx <= m_selection2.y()))
		{
			if(idx == m_selection2.y()){
				l->text.remove(0,m_selection2.x());
				first->text.append(l->text);
				list.append(l);
			} else list.append(l);
			l = m_pLines->next();
			idx++;
		}
		// Kill the middle lines
		for(SSEXEditorTextLine *line=list.first();line;line=list.next())m_pLines->removeRef(line);
		setNumRows(m_pLines->count());
	}

	first->length = first->text.length();
	first->width = getTextWidthWithTabs(first->text.data());

	setHasSelection(false);
	m_iCursorRow = m_selection1.y();
	m_iCursorPosition = m_selection1.x() < first->length ? m_selection1.x() : first->length;


	recalcCursorPosition(first);
	if(bRecalcWidths)
	{
		updateMaxTextWidth();
		updateCellSize();
		if(m_mode == Cpp)cppModeComputeCommentState(first);
		else if(m_mode == Html)htmlModeComputeTagState(first);
	}
	if(bUpdate){
		ensureCursorVisible();
		update();
	}
	setModified(true);
}

void SSEXEditor::insertText(QCString &text,bool bUpdate,bool bRecalcWidths)
{
	if(m_bHasSelection)killSelection(false,false);
	SSEXEditorTextLine * first = m_pLines->at(m_iCursorRow);
	if(!first)return;
	QCString rightPart = (m_iCursorPosition < first->length) ? (first->text.data() + m_iCursorPosition) : "";
//	__assert(first->text.length() == first->length);
	first->text.remove(m_iCursorPosition,first->length - m_iCursorPosition);

	SSEXEditorTextLine * l = first;
	int row = m_iCursorRow;

	QCString theText = text;
	int idx = theText.find('\n');
	while(idx != -1)
	{
		l->text.append(theText.left(idx));
		theText.remove(0,idx + 1);
		l->length = l->text.length();
		l->width = getTextWidthWithTabs(l->text.data());
		row++;
		l = new SSEXEditorTextLine;
		l->text = "";
		l->length = 0;
		l->width = 0;
		m_pLines->insert(row,l);
		idx = theText.find('\n');
	}
	setNumRows(m_pLines->count());

	m_iCursorRow = row;
	m_iCursorPosition = l->text.length() + theText.length(); 

	l->text.append(theText);
	l->text.append(rightPart);
	l->length = l->text.length();
	l->width = getTextWidthWithTabs(l->text.data());


	recalcCursorPosition(l);
	if(bRecalcWidths)
	{
		updateMaxTextWidth();
		updateCellSize();
		if(m_mode == Cpp)cppModeComputeCommentState(first);
		else if(m_mode == Html)htmlModeComputeTagState(first);
	}
	if(bUpdate)
	{
		ensureCursorVisible();
		update();
	}
	setModified(true);
}

QCString SSEXEditor::selectedText()
{
	QCString ret;
	if(!m_bHasSelection)return ret;
	SSEXEditorTextLine * l = m_pLines->at(m_selection1.y());
	if(!l)return ret;
	if(m_selection1.y() == m_selection2.y()){
		return l->text.mid(m_selection1.x(),m_selection2.x() - m_selection1.x());
	} else {
		if(m_selection1.x() < l->length)ret = (l->text.data() + m_selection1.x());
		int idx = m_selection1.y();
		do{
			l = m_pLines->next();
			idx++;
			ret.append("\n");
			if(m_selection2.y() == idx)
			{
				ret.append(l->text.left(m_selection2.x()));
			} else ret.append(l->text);
		} while(l && (idx < m_selection2.y()));
	}
	return ret;	
}

void SSEXEditor::findNext()
{
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Find Next","No text to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	int row = m_iCursorRow;
	int col = m_iCursorPosition;

	while(l){

		if(col < l->length){
			int idx = l->text.find(toFind.data(),col,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
			if(idx != -1)
			{
				// Found an occurrence
				m_iCursorRow = row;
				m_iCursorPosition = idx + toFind.length();
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,m_iCursorPosition,row);
				ensureCursorVisible();
				m_bCursorOn = true;
				update();
				setFocus();
				return;
			}
		}
		if(((int)row) >= ((int)(m_pLines->count() - 1)))
		{
			// End reached
			if(QMessageBox::information(this,"Find Next","No matches found\nContinue from the beginning?",
				QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) != QMessageBox::Yes)return;
			row = 0;

		} else row++;

		col = 0;
		l = m_pLines->at(row);
	}
	// newer here
}

void SSEXEditor::findPrev()
{
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Find Previous","No text to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	int row = m_iCursorRow;
	int col = m_iCursorPosition - 1;

	while(l){

		if(((int)col) >= ((int)(toFind.length() - 1))){
			int idx = l->text.findRev(toFind.data(),col,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
			if(idx != -1)
			{
				// Found an occurrence
				m_iCursorRow = row;
				m_iCursorPosition = idx;
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,idx + toFind.length(),row);
				ensureCursorVisible();
				m_bCursorOn = true;
				update();
				setFocus();
				return;
			}
		}
		if(row <= 0)
		{
			// End reached
			if(QMessageBox::information(this,"Find Previous","No matches found\nContinue from the end?",
				QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) != QMessageBox::Yes)return;
			row = m_pLines->count() - 1;

		} else row--;

		l = m_pLines->at(row);
		col = l->length;
	}
	// newer here
}

void SSEXEditor::findNextRegExp()
{
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Find Next RegExp","No regular expression to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	int row = m_iCursorRow;
	int col = m_iCursorPosition;

	QRegExp rx(toFind,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());

	int len = 1;

	while(l){

		if(col < l->length){
			int idx = rx.match(l->text,col,&len);
			if(idx != -1)
			{
				// Found an occurrence
				m_iCursorRow = row;
				m_iCursorPosition = idx + len;
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,m_iCursorPosition,row);
				ensureCursorVisible();
				m_bCursorOn = true;
				update();
				setFocus();
				return;
			}
		}
		if(((int)row) >= ((int)(m_pLines->count() - 1)))
		{
			// End reached
			if(QMessageBox::information(this,"Find Next RegExp","No matches found\nContinue from the beginning?",
				QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) != QMessageBox::Yes)return;
			row = 0;

		} else row++;

		col = 0;
		l = m_pLines->at(row);
	}
	// newer here
}

void SSEXEditor::findPrevRegExp()
{
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Find Previous RegExp","No regular expression to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	int row = m_iCursorRow;
	int col = m_iCursorPosition - 1;

	QRegExp rx(toFind,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
	int len = 1;

	while(l){

		if(((int)col) >= ((int)(toFind.length() - 1))){
			int idx = col;
			bool bFound = false;

			while((idx >= 0) && (!bFound)){
				if(rx.match(l->text,idx,&len) != idx)idx--;
				else bFound = true;
			}

			if(idx != -1)
			{
				// Found an occurrence
				m_iCursorRow = row;
				m_iCursorPosition = idx;
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,idx + len,row);
				ensureCursorVisible();
				m_bCursorOn = true;
				update();
				setFocus();
				return;
			}
		}
		if(row <= 0)
		{
			// End reached
			if(QMessageBox::information(this,"Find Previous RegExp","No matches found\nContinue from the end?",
				QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) != QMessageBox::Yes)return;
			row = m_pLines->count() - 1;

		} else row--;

		l = m_pLines->at(row);
		col = l->length;
	}
	// newer here
}

void SSEXEditor::replace()
{
	if(!m_bHasSelection)
	{
		QMessageBox::warning(this,"Replace","No text selected",QMessageBox::Ok,0);
		return;
	}
	QCString toReplace = m_pFindWidget->m_pReplaceStringEdit->text().ascii();
	if(toReplace.isNull())toReplace = "";
	insertText(toReplace);
	setFocus();
}

void SSEXEditor::replaceAndFindNext()
{
	replace();
	findNext();
}

void SSEXEditor::replaceAndFindNextRegExp()
{
	replace();
	findNextRegExp();
}

void SSEXEditor::replaceAllInSelection()
{
	if(!m_bHasSelection)
	{
		QMessageBox::warning(this,"Replace in Selection","No selection to search in",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	QPoint topLeft = m_selection1;
	QPoint bottomRight = m_selection2;

	clearSelection(false);

	QCString toReplace = m_pFindWidget->m_pReplaceStringEdit->text().ascii();
	if(toReplace.isNull())toReplace = "";
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Replace in Selection","No text to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(topLeft.y());

	int row = topLeft.y();
	int col = topLeft.x();

	bool bFound;
	int count = 0;

	while(l && (row <= bottomRight.y())){

		bFound = false;

		if(col < l->length){
			int idx = l->text.find(toFind.data(),col,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
			if(idx != -1)
			{
				if(row == bottomRight.y())
				{
					if(((int)(idx + toFind.length())) > ((int)bottomRight.x()))
					{
						// finished
						break;
					}
				}
				// Found an occurrence
				m_iCursorRow = row;
				m_iCursorPosition = idx + toFind.length();
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,m_iCursorPosition,row);
				insertText(toReplace,false,false);
				col = m_iCursorPosition;
				bFound = true;
				count ++;
			} else bFound = false;
		}


		if(!bFound)
		{
			row++;
			col = 0;
			l = m_pLines->at(row);
		}
	}

	updateMaxTextWidth();
	updateCellSize();
	if(m_mode == Cpp)cppModeComputeCommentState(m_pLines->first());
	else if(m_mode == Html)htmlModeComputeTagState(m_pLines->first());
	m_bCursorOn = true;
	ensureCursorVisible();
	update();

	QCString msg;
	msg.sprintf("Replaced %d occurrences",count);
	emit textMessage(this,msg);
	setFocus();
}

void SSEXEditor::replaceAllInSelectionRegExp()
{
	if(!m_bHasSelection)
	{
		QMessageBox::warning(this,"Replace in Selection (RegExp)","No selection to search in",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	QPoint topLeft = m_selection1;
	QPoint bottomRight = m_selection2;

	clearSelection(false);

	QCString toReplace = m_pFindWidget->m_pReplaceStringEdit->text().ascii();
	if(toReplace.isNull())toReplace = "";
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Replace in Selection (RegExp)","No regular expression to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	QRegExp rx(toFind,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());

	SSEXEditorTextLine *l = m_pLines->at(topLeft.y());

	int row = topLeft.y();
	int col = topLeft.x();

	bool bFound;
	int count = 0;
	int len = 1;

	while(l && (row <= bottomRight.y())){

		bFound = false;

		if(col < l->length){
			int idx = rx.match(l->text,col,&len);
			if(idx != -1)
			{
				if(row == bottomRight.y())
				{
					if((idx + len) > bottomRight.x())
					{
						// finished
						break;
					}
				}
				// Found an occurrence
				m_iCursorRow = row;
				m_iCursorPosition = idx + len;
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,m_iCursorPosition,row);
				insertText(toReplace,false,false);
				col = m_iCursorPosition;
				bFound = true;
				count ++;
			} else bFound = false;
		}


		if(!bFound)
		{
			row++;
			col = 0;
			l = m_pLines->at(row);
		}
	}

	updateMaxTextWidth();
	updateCellSize();
	if(m_mode == Cpp)cppModeComputeCommentState(m_pLines->first());
	else if(m_mode == Html)htmlModeComputeTagState(m_pLines->first());
	m_bCursorOn = true;
	ensureCursorVisible();
	update();

	QCString msg;
	msg.sprintf("Replaced %d occurrences",count);
	emit textMessage(this,msg);
	setFocus();
}



void SSEXEditor::replaceAll()
{
	if(QMessageBox::warning(this,"Replace All",
		"This may be a destructive action\n" \
		"Replace all matches from the cursor\n" \
		"to end of the file?",
		QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) == QMessageBox::No)return;

	QCString toReplace = m_pFindWidget->m_pReplaceStringEdit->text().ascii();
	if(toReplace.isNull())toReplace = "";
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Replace All","No text to find",QMessageBox::Ok,0);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	int row = m_iCursorRow;
	int col = m_iCursorPosition;

	bool bFound;
	int count = 0;

	while(l){

		bFound = false;

		if(col < l->length){
			int idx = l->text.find(toFind.data(),col,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
			if(idx != -1)
			{
				// Found an occurrence
				m_iCursorRow = row;
				m_iCursorPosition = idx + toFind.length();
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,m_iCursorPosition,row);
				insertText(toReplace,false,false);
				col = m_iCursorPosition;
				bFound = true;
				count ++;
			} else bFound = false;
		}

		if(!bFound)
		{
			if(((int)row) >= ((int)(m_pLines->count() - 1)))
			{
				// End reached
				updateMaxTextWidth();
				updateCellSize();
				if(m_mode == Cpp)cppModeComputeCommentState(m_pLines->first());
				else if(m_mode == Html)htmlModeComputeTagState(m_pLines->first());
				m_bCursorOn = true;
				ensureCursorVisible();
				update();
				QCString msg;
				msg.sprintf("Replaced %d occurrences",count);
				emit textMessage(this,msg);
				if(QMessageBox::information(this,"Replace All","No matches found\nContinue from the beginning?",
					QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) != QMessageBox::Yes)
				{
					setFocus();
					return;
				}
				row = 0;
				count = 0;
			} else row++;

			col = 0;
			l = m_pLines->at(row);
		}
	}
}

void SSEXEditor::replaceAllRegExp()
{
	if(QMessageBox::warning(this,"Replace All (RegExp)",
		"This may be a destructive action\n" \
		"Replace all matches from the cursor\n" \
		"position to end of the file?",
		QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) == QMessageBox::No)return;


	QCString toReplace = m_pFindWidget->m_pReplaceStringEdit->text().ascii();
	if(toReplace.isNull())toReplace = "";
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Replace All (RegExp)","No regular expression to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	QRegExp rx(toFind,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	int row = m_iCursorRow;
	int col = m_iCursorPosition;

	bool bFound;
	int len = 1;

	while(l){

		bFound = false;

		if(col < l->length){
			int idx = rx.match(l->text,col,&len);
			if(idx != -1)
			{
				// Found an occurrence
				m_iCursorRow = row;
				m_iCursorPosition = idx + len;
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,m_iCursorPosition,row);
				insertText(toReplace,false,false);
				col = m_iCursorPosition;
				bFound = true;
			} else bFound = false;
		}

		if(!bFound)
		{
			if(((int)row) >= ((int)(m_pLines->count() - 1)))
			{
				// End reached
				updateMaxTextWidth();
				updateCellSize();
				if(m_mode == Cpp)cppModeComputeCommentState(m_pLines->first());
				else if(m_mode == Html)htmlModeComputeTagState(m_pLines->first());
				m_bCursorOn = true;
				ensureCursorVisible();
				update();
				if(QMessageBox::information(this,"Replace All (RegExp)","No matches found\nContinue from the beginning?",
					QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) != QMessageBox::Yes)
				{
					setFocus();
					return;
				}
				row = 0;

			} else row++;

			col = 0;
			l = m_pLines->at(row);
		}
	}

}

void SSEXEditor::switchMode()
{
	switch(m_mode)
	{
		case Normal: setMode(Cpp);    break;
		case Cpp:    setMode(Html);   break;
		default:     setMode(Normal); break;
	}
}

void SSEXEditor::setMode(ColorMode mode)
{
	if(mode != m_mode)
	{
		m_mode = mode;
		emit modeChanged(this,m_mode);
		initializeCurrentMode();
		update();
	}
}

void SSEXEditor::recordKeystrokes()
{
	if(m_bRecordingKeystrokes)
	{
		m_bRecordingKeystrokes = false;
		emit recordingKeystrokes(false);
		return;
	}
	m_pKeystrokes->clear();
	m_bRecordingKeystrokes = true;
	emit recordingKeystrokes(true);
}

#undef KeyPress

void SSEXEditor::replayKeystrokes()
{
	if(m_bRecordingKeystrokes)
	{
		m_bRecordingKeystrokes = false;
		emit recordingKeystrokes(false);
		return;
	}

	for(SSEXEditorKeystroke *k=m_pKeystrokes->first();k;k=m_pKeystrokes->next())
	{
		QKeyEvent ev(QEvent::KeyPress,k->key,k->ascii,k->state);
		keyPressEvent(&ev);
	}
}

void SSEXEditor::keyPressEvent(QKeyEvent *e)
{
	if(	(e->key() == Qt::Key_Alt) ||
		(e->key() == Qt::Key_Meta) ||
		(e->key() == Qt::Key_Control) ||
		(e->key() == Qt::Key_Shift))
	{
		e->ignore();
		return;
	}

	if(m_bRecordingKeystrokes)
	{
		if(!(((e->key() == Qt::Key_T) || (e->key() == Qt::Key_R)) && (e->state() & ControlButton)))
		{
			SSEXEditorKeystroke * k = new SSEXEditorKeystroke;
			k->ascii = e->ascii();
			k->key = e->key();
			k->state = e->state();
			m_pKeystrokes->append(k);
		}
	}

	m_bCursorOn = true;
	bool bShiftOn = (e->state() & ShiftButton);

	if(e->state() & ControlButton)
	{
//		debug("CONTROL");
		switch(e->key())
		{
			case Qt::Key_C:     copy();                        e->accept(); return; break;
			case Qt::Key_X:     cut();                         e->accept(); return; break;
			case Qt::Key_V:     paste();                       e->accept(); return; break;
			case Qt::Key_M:     switchMode();                  e->accept(); return; break;
			case Qt::Key_I:     indent();                      e->accept(); return; break;
			case Qt::Key_U:     unindent();                    e->accept(); return; break;
			case Qt::Key_O:     commentOut(bShiftOn);          e->accept(); return; break;
			case Qt::Key_P:     removeComment();               e->accept(); return; break;
			case Qt::Key_R:     recordKeystrokes();            e->accept(); return; break;
			case Qt::Key_T:     replayKeystrokes();            e->accept(); return; break;
			case Qt::Key_S:     saveFile();                    e->accept(); return; break;
			case Qt::Key_A:     saveFileAs();                  e->accept(); return; break;
			case Qt::Key_F:     toggleFindWidget();            e->accept(); return; break;
			case Qt::Key_N:     findNext();                    e->accept(); return; break;
			case Qt::Key_H:     findPrev();                    e->accept(); return; break;
			case Qt::Key_B:     findNextRegExp();              e->accept(); return; break;
			case Qt::Key_G:     findPrevRegExp();              e->accept(); return; break;
			case Qt::Key_J:     replace();                     e->accept(); return; break;
			case Qt::Key_K:     replaceAll();                  e->accept(); return; break;
			case Qt::Key_L:     replaceAllRegExp();            e->accept(); return; break;
			case Qt::Key_Q:     replaceAndFindNext();          e->accept(); return; break;
			case Qt::Key_W:     replaceAndFindNextRegExp();    e->accept(); return; break;
			case Qt::Key_E:     replaceAllInSelection();       e->accept(); return; break;
			case Qt::Key_D:     replaceAllInSelectionRegExp(); e->accept(); return; break;
		}
		e->ignore();
		return;
	}


	switch(e->key())
	{
		case Qt::Key_Up:        cursorUp(bShiftOn);         e->accept(); return; break;
		case Qt::Key_Down:      cursorDown(bShiftOn);       e->accept(); return; break;
		case Qt::Key_Right:     cursorRight(bShiftOn);      e->accept(); return; break;
		case Qt::Key_Left:      cursorLeft(bShiftOn);       e->accept(); return; break;
		case Qt::Key_PageUp:    cursorPageUp(bShiftOn);     e->accept(); return; break;
		case Qt::Key_PageDown:  cursorPageDown(bShiftOn);   e->accept(); return; break;
		case Qt::Key_Home:      cursorHome(bShiftOn);       e->accept(); return; break;
		case Qt::Key_End:       cursorEnd(bShiftOn);        e->accept(); return; break;

		case Qt::Key_Backspace: backspace();                e->accept(); return; break;
		case Qt::Key_Delete:    del();                      e->accept(); return; break;
		case Qt::Key_Insert:    m_bOverwrite = !m_bOverwrite; e->accept(); return; break;
		case Qt::Key_Return:
		case Qt::Key_Enter:
			if(m_pFindWidget->isVisible())findNext();
			else newLine();
			e->accept();
			return;
		break;
		default:
			if(    ((((unsigned char)e->ascii()) >= 32) && (((unsigned char)e->ascii()) <= 255)) || (e->ascii() == '\t')){
				insertChar(e->ascii());
				e->accept();
				return;
			}
			break;
	}

	e->ignore();
}

void SSEXEditor::mousePressEvent(QMouseEvent *e)
{
	if(e->button() & LeftButton){
		m_iCursorRow = findRow(e->pos().y());
		if(m_iCursorRow < 0)m_iCursorRow = m_pLines->count() - 1;
		SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
		m_iCursorPosition = findCharacterAt((e->pos().x() - frameWidth()) + xOffset(),l);
		recalcCursorPosition(l);
		ensureCursorVisible();
		if(m_bHasSelection)clearSelection(false);
		m_selection1 = QPoint(m_iCursorPosition,m_iCursorRow);
		m_selection2 = m_selection1;
		m_mouseAnchor = m_selection1;
		m_lastMousePressCoords = QPoint(e->pos().x() + xOffset(),e->pos().y() + yOffset());
		update();
	} else if(e->button() & MidButton)paste();
	else if(e->button() & RightButton)
	{
		int theRow = findRow(e->pos().y());
		if(theRow < 0)theRow = m_pLines->count() - 1;
		SSEXEditorTextLine * l = m_pLines->at(theRow);
		int charIndex = findCharacterAt((e->pos().x() - frameWidth()) + xOffset(),l);
		if(charIndex >= l->length)charIndex = l->length - 1;
		contextPopup(l,charIndex);
	}
	QTableView::mousePressEvent(e);
}

void SSEXEditor::contextPopup(SSEXEditorTextLine *l,int charIndex)
{
	m_pContextPopup->clear();
	int id = m_pContextPopup->insertItem("&Copy",this,SLOT(copy()),QAccel::stringToKey("Ctrl+C"));
	if(!m_bHasSelection)m_pContextPopup->setItemEnabled(id,false);
	id = m_pContextPopup->insertItem("&Paste",this,SLOT(paste()),QAccel::stringToKey("Ctrl+V"));
	QString t = QApplication::clipboard()->text();
	if(t.isNull() || t.isEmpty())m_pContextPopup->setItemEnabled(id,false);
	id = m_pContextPopup->insertItem("&Cut",this,SLOT(cut()),QAccel::stringToKey("Ctrl+X"));
	if(!m_bHasSelection)m_pContextPopup->setItemEnabled(id,false);
	m_pContextPopup->insertSeparator();

	m_pFindPopup->clear();

	m_pFindPopup->insertItem(m_pFindWidget->isVisible() ? "Hide &Find Widget" : "Show &Find Widget",this,SLOT(toggleFindWidget()),QAccel::stringToKey("Ctrl+F"));
	m_pFindPopup->insertSeparator();

	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().local8Bit();
	bool bHasToFind = !(toFind.isEmpty() || toFind.isNull());
	id = m_pFindPopup->insertItem("Find &Next",this,SLOT(findNext()),QAccel::stringToKey("Ctrl+N"));
	if(!bHasToFind)m_pFindPopup->setItemEnabled(id,false);
	id = m_pFindPopup->insertItem("Find &Previous",this,SLOT(findPrev()),QAccel::stringToKey("Ctrl+P"));
	if(!bHasToFind)m_pFindPopup->setItemEnabled(id,false);
	id = m_pFindPopup->insertItem("Find Next &RegExp",this,SLOT(findNextRegExp()),QAccel::stringToKey("Ctrl+B"));
	if(!bHasToFind)m_pFindPopup->setItemEnabled(id,false);
	id = m_pFindPopup->insertItem("Find Previous Reg&Exp",this,SLOT(findPrevRegExp()),QAccel::stringToKey("Ctrl+G"));
	if(!bHasToFind)m_pFindPopup->setItemEnabled(id,false);
	m_pFindPopup->insertSeparator();
	id = m_pFindPopup->insertItem("Rep&lace",this,SLOT(replace()),QAccel::stringToKey("Ctrl+J"));
	if(!m_bHasSelection)m_pFindPopup->setItemEnabled(id,false);
	id = m_pFindPopup->insertItem("Replace &All",this,SLOT(replaceAll()),QAccel::stringToKey("Ctrl+K"));
	if(!bHasToFind)m_pFindPopup->setItemEnabled(id,false);
	id = m_pFindPopup->insertItem("Replace All (Re&gExp)",this,SLOT(replaceAllRegExp()),QAccel::stringToKey("Ctrl+L"));
	if(!bHasToFind)m_pFindPopup->setItemEnabled(id,false);
	id = m_pFindPopup->insertItem("Replace All in &Selection",this,SLOT(replaceAllInSelection()),QAccel::stringToKey("Ctrl+E"));
	if(!(bHasToFind && m_bHasSelection))m_pFindPopup->setItemEnabled(id,false);
	id = m_pFindPopup->insertItem("Replace All &in Selection (RegExp)",this,SLOT(replaceAllInSelectionRegExp()),QAccel::stringToKey("Ctrl+D"));
	if(!(bHasToFind && m_bHasSelection))m_pFindPopup->setItemEnabled(id,false);

	m_pContextPopup->insertItem("&Find",m_pFindPopup);
	m_pContextPopup->insertSeparator();
	id = m_pContextPopup->insertItem("&Save",this,SLOT(saveFile()),QAccel::stringToKey("Ctrl+S"));
	if(!m_bModified)m_pContextPopup->setItemEnabled(id,false);
	m_pContextPopup->insertItem("Save &As...",this,SLOT(saveFileAs()),QAccel::stringToKey("Ctrl+A"));
	m_pContextPopup->insertSeparator();
	m_pContextPopup->insertItem(m_bRecordingKeystrokes ? "Stop &Recording Keystrokes" : "&Record Keystrokes",this,SLOT(recordKeystrokes()),QAccel::stringToKey("Ctrl+R"));
	id = m_pContextPopup->insertItem("Replay Keys&trokes",this,SLOT(replayKeystrokes()),QAccel::stringToKey("Ctrl+T"));
	if(m_bRecordingKeystrokes || (m_pKeystrokes->count() == 0))m_pContextPopup->setItemEnabled(id,false);

	emit rightClickOnTextRow(this,l->text,charIndex,m_pContextPopup);

	m_pContextPopup->popup(QCursor::pos());
}

void SSEXEditor::dragTimer()
{
	// Send fake mouse events....
	QMouseEvent e(QEvent::MouseMove,mapFromGlobal(QCursor::pos()),LeftButton,LeftButton);
	mouseMoveEvent(&e);
}

void SSEXEditor::mouseMoveEvent(QMouseEvent *e)
{
	if(e->state() & LeftButton)
	{
		QPoint mousePos = e->pos();

		int lastCursorYPosition = m_iCursorRow;

		int rowUnderMouse = findRow(e->pos().y());
		if(rowUnderMouse < 0){
			if(!m_pDragTimer->isActive())m_pDragTimer->start(SSEX_EDITOR_DRAG_TIMEOUT);
			// Not found...we're out of the widget ?
			if(mousePos.y() <= frameWidth()){
				if(m_iCursorRow > 0)rowUnderMouse = m_iCursorRow - 1;
				else return;
				mousePos.setY(frameWidth() + 1);
			} else {
				if(((int)m_iCursorRow) < ((int)(m_pLines->count() - 1)))rowUnderMouse = m_iCursorRow + 1;
				else return;
				mousePos.setY(frameWidth() + viewHeight() - 1);
			}
		} else if(m_pDragTimer->isActive())m_pDragTimer->stop();

		m_iCursorRow = rowUnderMouse;

		SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
		m_iCursorPosition = findCharacterAt((mousePos.x() - frameWidth()) + xOffset(),l);
		recalcCursorPosition(l);
		ensureCursorVisible();

		if(!m_bHasSelection)setHasSelection(true);

		lastCursorYPosition = ((lastCursorYPosition * cellHeight()) - yOffset()) + frameWidth();
		
		if(m_mouseAnchor.y() == m_iCursorRow){
			if(m_lastMousePressCoords.x() <= mousePos.x())
			{
				m_selection1 = m_mouseAnchor;
				m_selection2 = QPoint(m_iCursorPosition,m_iCursorRow);
			} else {
				m_selection2 = m_mouseAnchor;
				m_selection1 = QPoint(m_iCursorPosition,m_iCursorRow);
			}
		} else {
			if(m_lastMousePressCoords.y() <= (mousePos.y() + yOffset()))
			{
				m_selection1 = m_mouseAnchor;
				m_selection2 = QPoint(m_iCursorPosition,m_iCursorRow);
			} else {
				m_selection2 = m_mouseAnchor;
				m_selection1 = QPoint(m_iCursorPosition,m_iCursorRow);
			}
		}

		int cH = cellHeight() + 1;
		if(lastCursorYPosition > mousePos.y())update(QRect(0,mousePos.y() - cH,width(),(lastCursorYPosition - mousePos.y()) + (cH << 1)));
		else update(QRect(0,lastCursorYPosition - cH,width(),(mousePos.y() - lastCursorYPosition) + (cH << 1)));
	}
}

void SSEXEditor::mouseReleaseEvent(QMouseEvent *)
{
	if(m_pDragTimer->isActive())m_pDragTimer->stop();
}

void SSEXEditor::updateCellSize()
{
	setCellHeight(m_iFontLineSpacing);
	setCellWidth(m_iMaxTextWidth + (SSEX_EDITOR_BORDER * 2));
	m_pMemBuffer->resize(cellWidth() + 1,cellHeight() + 1);
}


void SSEXEditor::updateFontDependantVariables()
{
	QFontMetrics fm(font());
	int spaceSize = fm.width(' ');
	m_iTabsNumPixels = spaceSize * m_iTabsNumSpaces;
	for(int i=0;i<256;i++)m_iCharWidth[i] = fm.width((char)i);
	m_iFontAscent = fm.ascent();
	m_iFontLineSpacing = fm.lineSpacing();
	updateCellSize();
}

void SSEXEditor::setFont(const QFont &f)
{
	QTableView::setFont(f);
	updateFontDependantVariables();
	if(((int)m_pLines->count()) > ((int)m_iCursorRow))
	{
		SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
		if(l)recalcCursorPosition(l);
	}
	update();
}

void SSEXEditor::clear()
{
	setText("");
	setModified(true);
}

int SSEXEditor::getTextWidthWithTabsForCursorPosition(const char *text,int cursorPos)
{
	int totWidth = 0;
	int lastTabStop = 0;
	while(*text && cursorPos)
	{
		if(*text == '\t')
		{
			while(lastTabStop <= totWidth)lastTabStop += m_iTabsNumPixels;
			totWidth = lastTabStop;
		} else totWidth += m_iCharWidth[(unsigned char)(*text)];
		text++;
		if(!(--cursorPos))return totWidth;
	}
	return totWidth;
}

int SSEXEditor::findCharacterAt(int xPositionInCell,SSEXEditorTextLine * l)
{
	const char *aux = l->text.data();
	int curXPos = SSEX_EDITOR_BORDER;
	int lastTabStop = SSEX_EDITOR_BORDER;
	while(*aux)
	{
		if(*aux == '\t')
		{
			while(lastTabStop <= curXPos)lastTabStop += m_iTabsNumPixels;
			if(xPositionInCell < (curXPos + ((lastTabStop - curXPos) >> 1)))return (aux - l->text.data());
			else {
				curXPos = lastTabStop;
				aux++;
			}
		} else {
			if(xPositionInCell < (curXPos + (m_iCharWidth[(unsigned char)*aux] >> 1)))return (aux - l->text.data());
			else {
				curXPos += m_iCharWidth[(unsigned char)*aux];
				aux++;
			}
		}
	}
	return l->length;
}

int SSEXEditor::getTextWidthWithTabs(const char *text)
{
	int totWidth = 0;
	int lastTabStop = 0;
	while(*text)
	{
		if(*text == '\t')
		{
			while(lastTabStop <= totWidth)lastTabStop += m_iTabsNumPixels;
			totWidth = lastTabStop;
		} else totWidth += m_iCharWidth[(unsigned char)*text];
		text++;
	}
	return totWidth;
}

void SSEXEditor::setText(const QCString &text)
{
	m_iCursorRow = 0;
	m_iCursorPosition = 0;

	m_pLines->clear();

	const char *aux = text.data();

	if(!aux)aux = "";

	do {
		const char * begin = aux;
		while(*aux && (*aux != '\n'))aux++;
		SSEXEditorTextLine * l = new SSEXEditorTextLine;
		l->length = aux - begin;
		l->text.resize(l->length + 1);
		memmove(l->text.data(),begin,(aux - begin));
		*(l->text.data() + l->length) = '\0';
		if(*aux)aux++;
		l->width = getTextWidthWithTabs(l->text.data());
		l->flags = 0;
		m_pLines->append(l);
	} while(*aux);

	initializeCurrentMode();

	updateMaxTextWidth();
	updateCellSize();
	recalcCursorPosition(m_pLines->first());

	setNumRows(m_pLines->count());
	setTopLeftCell(0,0);

	update();
}

bool SSEXEditor::loadFile(const char *filename)
{
	QFile f(filename);
	if(!f.open(IO_ReadOnly))return false;
	QCString buffer;
	buffer.resize(f.size() + 1);
	int readed = f.readBlock(buffer.data(),f.size());
	*(buffer.data() + readed) = '\0';
	f.close();
	QCString fExtension = filename;
	int idx = fExtension.findRev('.');
	ColorMode mode = Normal;
	if(idx != -1)
	{
		fExtension.remove(0,idx+1);
		if(strcmp(fExtension.data(),"cpp") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"c") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"C") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"cxx") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"CPP") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"h") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"H") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"hxx") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"hpp") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"HPP") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"moc") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"s") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"xpm") == 0)mode = Cpp;
		else if(strcmp(fExtension.data(),"html") == 0)mode = Html;
		else if(strcmp(fExtension.data(),"Html") == 0)mode = Html;
		else if(strcmp(fExtension.data(),"htm") == 0)mode = Html;
		else if(strcmp(fExtension.data(),"HTML") == 0)mode = Html;
		else if(strcmp(fExtension.data(),"HTM") == 0)mode = Html;
		else if(strcmp(fExtension.data(),"SGML") == 0)mode = Html;
		else if(strcmp(fExtension.data(),"sgml") == 0)mode = Html;
		else if(strcmp(fExtension.data(),"xml") == 0)mode = Html;
		else if(strcmp(fExtension.data(),"XML") == 0)mode = Html;
	}
	setMode(mode);
	setText(buffer);
	m_szFileName = filename;
	setModified(false);
	emit fileNameChanged(this,m_szFileName);
	return true;
}

bool SSEXEditor::saveFile()
{
	if(m_szFileName.isEmpty() || m_szFileName.isNull())return saveFileAs();
	else return saveFile(m_szFileName.data());
}

bool SSEXEditor::saveFile(const char *filename)
{
	QFile f(filename);
	if(!f.open(IO_WriteOnly)){
		QMessageBox::warning(this,"Warning","Cannot open file for writing\nSave failed",QMessageBox::Ok | QMessageBox::Default,0);
		return false;
	}
	int i = 0;
	int lastProg = -1;
	int progress;
	if(m_pLines->count() == 0){
		emit saveProgress(m_szFileName,100);
	} else {
		emit saveProgress(m_szFileName,0);
		for(SSEXEditorTextLine *l=m_pLines->first();l;l=m_pLines->next())
		{
//			__assert(l->text.length() == l->length);
//#ifdef __DEBUG
//			if(l->text.length() != l->length)debug("WARNING : Wrong line size (%d) , should be %d for (%s)",l->length,l->text.length(),l->text.data());
//#endif
			if((f.writeBlock(l->text.data(),l->length) != l->length) || (f.writeBlock("\n",1) != 1))
			i++;
			progress = (i * 100) / m_pLines->count();
			if(progress != lastProg){
				emit saveProgress(m_szFileName,progress);
				lastProg = progress;
			}
		}
	}
	f.close();
	if(strcmp(m_szFileName.data(),filename) != 0){
		m_szFileName = filename;
		emit fileNameChanged(this,m_szFileName);
	}
	setModified(false);
	emit saved(m_szFileName);
	return true;
}

bool SSEXEditor::saveFileAs()
{
	QString newName = QFileDialog::getSaveFileName(m_szFileName,QString::null,0);
	if(newName.isNull())return false; //Aborted
	QFileInfo fi(newName);
	if(fi.exists())
	{
		QString msg;
		msg.sprintf("The file %s already exists\nDo you wish to overwrite it?",newName.ascii());
		int result = QMessageBox::information(this,"Warning",msg,QMessageBox::Yes,QMessageBox::No | QMessageBox::Default);
		if(result != QMessageBox::Yes)return false; // Aborted
	}
	return saveFile(newName.ascii());
}

void SSEXEditor::initializeCurrentMode()
{
	QPalette pal = palette();
	switch(m_mode)
	{
		case Cpp:
			cppModeComputeCommentState(m_pLines->first());
			setBackgroundColor(m_pColors->cppExtBackground);
			setBackgroundMode(NoBackground);
			setFont(m_pColors->cppFont);
		break;
		case Html:
			htmlModeComputeTagState(m_pLines->first());
			setBackgroundColor(m_pColors->htmlExtBackground);
			setBackgroundMode(NoBackground);
			setFont(m_pColors->htmlFont);
		break;
		default:
			setBackgroundColor(m_pColors->extBackground);
			setBackgroundMode(NoBackground);
			setFont(m_pColors->font);
		break;
	}
}

void SSEXEditor::cppModeComputeCommentState(SSEXEditorTextLine * start)
{
	if(!start)return;
	if(m_pLines->findRef(start) < 0)return;
	bool bInComment = (start->flags & SSEX_EDITOR_BEGIN_IN_COMMENT);
	for(SSEXEditorTextLine * l=start;l;l=m_pLines->next())
	{
		l->flags = (bInComment ? SSEX_EDITOR_BEGIN_IN_COMMENT : 0);
		bool bInString = false;
		bool bInChar   = false;
		const char *c = l->text.data();
		while(*c)
		{
			if(bInComment){
				while(*c){
					if(*c == '*'){
						c++;
						if(*c == '/'){
							c++;
							bInComment = false;
							break;
						}
					} else c++;
				}
			} else {	
				while(*c){
					if(*c == '/'){
						c++;
						if(!(bInString || bInChar)){
							if(*c == '/'){
								// line comment
								while(*c)c++;
							} else if(*c =='*'){
								c++;
								bInComment = true;
								break;
							}
						} // else in string..no comment begins
					} else if(*c == '"'){
						// no comment begins in strings
						if(bInString){
							if(c != l->text.data()){
								const char * aux = c - 1;
								if(*aux != '\\')bInString = !bInString;								
							}
						} else bInString = !bInString;
						c++;
					} else if(*c == '\''){
						// no comment begins in character constants
						if(bInChar){
							if(c != l->text.data()){
								const char * aux = c - 1;
								if(*aux != '\\')bInChar = !bInChar;								
							}
						} else bInChar = !bInChar;
						c++;
					} else c++;
				}
			}
		}
		if(bInComment)l->flags |= SSEX_EDITOR_END_IN_COMMENT;
	}
}


void SSEXEditor::htmlModeComputeTagState(SSEXEditorTextLine * start)
{
	if(!start)return;
	if(m_pLines->findRef(start) < 0)return;

	bool bInComment = (start->flags & SSEX_EDITOR_BEGIN_IN_COMMENT);
	bool bInTag     = (start->flags & SSEX_EDITOR_BEGIN_IN_TAG);

	for(SSEXEditorTextLine * l=start;l;l=m_pLines->next())
	{
		l->flags = (bInComment ? SSEX_EDITOR_BEGIN_IN_COMMENT : (bInTag ? SSEX_EDITOR_BEGIN_IN_TAG : 0));

		const char *c = l->text.data();

		while(*c)
		{
			if(bInComment){
				while(*c){
					if(*c == '-'){
						c++;
						if(!(*c))break;
						if(*c == '-'){
							c++;
							if(!(*c))break;
							if(*c == '>'){
								c++;
								bInComment = false;
								break;
							}
						} else c++;
					} else c++;
				}
			} else if(bInTag){
				while(*c){
					if(*c == '>')
					{
						c++;
						bInTag = false;
						break;
					} else if(*c == '<')
					{
						c++;
						if(!(*c))break;
						if(*c == '!')
						{
							c++;
							if(!(*c))break;
							if(*c == '-')
							{
								c++;
								if(!(*c))break;
								if(*c == '-')
								{
									c++;
									bInTag = false;
									bInComment = true;
									break;
								} else c++;
							} else c++;
						} else c++;
					} else c++;
				}
			} else {
				// not in tag nor comment
				while(*c){
					if(*c == '<'){
						c++;
						bInTag = true;
						if(!(*c))break;
						if(*c == '!')
						{
							c++;
							if(!(*c))break;
							if(*c == '-')
							{
								c++;
								if(!(*c))break;
								if(*c == '-')
								{
									c++;
									bInTag = false;
									bInComment = true;
								}
							}
						}
						break;
					} else c++;
				}
			}
		}

		if(bInComment)l->flags |= SSEX_EDITOR_END_IN_COMMENT;
		else if(bInTag)l->flags |= SSEX_EDITOR_END_IN_TAG;
	}
}

#include "moc_editor.cpp"
