//
//   File : kvi_window.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_window.cpp)
//   Last major modification : Tue Jul 6 1999 14:52:11 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_WINDOW_CPP_

#define _KVI_DEBUG_CHECK_RANGE_
//#define _KVI_DEBUG_CLASS_NAME_ "KviWindow"

#include "kvi_debug.h"
#include "kvi_app.h"

#include "kvi_window.h"
#include "kvi_frame.h"
#include "kvi_taskbar.h"
#include "kvi_ircview.h"
#include "kvi_uparser.h"
#include "kvi_dlgfind.h"

#include "kvi_mdi.h"
#include "kvi_string.h"
#include "kvi_malloc.h"
#include "kvi_listbox.h"
#include "kvi_input.h"
#include "kvi_console.h"
#include "kvi_locale.h"

#include "kvi_options.h"

#include <qdatetime.h>

//============ KviWindow ============//

KviWindow::KviWindow(const char *name,int type,KviFrame *lpFrm)
:QWidget(0,name)
{
	m_szCaption      = name ? name : __tr("Unnamed");
	m_pFrm           = lpFrm;
	m_type           = type;
	m_pTaskBarButton = 0;
	m_pView          = 0;
	m_pFindDialog    = 0;
	m_pListBox       = 0;
	m_pInput         = 0;
	m_pSplitter      = 0;
	m_pFocusHandler  = 0;
	m_pAccelerators  = 0;

#ifdef COMPILE_PLUGIN_SUPPORT
	m_pluginHandle   = 0;
#endif //COMPILE_PLUGIN_SUPPORT

	setMinimumSize(QSize(KVI_WINDOW_MIN_WIDTH,KVI_WINDOW_MIN_HEIGHT));
	setBackgroundMode(NoBackground);
	setFocusPolicy(StrongFocus);
}

//============ ~KviWindow ============//

KviWindow::~KviWindow()
{
	if(m_pFindDialog)delete m_pFindDialog;
	m_pFrm->m_pUserParser->unregisterTimersFor(this);
}

void KviWindow::wheelEvent(QWheelEvent *e)
{
	if(m_pView)m_pView->wheelEvent(e);
	else e->ignore();
}

void KviWindow::installAccelerators()
{
	if(!m_pAccelerators)
	{
		m_pAccelerators = m_pFrm->installAccelerators(this);
	}
}

void KviWindow::destroyAccelerators()
{
	if(m_pAccelerators)
	{
		delete m_pAccelerators;
		m_pAccelerators = 0;
	}
}

void KviWindow::getDefaultLogName(KviStr &str)
{
	QDate dt(QDate::currentDate());
	// Keep the simil-alphabetic order...
	KviStr fName = caption();
	if(g_pOptions->m_bAppendDateToLogName)
	{
		KviStr theDate(KviStr::Format,"_%d_%d_%d",dt.year(),dt.month(),dt.day());
		fName.append(theDate);
	}
	if(g_pOptions->m_bAppendServerNameToLogName){
		if(m_pFrm->m_state == KviFrame::Connected){
			fName.append('_');
			fName.append(m_pFrm->m_global.szCurrentServerHost);
		}
	}
	fName.append(".log");
	fName.replaceAll(' ',"_");
	g_pApp->getLocalKVircDirectory(str,KviApp::Log,fName.ptr(),true);
}

//============== findText ==============//

void KviWindow::findText()
{
	if(m_pView){
		if(!m_pFindDialog)m_pFindDialog = new KviFindDialog(this);
		if(mdiParent()&&isMinimized())mdiParent()->setState(KviMdiChild::Normal);
		else showNormal();
		setFocus();
		m_pFindDialog->exec();
	}
}

//============= setTaskBarButton ==============//

void KviWindow::setTaskBarButton(KviTaskBarButton *btn_ptr)
{
	m_pTaskBarButton = btn_ptr;
}

QRect KviWindow::externalGeometry()
{
	return mdiParent() ? mdiParent()->geometry() : geometry();
}

//============== minimize ==============//

void KviWindow::minimize(bool bAnimate)
{
	if(mdiParent()){
		if(!isMinimized()){
			mdiParent()->setState(KviMdiChild::Minimized,bAnimate);
		}
	} else showMinimized();
}

void KviWindow::minimize(){ minimize(true); }

//============= maximize ==============//

void KviWindow::maximize(bool bAnimate)
{
	if(mdiParent()){
		if(!isMaximized()){
			mdiParent()->setState(KviMdiChild::Maximized,bAnimate);
		}
	} else showMaximized();
}

void KviWindow::maximize(){ maximize(true); }

//============== attach ================//

void KviWindow::attach()
{
	m_pFrm->attachWindow(this);
}

//============== detach =================//

void KviWindow::detach()
{
	m_pFrm->detachWindow(this);
}

void KviWindow::addWidgetToSplitter(QWidget *w)
{
	if(m_pFocusHandler)
	{
		setFocusHandler(m_pFocusHandler,w);
	}
}

//============== setFocusHandler ==============//

void KviWindow::setFocusHandler(QWidget *focusedOne,QWidget *w)
{
	__range_valid(focusedOne);
	if(m_pFocusHandler != 0)m_pFocusHandler = focusedOne;
	if(w==0)w=this;
	if(w != focusedOne)w->setFocusProxy(focusedOne);
	w->installEventFilter(this);
	if(w->focusPolicy() == QWidget::NoFocus)w->setFocusPolicy(QWidget::ClickFocus);
	QList<QObject> *list = (QList<QObject> *)(w->children());
	if(list){
		for (unsigned int i=0; i< list->count(); i++){
			QObject *child = list->at(i);
			if(child->inherits("QWidget"))setFocusHandler(focusedOne,((QWidget *)child));
		}
	}
}

void KviWindow::setFocusHandlerNoChildren(QWidget *focusedOne,QWidget *w,bool bInChildTree)
{
	__range_valid(focusedOne);
	if(m_pFocusHandler != 0)m_pFocusHandler = focusedOne;
	if(w==0)w=this;
	if(w != focusedOne)
	{
		if(!bInChildTree)w->setFocusProxy(focusedOne);
	} else bInChildTree = true;

	w->installEventFilter(this);
	if(w->focusPolicy() == QWidget::NoFocus)w->setFocusPolicy(QWidget::ClickFocus);
	QList<QObject> *list = (QList<QObject> *)(w->children());
	if(list){
		for (unsigned int i=0; i< list->count(); i++){
			QObject *child = list->at(i);
			if(child->inherits("QWidget"))setFocusHandlerNoChildren(focusedOne,((QWidget *)child),bInChildTree);
		}
	}
}

void KviWindow::setFocusHandlerNoClass(QWidget *focusedOne,const char *noClass,QWidget *w)
{
	__range_valid(focusedOne);
	__range_valid(noClass);
	if(m_pFocusHandler != 0)m_pFocusHandler = focusedOne;
	if(w==0)w=this;
	if(w != focusedOne)
	{
		if(!w->inherits(noClass))w->setFocusProxy(focusedOne);
	}
	w->installEventFilter(this);
	if(w->focusPolicy() == QWidget::NoFocus)w->setFocusPolicy(QWidget::ClickFocus);
	QList<QObject> *list = (QList<QObject> *)(w->children());
	if(list){
		for (unsigned int i=0; i< list->count(); i++){
			QObject *child = list->at(i);
			if(child->inherits("QWidget"))setFocusHandlerNoClass(focusedOne,noClass,((QWidget *)child));
		}
	}
}

//============== eventFilter =================//
#ifdef FocusIn
	// Hack for X.h
	#undef FocusIn
#endif

bool KviWindow::eventFilter(QObject *o,QEvent *e)
{
	if(e->type() == QEvent::FocusIn)m_pFrm->childWindowGainFocus(this);
	return false;
}

//=============== isMinimized ? =================//

bool KviWindow::isMinimized()
{
	if(mdiParent())return (mdiParent()->state() == KviMdiChild::Minimized);
	else return QWidget::isMinimized();
}

//============== isMaximized ? ==================//

bool KviWindow::isMaximized()
{
	if(mdiParent())return (mdiParent()->state() == KviMdiChild::Maximized);
	// Heh...how to check it ?
	// Empirical check
	int wdth = (g_pApp->desktop()->width() * 75) / 100;
	int hght = (g_pApp->desktop()->height() * 75) / 100;
	
	return ((x() <= 1)&&(y() <= 1)&&(width() >= wdth)&&(height() >= hght));
}

//============== restore ================//

void KviWindow::restore()
{
	if(mdiParent()){
		if(isMinimized()||isMaximized())mdiParent()->setState(KviMdiChild::Normal);
	} else showNormal();
}

//=============== youAreAttached ============//

void KviWindow::youAreAttached(KviMdiChild *lpC)
{
	lpC->setCaption(m_szCaption.ptr());
	lpC->setIconPointer(myIconPtr()); //It is ok to set a NULL icon too here
}

//================ youAreDetached =============//

void KviWindow::youAreDetached()
{
	setCaption(m_szCaption.ptr());
	if(myIconPtr())setIcon(*(myIconPtr()));
	setFocusPolicy(QWidget::StrongFocus);
}

//================ setWindowCaption ================//

void KviWindow::setWindowCaption(const char *szCaption)
{
	m_szCaption=szCaption;
	if(mdiParent()){
		mdiParent()->setCaption(m_szCaption.ptr());
		if(isMaximized())m_pFrm->updateCaption();
	} else setCaption(m_szCaption.ptr());
	if(m_pTaskBarButton)m_pTaskBarButton->updateButton();
}

//============== closeEvent ================//

void KviWindow::closeEvent(QCloseEvent *e)
{
	e->ignore(); //we ignore the event , and then close later if needed.
	m_pFrm->childWindowCloseRequest(this);
}

//============== output ================//

void KviWindow::output(int msg_type,const char *format,...)
{
//#ifdef _KVI_DEBUG_CLASS_NAME_
//	if(m_pView == 0)debug("WARNING : No KviIrcView class has been set.\nYou must override KviWindow::output\nDying.");
//#endif
	char txt_ptr[512]; //It should be enough for all outputs...
	va_list list;
	va_start(list,format);
	if(kvi_vsnprintf(txt_ptr,512,format,list) < 0){
		//Just in case...
		va_end(list);
		int len = 512;
		char *long_txt_ptr = 0;
		int result;
		do{
			len += 512;
			//first time long_txt_ptr == 0 so it is equivalent to malloc
			//At least the man page says that...
			long_txt_ptr = (char *)kvi_realloc((void *)long_txt_ptr,len);
			va_start(list,format);
			result = kvi_vsnprintf(long_txt_ptr,len,format,list);
			va_end(list);
		} while(result < 0);
		if(m_pView)m_pView->appendText(msg_type,long_txt_ptr);
		else {
			// Redirect to the nearest console
			m_pFrm->m_pConsole->outputNoFmt(msg_type,long_txt_ptr);
		}
		kvi_free((void *)long_txt_ptr);
	} else {
		//Succesful vsnprintf
		va_end(list);
		if(m_pView)m_pView->appendText(msg_type,txt_ptr);
		else {
			// Redirect to the nearest console
			m_pFrm->m_pConsole->outputNoFmt(msg_type,txt_ptr);
		}
	}
	if(m_pTaskBarButton)m_pTaskBarButton->highlight((msg_type == KVI_OUT_HIGHLIGHT));
}

//================ outputNoFmt ================//

void KviWindow::outputNoFmt(int msg_type,const char * text)
{
	__range_valid(text);
	if(m_pView)m_pView->appendText(msg_type,text);
	else m_pFrm->m_pConsole->outputNoFmt(msg_type,text);
	if(m_pTaskBarButton)m_pTaskBarButton->highlight((msg_type == KVI_OUT_HIGHLIGHT));
}

//================ myIconPtr =================//

QPixmap * KviWindow::myIconPtr()
{
	__debug("WARNING: No icon has been set for this window");
	return 0;
}

//================ applyOptions =================//

void KviWindow::applyOptions()
{
	//Nothing here
}

//=============== highlight =================//

void KviWindow::highlight(bool bAltColor)
{
	if(m_pTaskBarButton)m_pTaskBarButton->highlight(bAltColor);
}

//============ setProgress ==============//

void KviWindow::setProgress(int progress)
{
	if(m_pTaskBarButton)m_pTaskBarButton->setProgress(progress);
}

void KviWindow::setProperties(KviWindowProperty *)
{
}

void KviWindow::saveProperties()
{
}

void KviWindow::delayedAutoRaise()
{
	QTimer::singleShot(100,this,SLOT(autoRaise()));
}

void KviWindow::autoRaise()
{
	if(isMinimized())restore();
	raise();
	setFocus();
}

#include "m_kvi_window.moc"
