//
//   File : kvi_opaquelabel.cpp
//   Last major modification : Thu Jan 23 2000 02:15:12 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_opaquelabel.h"
#include "kvi_xutils.h"
#include "kvi_defines.h"
#include "kvi_malloc.h"

#include <qfontmetrics.h>
#include <qpainter.h>

#define KVI_CHAN_LABEL_BORDER 2

#ifdef COMPILE_USE_AA_FONTS
	XftFont        * g_pXftFont;
	XftDraw        * g_pXftDraw;
	extern int qt_use_xft (void); // qpainter_x11.cpp
	extern void *qt_ft_font (const QFont *f); // qfont_x11.cpp
	extern XftDraw * qt_lookup_ft_draw (Drawable draw, bool paintEventClipOn, QRegion *crgn);
#endif

//
//   A non themeable label
//

KviOpaqueLabel::KviOpaqueLabel(QWidget *parent,const char *name)
:QLabel(parent,name)
{
}

KviOpaqueLabel::KviOpaqueLabel(const QString &text,QWidget *parent,const char *name)
:QLabel(text,parent,name)
{
}

KviOpaqueLabel::~KviOpaqueLabel()
{
}

// Kill these methods


void KviOpaqueLabel::setMask(const QBitmap &)
{
}

void KviOpaqueLabel::setMask(const QRegion &)
{
}

void KviOpaqueLabel::setAutoMask(bool)
{
}

//
//   A non themeable (from outside) label
//   Used for the channel labels
//

KviChanLabel::KviChanLabel(const char * text,QWidget *parent,const char *name)
:QWidget(parent,name)
{
	m_bCanEdit = false;
	m_szText = text ? text : "";
	m_iFirstVisibleChar = 0;
	m_textColor = colorGroup().text();
	m_backColor = colorGroup().button();
	m_pPixPointer = 0;
	setBackgroundMode(NoBackground);
	
	m_bLeft = new QToolButton(LeftArrow,this);
	m_bRight = new QToolButton(RightArrow,this);

	m_bLeft->setFocusPolicy(QWidget::NoFocus);
	m_bLeft->setAutoRepeat(true);

	m_bRight->setFocusPolicy(QWidget::NoFocus);
	m_bRight->setAutoRepeat(true);

	m_bLeft->hide();
	m_bRight->hide();

	connect(m_bLeft,SIGNAL(pressed()),this,SLOT(scrollLeft()));
	connect(m_bRight,SIGNAL(pressed()),this,SLOT(scrollRight()));
}

KviChanLabel::~KviChanLabel()
{
	delete m_bLeft,m_bRight;
}

const char * KviChanLabel::strippedText()
{
	KviStr text;
	char * p = m_szText.ptr();
	while(*p){
		switch(*p){
			case KVI_TEXT_BOLD:
			case KVI_TEXT_REVERSE:
			case KVI_TEXT_RESET:
			case KVI_TEXT_UNDERLINE:
				p++;
			break;
			case KVI_TEXT_COLOR:
				p++;
				if((*p)&&(*p >= '0')&&(*p <= '9')){
					p++;
					if((*p)&&(*p >= '0')&&(*p <= '9'))
						p++;
					if((*p)&&(*p == ',')){
						p++;
						if((*p)&&(*p >= '0')&&(*p <= '9'))
							p++;
						if((*p)&&(*p >= '0')&&(*p <= '9'))
							p++;
					}
				}
			break;
			default:
				text.append(*p);
				p++;
			break;
		}
	}
	p = (char *)kvi_malloc(text.len()+1);
        qstrcpy(p, text.ptr());
	return p;
}

void KviChanLabel::setBackgroundPixmapPointer(QPixmap * pix)
{
	m_pPixPointer = pix;
	update();
}

void KviChanLabel::setTextColor(QColor &clr)
{
	m_textColor = clr;
	update();
}

void KviChanLabel::setBackColor(QColor &clr)
{
	m_backColor = clr;
	update();
}

void KviChanLabel::paintEvent(QPaintEvent *e)
{
	QPixmap * pix = new QPixmap(width(),height());
	QColorGroup g = colorGroup();
	Display * dpy = qt_xdisplay();
	GC the_gc = XCreateGC(dpy,pix->handle(),0,0);

	if(m_pPixPointer)
	{
		XSetTile(dpy,the_gc,m_pPixPointer->handle());
		XSetFillStyle(dpy,the_gc,FillTiled);
	} else {
		XSetForeground(dpy,the_gc,m_backColor.pixel());
		XSetBackground(dpy,the_gc,m_backColor.pixel());
		XSetFillStyle(dpy,the_gc,FillSolid);
	}
	XFillRectangle(dpy,pix->handle(),the_gc,0,0,width(),height());

	kvi_xDraw3dRect(dpy,the_gc,pix->handle(),colorGroup(),0,0,width() - 1,height() - 1,true);

	KviStr pT(strippedText());
	if(m_iFirstVisibleChar < 0)m_iFirstVisibleChar = 0;
	KviStr t(pT.right(pT.len()-m_iFirstVisibleChar));
	if(fontMetrics().width(pT.ptr()) < width()){
		m_bLeft->hide();
		m_bRight->hide();
	} else {
		QSize bs;
		bs.setHeight(height());
		bs.setWidth(height()*5/6); // 5/8 - reciprocal golden mean
		m_bLeft->resize(bs);
		m_bRight->resize(bs);
		int buttonWidth = m_bRight->width();
		int x = width() - buttonWidth;
		m_bLeft->move(x-buttonWidth,0);
		m_bLeft->setEnabled((m_iFirstVisibleChar != 0));
		m_bRight->move(x,0);
		m_bRight->setEnabled(fontMetrics().width(t.ptr()) >= width()-m_bLeft->width()-m_bRight->width());
		m_bLeft->show();
		m_bRight->show();
	}

	if(t.len())
	{
//		const char * t = m_szText.ptr();
		XSetForeground(dpy,the_gc,m_textColor.pixel());
#ifdef COMPILE_USE_AA_FONTS
		if(qt_use_xft())
		{
			g_pXftFont = (XftFont *)qt_ft_font(&(font()));
			g_pXftDraw = qt_lookup_ft_draw (pix->handle(),false,0);
			if(!g_pXftDraw)
			{
				XSetFont(dpy,the_gc,font().handle());
				g_pXftFont = 0;
			}
		} else {
#endif
			XSetFont(dpy,the_gc,font().handle());
#ifdef COMPILE_USE_AA_FONTS
			g_pXftFont = 0;
			g_pXftDraw = 0;
		}
		if(g_pXftFont){
			XftColor color;
			QColor * clr = &m_textColor;
			color.color.red = clr->red() | clr->red() << 8;
			color.color.green = clr->green() | clr->green() << 8;
			color.color.blue = clr->blue() | clr->blue() << 8;
			color.color.alpha = 0xffff;
			color.pixel = clr->pixel();
			XftDrawString8(g_pXftDraw,&color,g_pXftFont,KVI_CHAN_LABEL_BORDER,
				(height() >> 1) + (fontMetrics().height() >> 1) - (fontMetrics().descent() + 1),(unsigned char *)t.ptr(),t.len());
		} else
#endif
			XDrawString(dpy,pix->handle(),the_gc,KVI_CHAN_LABEL_BORDER,
				(height() >> 1) + (fontMetrics().height() >> 1) - (fontMetrics().descent() + 1),t.ptr(),t.len());
	}

	XCopyArea(dpy,pix->handle(),this->handle(),the_gc,0,0,width(),height(),0,0);

	delete pix;
	XFreeGC(dpy,the_gc);
}

void KviChanLabel::setText(const char * text)
{
	m_szText = text ? text : "";
	m_iFirstVisibleChar = 0;
	update();
}

void KviChanLabel::scrollLeft()
{
	if(m_iFirstVisibleChar)m_iFirstVisibleChar--;
	update();
}

void KviChanLabel::scrollRight()
{
	KviStr pT(strippedText());
	KviStr t(pT.right(pT.len()-m_iFirstVisibleChar));

	if(fontMetrics().width(t.ptr()) >= width()-m_bLeft->width()-m_bRight->width())
		m_iFirstVisibleChar++;
	update();
}

void KviChanLabel::mousePressEvent(QMouseEvent *e)
{
	if(e->button() & RightButton)emit rightClicked();
	else QWidget::mousePressEvent(e);
}

void KviChanLabel::mouseDoubleClickEvent(QMouseEvent *)
{
	m_bCanEdit = true;
}

void KviChanLabel::mouseReleaseEvent(QMouseEvent *)
{ 
	if(m_bCanEdit)
	{
		emit doubleClicked();
		m_bCanEdit = false;
	}
}

QSize KviChanLabel::sizeHint()
{
	QFontMetrics fm = fontMetrics();
	return QSize(100,fm.height() + (KVI_CHAN_LABEL_BORDER << 1));
}

#include "m_kvi_opaquelabel.moc"
