//
//   File : kvi_ircserver.cpp (/usr/build/NEW_kvirc/kvirc/kvilib/kvi_ircserver.cpp)
//   Last major modification : Thu Jan 14 1999 18:09:53 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CHECK_RANGE_
//#define _KVI_DEBUG_CLASS_NAME_ "KviIrcServerManager"
#include "kvi_debug.h"

#include "kvi_ircserver.h"
#include "kvi_netutils.h"
#include "kvi_locale.h"
#include <qfile.h>

//===========================================================================//
// KviIrcNetwork class
//===========================================================================//

KviIrcNetwork::KviIrcNetwork(const char *szName)
{
	m_szName = szName;
	m_pServerList = new QList<KviIrcServer>;
	m_pServerList->setAutoDelete(true);
	m_pCurrentServer = 0;
}

KviIrcNetwork::KviIrcNetwork(KviIrcNetwork *net)
{
	__range_valid(net);
	m_pServerList = new QList<KviIrcServer>;
	m_pServerList->setAutoDelete(true);
	copyFrom(net);
}

KviIrcNetwork::~KviIrcNetwork()
{
	clear();
	delete m_pServerList;
}

QList<KviIrcServer> * KviIrcNetwork::serverList()
{
	return m_pServerList;
}

void KviIrcNetwork::clear()
{
	m_szName = "UNNAMED";
	while(!m_pServerList->isEmpty()){
		KviIrcServer *s=m_pServerList->first();
		removeServer(s);
	}
	m_pCurrentServer = 0;
}

uint KviIrcNetwork::serverCount()
{
	return m_pServerList->count();
}

void KviIrcNetwork::appendServer(KviIrcServer * ptr)
{
	__range_valid(ptr);
	m_pServerList->append(ptr);
	m_pCurrentServer = ptr;
}

bool KviIrcNetwork::removeServer(KviIrcServer * ptr)
{
	__range_valid(ptr);
	bool bWasHere = m_pServerList->removeRef(ptr);
	__range_valid(bWasHere);
	if(ptr == m_pCurrentServer){
		if(m_pServerList->isEmpty())m_pCurrentServer = 0;
		else m_pCurrentServer = m_pServerList->first();
	}
	return bWasHere;
}

bool KviIrcNetwork::setCurrentServer(KviIrcServer * ptr)
{
	__range_valid(ptr);
	if(m_pServerList->isEmpty()){
		m_pCurrentServer = 0;
		return false;
	}
	if(m_pServerList->findRef(ptr)==-1)return false;
	m_pCurrentServer = ptr;
	return true;
}

KviIrcServer * KviIrcNetwork::currentServer()
{
	if(m_pServerList->isEmpty())return 0;
	if(m_pCurrentServer){
		__range_valid(m_pServerList->findRef(m_pCurrentServer) != -1); //sanity check
		return m_pCurrentServer;
	}
	m_pCurrentServer = m_pServerList->first();
	return m_pCurrentServer;
}

KviIrcServer * KviIrcNetwork::nextServer()
{
	if(m_pServerList->isEmpty())return 0;
	if(m_pCurrentServer == 0){
		m_pCurrentServer = m_pServerList->first();
		return m_pCurrentServer;
	}
	__range_valid(m_pServerList->findRef(m_pCurrentServer) != -1); //sanity check
	m_pCurrentServer = m_pServerList->next();
	if(!m_pCurrentServer)m_pCurrentServer = m_pServerList->first();
	return m_pCurrentServer;
}

void KviIrcNetwork::serverFromString(KviStr &szSrv,KviIrcServer *srv)
{
	srv->szHost        = "irc.unknown.net";
	srv->szIp          = "127.0.0.1";
	srv->szPassword    = "";
	srv->szDescription = "Unknown";
	srv->szPort        = "6667";
#ifdef COMPILE_NEED_IPV6
	srv->bIpV6         = false;
#endif

	const char *ptr = szSrv.ptr();
	if((!*ptr)||(*ptr==':'))return;
	ptr = kvi_extractUpTo(srv->szHost,ptr,':');
	if(*ptr)ptr++;
	if(!*ptr)return;
	ptr = kvi_extractUpTo(srv->szIp,ptr,':');
	srv->szIp.replaceAll('#',":");
	if(*ptr)ptr++;
	if(!*ptr)return;
	ptr = kvi_extractUpTo(srv->szPassword,ptr,':');
	if(*ptr)ptr++;
	if(!*ptr)return;
	ptr = kvi_extractUpTo(srv->szPort,ptr,':');
	if(*ptr)ptr++;
	if(!*ptr)return;
	ptr = kvi_extractUpTo(srv->szDescription,ptr,':');
	if(*ptr)ptr++;
	if(!*ptr)return;
#ifdef COMPILE_NEED_IPV6
	KviStr tmp;
	kvi_extractUpTo(tmp,ptr,':');
	srv->bIpV6 = kvi_strEqualCI("IPv6",tmp.ptr());
#endif
}

void KviIrcNetwork::serverToString(KviStr &szSrv,KviIrcServer *srv)
{
	szSrv = (srv->szHost.hasData() ? srv->szHost.ptr() : "irc.unknown.net");
	szSrv.append(':');
	srv->szIp.replaceAll(':',"#");
	szSrv.append(srv->szIp);
	srv->szIp.replaceAll('#',":");
	szSrv.append(':');
	szSrv.append(srv->szPassword);
	szSrv.append(':');
	szSrv.append(srv->szPort.hasData() ? srv->szPort.ptr() : "6667");
	szSrv.append(':');
	szSrv.append(srv->szDescription);
#ifdef COMPILE_NEED_IPV6
	szSrv.append(':');
	szSrv.append(srv->bIpV6 ? "IPv6" : "IPv4");
#endif
}

void KviIrcNetwork::save(KviConfig *cfg)
{
	cfg->writeEntry("Name",m_szName.ptr());
	cfg->writeEntry("Servers",m_pServerList->count());
	KviStr szSrv,szNum;
	for(uint i=0;i<m_pServerList->count();i++){
		KviIrcServer *srv = m_pServerList->at(i);
		if(srv == m_pCurrentServer)cfg->writeEntry("CurrentServer",i);
		__range_valid(srv);
		serverToString(szSrv,srv);
		szNum.sprintf("Server_%d",i);
		cfg->writeEntry(szNum.ptr(),szSrv.ptr());
	}
}

void KviIrcNetwork::load(KviConfig *cfg)
{
	m_szName = cfg->readEntry("Name","UnknownNet");
	uint count = cfg->readUIntEntry("Servers",0);
	m_pCurrentServer = 0;
	uint curr  = cfg->readUIntEntry("CurrentServer",0);
	KviStr szSrv,szNum;
	for(uint i=0;i<count;i++){
		KviIrcServer *srv = new KviIrcServer;
		__range_valid(srv);
		szNum.sprintf("Server_%d",i);
		szSrv = cfg->readEntry(szNum.ptr(),"irc.unknown.net:127.0.0.1::6667:Unknown:IPv4");
		serverFromString(szSrv,srv);
		m_pServerList->append(srv);
		if(curr == i)m_pCurrentServer = srv;
	}
	if(!m_pCurrentServer){
		if(!m_pServerList->isEmpty())m_pCurrentServer = m_pServerList->first();
	}
}

void KviIrcNetwork::copyFrom(KviIrcNetwork *net)
{
	__range_valid(net);
	clear();
	m_szName = net->m_szName.ptr();
	m_pCurrentServer = 0;
	for(KviIrcServer *s=net->m_pServerList->first();s;s=net->m_pServerList->next()){
		KviIrcServer *v=new KviIrcServer;
		v->szHost        = s->szHost.ptr();
		v->szIp          = s->szIp.ptr();
		v->szDescription = s->szDescription.ptr();
		v->szPassword    = s->szPassword.ptr();
		v->szPort        = s->szPort.ptr();
#ifdef COMPILE_NEED_IPV6
		v->bIpV6         = s->bIpV6;
#endif
		if(s == net->m_pCurrentServer)m_pCurrentServer = v;
		m_pServerList->append(v);
	}
	__range_valid((m_pCurrentServer || m_pServerList->isEmpty()) &&
					(!(m_pCurrentServer && m_pServerList->isEmpty())));
}

KviIrcServer * KviIrcNetwork::getServerByName(const char *szServer)
{
	if(m_pServerList->isEmpty())return 0;
	for(KviIrcServer *s=m_pServerList->first();s;s=m_pServerList->next()){
		if(kvi_strEqualCI(szServer,s->szHost.ptr()))return s;
	}
	return 0;
}

void KviIrcNetwork::updateServerIp(const char *szServer,const char *szIp)
{
	KviIrcServer *srv = getServerByName(szServer);
	if(!srv)return;
#ifdef COMPILE_NEED_IPV6
	if(srv->bIpV6)
	{
		if(!kvi_isValidStringIp_V6(szIp))srv->bIpV6 = false;
	}
#endif
	srv->szIp = szIp;
}

void KviIrcNetwork::sortServers()
{
	m_pServerList->setAutoDelete(false);
	QList<KviIrcServer> l(*m_pServerList); //shallow copy
	l.setAutoDelete(false);
	while(!m_pServerList->isEmpty())m_pServerList->removeLast();
	KviStr sz1,sz2;
	while(!l.isEmpty()){
		KviIrcServer *s=l.first();
		getServerCompareString(sz1,s);
		uint idx = 0;
		bool bDone = false;
		for(KviIrcServer *is=m_pServerList->first();is && (!bDone);is=m_pServerList->next()){
			getServerCompareString(sz2,is);
			if(stricmp(sz1.ptr(),sz2.ptr()) > 0){
				//The new one is smaller than the current one
				m_pServerList->insert(idx,s);
				bDone = true;
			}
			idx++;
		}
		if(!bDone)m_pServerList->append(s);
		l.removeFirst();
	}
	m_pServerList->setAutoDelete(true);
}

void KviIrcNetwork::getServerCompareString(KviStr &szStr,KviIrcServer *srv)
{
	int idx = srv->szHost.findLastIdx('.');
	if(idx >= 0){
		szStr = srv->szHost.right(srv->szHost.len() - idx);
		szStr += srv->szHost.left(idx);
	} else szStr = srv->szHost.ptr();
}

//===========================================================================//
// KviIrcServerManager class
//===========================================================================//

KviIrcServerManager::KviIrcServerManager()
{
	m_pNetList = new QList<KviIrcNetwork>;
	m_pNetList->setAutoDelete(true);
	m_pCurrentNet   = 0;
}

KviIrcServerManager::~KviIrcServerManager()
{
	clear();
	delete m_pNetList;
}

void KviIrcServerManager::clear()
{
	while(!m_pNetList->isEmpty())m_pNetList->removeFirst();
	m_pCurrentNet = 0;
}

void KviIrcServerManager::save(KviConfig *cfg)
{
	__range_valid(cfg);
	cfg->setGroup("IrcServerManager");
	cfg->writeEntry("Networks",m_pNetList->count());
	for(uint i=0;i<m_pNetList->count();i++){
		KviIrcNetwork *net = m_pNetList->at(i);
		__range_valid(net);
		KviStr szNet(KviStr::Format,"Net_%d",i);
		cfg->setGroup(szNet.ptr());
		net->save(cfg);
		if(net == m_pCurrentNet){
			cfg->setGroup("IrcServerManager");
			cfg->writeEntry("Current_Net",i);
		}
	}
	cfg->sync();
}

void KviIrcServerManager::load(KviConfig *cfg)
{
	clear();
	__range_valid(cfg);
	cfg->setGroup("IrcServerManager");
	uint count = cfg->readUIntEntry("Networks",0);
	uint curr  = cfg->readUIntEntry("Current_Net",0);
	m_pCurrentNet = 0;
	KviIrcNetwork * curNet = 0;
	for(uint i=0;i<count;i++){
		KviIrcNetwork *net = new KviIrcNetwork();
		__range_valid(net);
		KviStr szNet(KviStr::Format,"Net_%d",i);
		cfg->setGroup(szNet.ptr());
		net->load(cfg);
		insertNetwork(net);
		if(i==curr)curNet = net;
	}
	if(curNet)m_pCurrentNet = curNet;
	else {
		if(m_pNetList->isEmpty())m_pCurrentNet = 0;
		else m_pCurrentNet = m_pNetList->first();
	}
}

KviIrcNetwork * KviIrcServerManager::getCurrentNetwork()
{
	if(m_pNetList->isEmpty())return 0;
	if(m_pCurrentNet){
		__range_valid(m_pNetList->findRef(m_pCurrentNet) != -1);
		return m_pCurrentNet;
	}
	m_pCurrentNet = m_pNetList->first();
	return m_pCurrentNet;
}

QList<KviIrcNetwork> * KviIrcServerManager::networkList()
{
	return m_pNetList;
}

void KviIrcServerManager::insertNetwork(KviIrcNetwork *net)
{
	__range_valid(net);
	m_pCurrentNet = net;
	if(m_pNetList->isEmpty())m_pNetList->append(net);
	else {
		uint index = 0;
		for(KviIrcNetwork * n = m_pNetList->first();n;n=m_pNetList->next()){
			if(stricmp(net->name(),n->name()) < 0){
				m_pNetList->insert(index,net);
				return;
			}
			index++;
		}
		m_pNetList->append(net);
	}
}

KviIrcNetwork * KviIrcServerManager::getNetworkByName(const char *szName)
{
	__range_valid(szName);
	for(KviIrcNetwork * n = m_pNetList->first();n;n=m_pNetList->next()){
		if(kvi_strEqualCI(szName,n->name())){
			return n;
		}
	}
	return 0;
}

bool KviIrcServerManager::setCurrentNetwork(KviIrcNetwork * net)
{
	if(m_pNetList->isEmpty()){
		m_pCurrentNet = 0;
		return false;
	}
	if(m_pNetList->findRef(net) != -1){
		m_pCurrentNet = net;
		return true;
	}
	if(!m_pCurrentNet)m_pCurrentNet = m_pNetList->first();
	return false;
}

bool KviIrcServerManager::setCurrentNetwork(const char *szName)
{
	KviIrcNetwork *net = getNetworkByName(szName);
	if(!net)return false;
	return setCurrentNetwork(net);
}

bool KviIrcServerManager::setCurrentServer(KviIrcServer *srv)
{
	KviIrcNetwork *net = getCurrentNetwork();
	if(!net)return false;
	return net->setCurrentServer(srv);
}

void KviIrcServerManager::iWantThisServerToBeCurrent(const char *szName,const char *szPort,bool bIpV6)
{
	__range_valid(szName);
	for(KviIrcNetwork * n = m_pNetList->first();n;n=m_pNetList->next()){
		for(KviIrcServer *s = n->m_pServerList->first();s;s = n->m_pServerList->next()){
			if(kvi_strEqualCI(s->szHost.ptr(),szName)){
				if(szPort)
				{
					if(*szPort)s->szPort = szPort;
				}
				if(!s->szPort.isUnsignedNum())s->szPort = "6667";
				setCurrentNetwork(n);
				n->setCurrentServer(s);
				return;
			}
		}
	}
	KviIrcServer * srv = new KviIrcServer;
	srv->szHost = szName;
	if(szPort)
	{
		if(*szPort)srv->szPort = szPort;
	}
	if(!srv->szPort.isUnsignedNum())srv->szPort = "6667";
#ifdef COMPILE_NEED_IPV6
	srv->bIpV6 = bIpV6;
#endif
	insertNewServer(srv,"[UNKNOWNNET]");
	setCurrentNetwork("[UNKNOWNNET]");
	setCurrentServer(srv);
}

void KviIrcServerManager::setCurrentServerPassword(const char * pass)
{
	KviIrcServer * s = getCurrentServer();
	if(s)s->szPassword = pass;
}

bool KviIrcServerManager::removeNetwork(const char *szName)
{
	__range_valid(szName);
	KviIrcNetwork *net = getNetworkByName(szName);
	if(!net)return false;
	return removeNetwork(net);
}

bool KviIrcServerManager::removeNetwork(KviIrcNetwork * net)
{
	__range_valid(net);
	bool bWasHere = m_pNetList->removeRef(net);
	__range_valid(bWasHere);
	if(bWasHere){
		if(net == m_pCurrentNet){
			if(m_pNetList->isEmpty())m_pCurrentNet = 0;
			else m_pCurrentNet = m_pNetList->first();
		}
		return true;
	}
	return false;
}

void KviIrcServerManager::copyFrom(KviIrcServerManager * ptr)
{
	__range_valid(ptr);
	clear();
	KviIrcNetwork * curNet = 0;
	for(KviIrcNetwork *n=ptr->m_pNetList->first();n;n=ptr->m_pNetList->next()){
		//avoid copying empty networks
		if(n->serverCount() > 0){
			KviIrcNetwork *net = new KviIrcNetwork(n);
			if(n == ptr->m_pCurrentNet)curNet = net;
			insertNetwork(net);
		}
	}
	if(curNet)m_pCurrentNet = curNet;
	else {
		if(m_pNetList->isEmpty())m_pCurrentNet = 0;
		else m_pCurrentNet = m_pNetList->first();
	}
#ifdef _KVI_DEBUG_CHECK_RANGE_
	//sanity check
	if(m_pCurrentNet){
		if(m_pCurrentNet->m_pCurrentServer && ptr->m_pCurrentNet->m_pCurrentServer){
			__range_valid(m_pCurrentNet->m_pCurrentServer->szHost == ptr->m_pCurrentNet->m_pCurrentServer->szHost);
		} else debug(__tr("WARNING : current network has no active server"));
	}
#endif
}


KviIrcServer * KviIrcServerManager::getCurrentServer()
{
	KviIrcNetwork * net = getCurrentNetwork();
	if(!net)return 0;
	return net->currentServer();
}

KviIrcServer * KviIrcServerManager::getNextServer()
{
	KviIrcNetwork * net = getCurrentNetwork();
	if(!net)return 0;
	return net->nextServer();
}

//============================ updateServerIp ===============================//

void KviIrcServerManager::updateServerIp(const char *szNet,const char *szServer,const char *szIp)
{
	// Cache the IP's....
	KviIrcNetwork * net = getNetworkByName(szNet);
	if(!net)return;
	net->updateServerIp(szServer,szIp);
}

//============================ importFromIni ================================//

bool KviIrcServerManager::importFromIni(const char *filename)
{
	// Try to load the mIrc native format servers.ini

	//
	//[servers]
	//nXXX=DescriptionSERVER:server.name:port_listGROUP:group_name
	//....

	QFile serversIni(filename);
	if(!serversIni.exists())return false;
	// KviConfig opens in read-only mode if we not write entries or call sync()
	KviConfig cfg(filename);
	cfg.setGroup("servers");

	int curidx=0;
	KviStr szKey(KviStr::Format,"n%d",curidx);
	KviStr szNetwork,szDescription,szServer,szPorts,szPort;

	// Assume that the entries are numbered correctly...

	while(cfg.hasKey(szKey.ptr())){
		// Read the entry
		KviStr szEntry=cfg.readEntry(szKey.ptr());
		// Look for the description part
		int idx = szEntry.findFirstIdx("SERVER:");
		if(idx > -1){
			//K...have a description...
			szDescription = szEntry.left(idx);
			szEntry.cutLeft(idx + 7);
			//Now the server name
			szEntry.getToken(szServer,':');
			szServer.stripWhiteSpace();
			if(szServer.isEmpty())szServer = "irc.unknown.net";
			//Defaults
			szPorts = "6667";
			szNetwork = "UnknownNet";
			//Now look for the ports
			idx = szEntry.findFirstIdx("GROUP:");
			if(idx > -1){
				szPorts = szEntry.left(idx);
				//Only one port allowed
				szPorts.getToken(szPort,',');
				szEntry.cutLeft(idx + 6);
				//And the network name
				if(szEntry.hasData())szNetwork = szEntry;
			}
			KviIrcServer *lpE=new KviIrcServer;
			lpE->szHost=szServer;
			lpE->szDescription=szDescription;
			lpE->szIp="";
			lpE->szPassword="";
			lpE->szPort=szPort.isUnsignedNum() ? szPort.ptr() : "6667";
#ifdef COMPILE_NEED_IPV6
			lpE->bIpV6 = false;
#endif
			insertNewServer(lpE,szNetwork.ptr());
		}
		curidx++;
		szKey.sprintf("n%d",curidx);
	}
	return true;
}

//==================== insertNewServer =======================//

void KviIrcServerManager::insertNewServer(KviIrcServer *srv,const char *szNetName)
{
	// Insert a new server to the list
	// If the network szNetName already exists,
	// add the server to that network ,
	// otherwise create the net szNetName and then insert
	KviIrcNetwork *net = getNetworkByName(szNetName);
	if(!net){
		//Create a new one
		net = new KviIrcNetwork(szNetName);
		insertNetwork(net);
	}
	net->appendServer(srv);
}

//===========================================================================//
// KviIrcProxyClass class
//===========================================================================//

KviIrcProxyManager::KviIrcProxyManager()
{
	m_pProxyList = new QList<KviIrcProxy>;
	m_pProxyList->setAutoDelete(true);
	m_pCurrentProxy = 0;
}

KviIrcProxyManager::~KviIrcProxyManager()
{
	clear();
	delete m_pProxyList;
}

void KviIrcProxyManager::clear()
{
	while(!m_pProxyList->isEmpty())m_pProxyList->removeLast();
	m_pCurrentProxy = 0;
}

void KviIrcProxyManager::save(KviConfig *cfg)
{
	__range_valid(cfg);
	cfg->setGroup("IrcProxyManager");
	cfg->writeEntry("Hosts",m_pProxyList->count());
	for(uint i=0;i<m_pProxyList->count();i++){
		KviIrcProxy *prx = m_pProxyList->at(i);
		__range_valid(prx);
		KviStr szNum(KviStr::Format,"Host_%d",i);
		KviStr szPrx(KviStr::Format,"%s:%s:%s:%s:%s",prx->szHost.ptr(),
			prx->szIp.ptr(),prx->szPort.ptr(),prx->szUsername.ptr(),prx->szPassword.ptr());
		cfg->writeEntry(szNum.ptr(),szPrx.ptr());
		if(prx == m_pCurrentProxy)cfg->writeEntry("Current_Host",i);
	}
	cfg->sync();
}

void KviIrcProxyManager::load(KviConfig *cfg)
{
	clear();
	__range_valid(cfg);
	cfg->setGroup("IrcProxyManager");
	uint count = cfg->readUIntEntry("Hosts",0);
	uint curr  = cfg->readUIntEntry("Current_Host",0);
	m_pCurrentProxy = 0;
	KviIrcProxy * curPrx = 0;
	for(uint i=0;i<count;i++){
		KviIrcProxy *prx = new KviIrcProxy();
		__range_valid(prx);
		KviStr szNum(KviStr::Format,"Host_%d",i);
		KviStr szPrx=cfg->readEntry(szNum.ptr(),"proxy.localhost:127.0.0.1:1080::");

		m_pProxyList->append(prx);
		if(i==curr)curPrx = prx;

		prx->szHost = "proxy.localhost";
		prx->szIp   = "127.0.0.1";
		prx->szPort = "1080";

		const char *ptr = szPrx.ptr();
		if((!*ptr)||(*ptr==':'))continue;
		ptr = kvi_extractUpTo(prx->szHost,ptr,':');
		if(*ptr)ptr++;
		if(!*ptr)continue;
		ptr = kvi_extractUpTo(prx->szIp,ptr,':');
		if(*ptr)ptr++;
		if(!*ptr)continue;
		ptr = kvi_extractUpTo(prx->szPort,ptr,':');
		if(*ptr)ptr++;
		if(!*ptr)continue;
		ptr = kvi_extractUpTo(prx->szUsername,ptr,':');
		if(*ptr)ptr++;
		if(!*ptr)continue;
		kvi_extractUpTo(prx->szPassword,ptr,':');
	}
	if(curPrx)m_pCurrentProxy = curPrx;
	else {
		if(m_pProxyList->isEmpty())m_pCurrentProxy = 0;
		else m_pCurrentProxy = m_pProxyList->first();
	}
}

KviIrcProxy * KviIrcProxyManager::currentProxy()
{
	if(m_pCurrentProxy)return m_pCurrentProxy;
	if(!m_pProxyList->isEmpty()){
		m_pCurrentProxy = m_pProxyList->first();
		return m_pCurrentProxy;
	}
	return 0;
}

KviIrcProxy * KviIrcProxyManager::getProxyByName(const char *szName)
{
	for(KviIrcProxy * p=m_pProxyList->first();p;p=m_pProxyList->next()){
		if(kvi_strEqualCI(p->szHost.ptr(),szName))return p;
	}
	return 0;
}

bool KviIrcProxyManager::setCurrentProxy(KviIrcProxy *prx)
{
	__range_valid(m_pProxyList->findRef(prx) != -1);
	if(m_pProxyList->findRef(prx) != -1){
		m_pCurrentProxy = prx;
		return true;
	}
	return false;
}

void KviIrcProxyManager::copyFrom(KviIrcProxyManager *m)
{
	clear();
	for(KviIrcProxy *p=m->m_pProxyList->first();p;p=m->m_pProxyList->next()){
		KviIrcProxy *prx = new KviIrcProxy;
		prx->szHost     = p->szHost;
		prx->szIp       = p->szIp;
		prx->szUsername = p->szUsername;
		prx->szPassword = p->szPassword;
		prx->szPort     = p->szPort;
		m_pProxyList->append(prx);
		if(p==m->m_pCurrentProxy)m_pCurrentProxy = prx;
	}
}

void KviIrcProxyManager::appendProxy(KviIrcProxy * ptr)
{
	__range_valid(ptr);
	m_pProxyList->append(ptr);
	m_pCurrentProxy = ptr;
}

bool KviIrcProxyManager::removeProxy(KviIrcProxy * ptr)
{
	__range_valid(ptr);
	bool bWasHere = m_pProxyList->removeRef(ptr);
	__range_valid(bWasHere);
	if(ptr == m_pCurrentProxy){
		if(m_pProxyList->isEmpty())m_pCurrentProxy = 0;
		else m_pCurrentProxy = m_pProxyList->first();
	}
	return bWasHere;
}

void KviIrcProxyManager::updateProxyIp(const char *szName,const char *szIp)
{
	KviIrcProxy *prx = getProxyByName(szName);
	if(!prx)return;
	prx->szIp = szIp;
}
