#ifndef _KVI_STRING_H_INCLUDED_
#define _KVI_STRING_H_INCLUDED_

//
//   File : kvi_string.h (/usr/build/NEW_kvirc/kvirc/src/kvicore/kvi_string.h)
//   Last major modification : Fri Mar 19 1999 03:06:26 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//
//   KviStr class implementation and string utility functions
//

#include "kvi_settings.h"

#include <stdio.h>
#include <stdarg.h> //for va_list

#include <string.h>
#include <ctype.h>

#ifdef HAVE_STRINGS_H
	#include <strings.h>
#endif

#include <qglobal.h>
#include <qstring.h>

//
//  sigh...
//  IRC is not UNICODE ...(yet) :(
//


#ifdef COMPILE_USE_LOCAL_8BIT_ENCODING
	#define _CHAR_2_QSTRING(__chptr) QString::fromLocal8Bit(__chptr)
#else
	#define _CHAR_2_QSTRING(__chptr) QString(__chptr)
#endif
//
//#ifdef COMPILE_USE_LOCAL_8BIT_ENCODING
//	char * kvi_qstring2char(QString &str)
//	{
//		QCString str = str.local8Bit();
//		int l = str.length();
//		static_buffer = (char *)kvi_realloc(static_buffer,l + 1);
//		kvi_fastmove(static_buffer,str.data(),l);
//		static_buffer[l - 1] = '\0';
//		return static_buffer;
//	}
//#endif

#undef __kvi_extern

#ifdef _KVI_STRING_CPP_
	#define __kvi_extern
#else
	#define __kvi_extern extern
#endif

/**
* A simple string class.<br>
* -No data sharing.<br>
* -Not UNICODE.<br>
* -Has ALWAYS NON-NULL DATA.<br>
* -(Maybe)Unsafe :)<br>
* WARNING : Handle with care and use at own risk :)<br>
*
* @short A simple string class implementation
* @author Szymon Stefanek
*/

class KviStr
{
public:
	// No particular reason for these two names...
	// It is just because I like it :)
	enum KviFormatConstructorTag { Format , Sprintf }; 
	/**
	* Empty string == ""<br>
	* len = 0;<br>
	* 1 byte allocated
	*/
	KviStr();
	/**
	* Deep copy of the string<br>
	* If str is 0 an empty (ptr == "" ,len = 0) string is created.<br>
	* If str is non zero a deep copy is created. (len = strlen(str))<br>
	* Note that str MUST be null terminated in this case.<br>
	* len+1 bytes allocated
	*/
	KviStr(const char *str);
	/**
	* Deep copy of len bytes of str.<br>
	* Note that str can NOT be 0 here ,
	* and must be also at least len characters long.<br>
	* len+1 bytes allocated
	*/
	KviStr(const char *str,int len);
	/**
	* Special constructor.<br>
	* bg and end are pointers to a SINGLE string.<br>
	* A string is extracted starting from bg and ending at end (not included).<br>
	* Example:<br>
	* str=" this is a sample string !";<br>
	* bg = str+11; end = str+24;<br>
	* KviStr s(bg,end);<br>
	* now s=="sample string"<br>
	* len is calculated as (int)(end-bg)<br>
	* len+1 bytes allocated
	*/
	KviStr(const char *bg,const char *end);
	/**
	* Format constructor.<br>
	* tag is....yes....a dummy number used to resolve ambiguities.<br>
	* It is SAFE: will fail only if we run out of memory,<br>
	* but can handle only %s %d %u and %c.
	*/
	KviStr(KviFormatConstructorTag tag,const char *fmt,...); // format constructor. dummy is...yes..any number.:)
	/**
	* Carbon copy :)
	*/
	KviStr(const KviStr &str);
	/**
	* Compat with Qt...<br>
	* WARNING : With Qt2.0 it WILL loose UNICODE data.<br>
	* Safe even if the QString is null.
	*/
	KviStr(const QString &str);
	/**
	* Fill sonstructor.<br>
	* Creates a string long fillLen characters filled with character c.<br>
	* Avoid passing '\0' as 'c' :)
	*/
	KviStr(char c,int fillLen = 1);        // fillLen     // m_len + 1   // fillLen must be >= 0!
	/**
	* just free(m_ptr); :)
	*/
	~KviStr();
public:
	//yes...public..but think it as private...:)
	char *m_ptr;                           // pointer to allocated buffer , do not change this!
	int   m_len;                           // string data length not including the terminator
public:
	/**
	* Free access to data....<br>
	* (inline)
	*/
	char * ptr() const;
	/**
	* Direct len....not uses strlen.<br>
	* Simply fast. :)<br>
	* (inline)
	*/
	int len() const;
public:
	/**
	* Simple (len == 0)<br>
	* (inline)
	*/
	bool isEmpty() const;                  // :) (m_len == 0) why not ?
	/**
	* Simple (len != 0)<br>
	* Why not?<br>
	* (inline)
	*/
	bool hasData() const;                  // :) !isEmpty.... why not ?

	bool hasNonWhiteSpaceData() const;

	/**
	* Safe sprintf.<br>
	* This one will never write past the end of the string.<br>
	* It can handle only %s %d %u and %c format flags.<br>
	*/
	KviStr & sprintf(const char *fmt,...);
	/**
	* Appends the string str to this string.
	*/
	void append(const KviStr &str);        // just a memmove and realloc
	/**
	* Appends the string str to this string.
	*/
	void append(const QString &str);
	/**
	* Appends the char c to this string.
	*/
	void append(char c);
	/**
	* Appends the string str to this string unless str is 0.
	*/
	void append(const char *str);          // str can be NULL here.
	/**
	* Appends exactly len characters from str to this string.<br>
	* str can NOT be 0 , and MUST be at least len characters long.
	*/
	void append(const char *str,int len);  // str can NOT be NULL , len must be <= strlen(str) and >= 0
	void append(KviFormatConstructorTag dummy,const char *fmt,...);
	/**
	* Prepends the string str to this string.
	*/
	void prepend(const KviStr &str);       // two memmoves and a realloc
	/**
	* Prepends the string str to this string unless str is 0.
	*/
	void prepend(const char *str);         // str can be NULL here.
	/**
	* Prepends exactly len characters from str to this string.<br>
	* str can NOT be 0 , and MUST be at least len characters long.
	*/
	void prepend(const char *str,int len); // str can NOT be NULL , len must be <= strlen(str) and >= 0
	/**
	* Changes the characters in this string to uppercase.
	*/
	void toUpper();                        // moves this string to upper
	/**
	* Changes the characters in this string to lowercase.
	*/
	void toLower();                        // moves this string to lower
	/**
	* Returns an uppercase copy.<br>
	* 'this' string is not changed.
	*/
	KviStr upper() const;                  // copy and then to upper
	/**
	* Returns a lowercase copy.<br>
	* 'this' string is not changed.
	*/
	KviStr lower() const;                  // copy and then to lower
	/**
	* Returns true if the last characted of the string is ch
	*/
	bool lastCharIs(char ch) const;
	bool firstCharIs(char ch) const;

	void bufferToHex(char *buffer,int len);
	int hexToBuffer(char ** buffer,bool bNullToNewlines = false);

	void ensureLastCharIs(char ch); // if lastCharIs ch does nothing otherwise appends it

	KviStr left(int maxLen);               // safe , but maxLen must be >= 0!
	KviStr right(int maxLen);              // safe , but maxLen must be >= 0!
	KviStr middle(int idx,int maxLen);

	KviStr & cutLeft(int len);             // kills the first len characters
	KviStr & cutRight(int len);            // kills the last len characters
	KviStr & cut(int idx,int len);
	KviStr & cutToFirst(char c,bool bIncluded = true); // cuts the left part of the string up to the first character c or does nothing if the char c is not in the string
	KviStr & cutToLast(char c,bool bIncluded = true);
	KviStr & cutFromFirst(char c,bool bIncluded = true);

	KviStr & insert(int idx,const char *data);
	KviStr & insert(int idx,char c);

	KviStr & replaceAll(char c,const char *str); //Replaces all occurences of char c with the string str
	KviStr & replaceAll(char *toFind,const char *str,bool bCaseS = true); //same as above but with a string

	KviStr & setLen(int len);              // Null terminator NOT included!!!

	KviStr & setStr(const char *str,int len = -1); //safe...len can be anything....

	char & at(int idx) const;
	/**
	* Strips whitespace characters from beginning of this string.
	*/
	KviStr & stripLeftWhiteSpace();
	/**
	* Stips inital and final WHITESPACE characters (see man isspace),<br>
	* and returns a reference to this string.
	*/
	KviStr & stripWhiteSpace();
	/**
	* Strips spaces and tabs only
	*/
	KviStr & stripSpace();
	/**
	* Strips all occurences of the character c from the beginning of the string.<br>
	* Note that c can not be '\0' :)
	*/
	KviStr & stripLeft(char c);
	/**
	* Extracts (copy to str and remove) a token from this string ,<br>
	* and returns true if there are more tokens to extract<br>
	* Does not strip initial separators!!<br>
	* str can NOT be this string.
	*/
	bool getToken(KviStr &str,char sep);
	/**
	* Does not strip initial separators!<br>
	* Can assign also to this string.
	*/
	KviStr getToken(char sep);             // getToken(' ') => "string    with tokens" -> "string" + "with tokens"
	/**
	* Extracts a line from the string.<br>
	* Returns false if there was no data to extract
	*/
	bool getLine(KviStr &str);
	/**
	* Retuns true if the string contains only digits and an optional '-' character
	* at the beginning.<be>
	* Space characters are allowed at the begginning and the end.<br>
	* There is no overflow check!
	*/
	bool isNum();
	/**
	* Retuns true if the string contains only digits.<br>
	* Space characters are allowed at the beginning and the end.<br>
	* There is no overflow check!
	*/
	bool isUnsignedNum();
//	/**
//	* Transform a pointer to a string with all 0 and 1
//	*/
//	void pointerToBitString(const void * ptr);
//	/**
//	* Get a pointer from a string all of 0 and 1 : return 0 if invalid
//	*/
//	void * bitStringToPointer();

	/**
	* Returns the number of occurences of string str in this string.<br>
	* Overlapped matches are counted.<br><br>
	* KviStr str("1231231");<br>
	* str.contains("1231") == 2
	*/
	int contains(const char *str,bool caseS=true);
	/**
	* Returns the number of occurences of character c in this string
	*/
	int contains(char c,bool caseS=true);
	/**
	* Finds the first occurence of the character c in this string,<br>
	* and returns its zero-based index or -1 if c can not be found.<br>
	* c can NOT be '\0' here.
	*/
	int findFirstIdx(char c);
	/**
	* Finds the first occurence of the sub-string str in this string,<br>
	* and returns its zero-based index or -1 if the sub-string can not be found.<br>
	* str can NOT be 0 here.
	*/
	int findFirstIdx(const char *str,bool caseS = true);
	/**
	* Finds the last occurence of the character c in this string,<br>
	* and returns its zero-based index or -1 if the character can not be found.
	*/
	int findLastIdx(char c);
	/**
	* Finds the last occurence of the sub-string str in this string,<br>
	* and returns its zero-based index or -1 if the sub-string can not be found.<br>
	* str can NOT be 0 here.
	*/
	int findLastIdx(const char *str,bool caseS = true);
	/**
	* Like the special constructor that gets the same args.
	*/
	void extractFromString(const char *begin,const char *end);

	// everything in base 10
	char toChar(bool *bOk=0) const;          // no overflow checks here
	unsigned char toUChar(bool *bOk=0) const;//
	int toInt(bool *bOk=0) const;            //
	unsigned int toUInt(bool *bOk=0) const;  //
	short toShort(bool *bOk=0) const;        //
	unsigned short toUShort(bool *bOk=0) const;//
	long toLong(bool *bOk=0) const;          //
	unsigned long toULong(bool *bOk=0) const;//

	KviStr & setNum(long num);             //
	KviStr & setNum(unsigned long num);    //
	KviStr & setNum(int num);              //
	KviStr & setNum(unsigned int num);     //
	KviStr & setNum(short num);            //
	KviStr & setNum(unsigned short num);   //
	KviStr & setNum(char num);             //
	KviStr & setNum(unsigned char num);    //

	// special functions for multiple bases
	long toLongExt(bool *bOk = 0,int base = 0);
	// unsigned long toULongExt(bool *bOk = 0,int base = 0); //never used ?

	KviStr & operator=(const KviStr &str);       // deep copy
	KviStr & operator=(const char *str);   // str can be NULL here
	KviStr & operator=(char c);            // 2 bytes allocated ,m_len = 1
	KviStr & operator=(const QString &str);
	KviStr & operator+=(const KviStr &str);      // see related append()
	KviStr & operator+=(const char *str);  // see related append()
	KviStr & operator+=(char c);           // see related append()
	KviStr & operator+=(const QString &str);
};

//===========================================================================//
// inline stuff...
//

inline char * KviStr::ptr() const                       { return m_ptr; }
inline int KviStr::len() const                          { return m_len; }
inline char & KviStr::at(int idx) const                 { return ((m_len > idx) ? m_ptr[idx] : m_ptr[0]); };

inline bool KviStr::isEmpty() const                     { return (m_len == 0); }
inline bool KviStr::hasData() const                     { return (m_len != 0); }

inline bool KviStr::firstCharIs(char c) const           { return (*m_ptr == c); }
inline bool KviStr::lastCharIs(char c) const            { return (m_len > 0) ? (*(m_ptr + m_len - 1) == c) : false; }
inline void KviStr::ensureLastCharIs(char c)            { if(!lastCharIs(c))append(c); }

inline KviStr & KviStr::operator+=(const KviStr &str)   { append(str); return (*this); }
inline KviStr & KviStr::operator+=(const char *str)     { append(str); return (*this); }
inline KviStr & KviStr::operator+=(const QString &str)  { append(str); return (*this); }
inline KviStr & KviStr::operator+=(char c)              { append(c);   return (*this); }

inline KviStr & KviStr::setNum(char num)                { return setNum((long)num); }
inline KviStr & KviStr::setNum(unsigned char num)       { return setNum((unsigned long)num); }
inline KviStr & KviStr::setNum(short num)               { return setNum((long)num); }
inline KviStr & KviStr::setNum(unsigned short num)      { return setNum((unsigned long)num); }
inline KviStr & KviStr::setNum(int num)                 { return setNum((long)num); }
inline KviStr & KviStr::setNum(unsigned int num)        { return setNum((unsigned long)num); }

inline char KviStr::toChar(bool *bOk) const             { return (char)toLong(bOk); }
inline unsigned char KviStr::toUChar(bool *bOk) const   { return (unsigned char)toULong(bOk); }
inline int KviStr::toInt(bool *bOk) const               { return (int)toLong(bOk); }
inline unsigned int KviStr::toUInt(bool *bOk) const     { return (unsigned int)toULong(bOk); }
inline short KviStr::toShort(bool *bOk) const           { return (short)toLong(bOk); }
inline unsigned short KviStr::toUShort(bool *bOk) const { return (unsigned short)toULong(bOk); }

/**
* Cool string parsing function.<br>
* It will extract the first found token from the string aux_ptr , and return<br>
* a pointer to the beginning of the next token , or end of the string.<br>
* It skips the initial sep characters!<br>
* Example of use:<br>
* <br>
* KviStr str("String full of tokens");<br>
* char * ptr=str.data();<br>
* KviStr word;<br>
* while(*ptr){<br>
* &nbsp;&nbsp;&nbsp;&nbsp;ptr = kvi_extractToken(word,ptr);<br>
* &nbsp;&nbsp;&nbsp;&nbsp;messWithTheToken(word);<br>
* }<br>
* <br>
* Another example...
* KviStr word;
* char *ptr_to_World = kvi_extraxtToken(word,"Hello World");
* //Now word is equal to "Hello"
*/
__kvi_extern const char * kvi_extractToken(KviStr &str,const char *aux_ptr,char sep =' ');
/**
* Does not skip the beginning separators!<br>
* Extracts data from the string up to the next separator character or the end of the string.
* and returns a pointer to that separator (or string end).
*/
__kvi_extern const char * kvi_extractUpTo(KviStr &str,const char *aux_ptr,char sep=' ');
/**
* Reduced vsnprintf...
* Handles %s,%c,%d,%u  (%% are TWO percents here and not one.)
* Returns -1 if the formatted string exceeded the buffer length.
* Otherwise returns the length of the formatted buffer...(not including '\0')
*/
__kvi_extern int kvi_vsnprintf(char *buffer,int len,const char *fmt,va_list list);
/**
* Reduced vsnprintf: special version for irc.
* Handles %s,%c,%d,%u  (%% are TWO percents here and not one.)
* Writes up to 510 characters and terminates the string with a CRLF
* Sets bTruncated if the requested format string was too large to fit in 512 bytes
* otherwise sets it to false; The buffer MUST be at least 512 bytes long.
* Always returns the length of the formatted buffer...(max 512 - min 2=CRLF)
*/
__kvi_extern int kvi_irc_vsnprintf(char *buffer,const char *fmt,va_list list,bool *bTruncated);


#ifdef COMPILE_i386_ASM_CODE
	#include "kvi_strasm.h" // Include inlined assembly implementations
#else
	/**
	* Returns true if the string str1 is equal to str2.
	* case sensitive.
	*/
	__kvi_extern bool kvi_strEqualCS(const char *str1,const char *str2);
	/**
	* Returns true if the forst len characters of string str1 are equal to str2.
	* case sensitive.
	* Note that if str1 or str2 are shorter than len characters then are considered as NOT equal!
	*/
	__kvi_extern bool kvi_strEqualCSN(const char *str1,const char *str2,int len);

	#define kvi_strEqualNoLocaleCI(str1,str2) kvi_strEqualCI(str1,str2)
	#define kvi_strEqualNoLocaleCIN(str1,str2,len) kvi_strEqualCIN(str1,str2,len)
	#define kvi_strLen(str) strlen(str)
#endif

/**
* Returns true if the string str1 is equal to str2.
* case insensitive.
*/
__kvi_extern bool kvi_strEqualCI(const char *str1,const char *str2);
/**
* Returns true if the forst len characters of string str1 are equal to str2.
* case insensitive.
* Note that if str1 or str2 are shorter than len characters then are considered as NOT equal!
*/
__kvi_extern bool kvi_strEqualCIN(const char *str1,const char *str2,int len);
/**
* My own implementations of strcmp and strncasecmp
* Once I wrote it , I KNOW what they do : ALWAYS :)
* Note that greater here means that comes AFTER in the alphabetic order.
*/
__kvi_extern int kvi_strcmpCI(const char *str1,const char *str2);
__kvi_extern int kvi_strcmpCS(const char *str1,const char *str2);
/**
* Returns true if the two regular expressions with wildcards matches
*/
__kvi_extern bool kvi_matchWildExpr(register const char *m1,register const char *m2);
/**
* Returns true if the two regular expressions with wildcards matches, case sensitive
*/
__kvi_extern bool kvi_matchWildExprCS(register const char *m1,register const char *m2);
/**
* This function work like a particular case of strncmp.
* It evaluates if str2 is the terminal part of str1.
* example: if str1 is "this is an experiment" and str2 is "xperiment"
* return 0.
* With the index parameter, the match start on str1 from the specified
* index. For example:
* if str1 is "this is an experiment" and str2 is "an" we have return !0
* but "this is an experiment"
*      012345678901234567890
*               ^
* if we call kvi_strsubRevCS("this is an experiment","an", 9)
* we got a match.
*/
__kvi_extern int kvi_strMatchRevCS(const char *str1, const char *str2, int index=-1);
//
// KviStr comparison non-member operators
//

__kvi_extern inline bool operator==(const KviStr &left,const KviStr &right)
{ return kvi_strEqualCS(left.m_ptr,right.m_ptr); }
__kvi_extern inline bool operator==(const KviStr &left,const char *right)
{ return kvi_strEqualCS(left.m_ptr,right); }
__kvi_extern inline bool operator==(const char *left,const KviStr &right)
{ return kvi_strEqualCS(left,right.m_ptr); }
__kvi_extern inline bool operator!=(const KviStr &left,const KviStr &right)
{ return !kvi_strEqualCS(left.m_ptr,right.m_ptr); }
__kvi_extern inline bool operator!=(const KviStr &left,const char *right)
{ return !kvi_strEqualCS(left.m_ptr,right); }
__kvi_extern inline bool operator!=(const char *left,const KviStr &right)
{ return !kvi_strEqualCS(left,right.m_ptr); }

//
// The stuff below is here just for fun
// I probably newer used it in kvirc...
//

__kvi_extern inline KviStr operator+(const KviStr &left,const KviStr &right)
{ KviStr ret(left); ret += right; return ret; }
__kvi_extern inline KviStr operator+(const KviStr &left,const char *right)
{ KviStr ret(left); ret += right; return ret; }
__kvi_extern inline KviStr operator+(const char *left,const KviStr &right)
{ KviStr ret(left); ret += right; return ret; }
__kvi_extern inline KviStr operator+(const KviStr &left,char right)
{ KviStr ret(left); ret += right; return ret; }
__kvi_extern inline KviStr operator+(char left,const KviStr &right)
{ KviStr ret(left); ret += right; return ret; }

#endif //_KVI_STRING_H_INCLUDED_
