/*
	mactioneditsdialog.cpp - An action editor
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "configuration.h"
#include "mactioneditdialog.h"
#include "miscutils.h"
#include "mtip.h"

#include <qcheckbox.h>
#include <qcombobox.h>
#include <qhbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qvbox.h>
#include <qvgroupbox.h>
#include <qwhatsthis.h>

#include <klineedit.h>
#include <klocale.h>
#include <knuminput.h>
#include <kpushbutton.h>

MActionEditDialog::MActionEditDialog(QWidget *parent, Action::Type action)
	: KDialogBase(
		parent,
		"MActionEditDialog",
		true,
		"",
		Default | Ok | Cancel,
		Ok, // default button
		true
	),
	_action(action),
	_userCommand("")
{
	QVBox *main = new QVBox(this);
	setMainWidget(main);

	// title
	l_title = new QLabel(main);
	QFont f = QFont(l_title->font());
	f.setPointSize(16);
	l_title->setFont(f);
	l_title->setMargin(10);

	QHBox *gb_south = new QHBox(main);
	gb_south->setSpacing(5);

	QVBox *gb_widgets = new QVBox(gb_south);

	QVGroupBox *gb_method = new QVGroupBox(i18n("Method"), gb_widgets);

	// method label
	QLabel *l_method = new QLabel(i18n("Select a method:"), gb_method);

	// method combo box
	_methods = new QComboBox(gb_method, "QComboBox::_methods");
	_methods->setFocusPolicy(StrongFocus);
	_methods->insertItem(i18n("KDE (default)"));
	l_method->setBuddy(_methods);
	connect(_methods, SIGNAL(activated(int)), SLOT(slotMethodChange(int)));

	// command label
	l_command = new QLabel(i18n("Enter a custom command:"), gb_method);

	// command edit line
	in_command = new KLineEdit(gb_method, "KLineEdit::in_command");
	l_command->setBuddy(in_command);

	QVGroupBox *gb_commandBeforeAction = new QVGroupBox(i18n("Command before action"), gb_widgets);
	c_runCommandBeforeAction = new QCheckBox(i18n("Run command"), gb_commandBeforeAction);
	i_commandBeforeAction = new KLineEdit(gb_commandBeforeAction, "KLineEdit::i_commandBeforeAction");
	i_commandBeforeActionPause = new KIntNumInput(gb_commandBeforeAction, "KIntNumInput::i_commandBeforeActionPause");
	i_commandBeforeActionPause->setLabel(i18n("Pause after run command:"));
	i_commandBeforeActionPause->setRange(0, 300, 1, true);
	i_commandBeforeActionPause->setSuffix(" " + i18n("second(s)"));
	b_testCommandBeforeAction = new KPushButton(i18n("Test"), gb_commandBeforeAction, "KPushButton::b_testCommandBeforeAction");
	connect(b_testCommandBeforeAction, SIGNAL(clicked()), SLOT(slotTestCommandBeforeAction()));
	connect(
		c_runCommandBeforeAction, SIGNAL(toggled(bool)),
		i_commandBeforeAction, SLOT(setEnabled(bool)));
	connect(
		c_runCommandBeforeAction, SIGNAL(toggled(bool)),
		i_commandBeforeActionPause, SLOT(setEnabled(bool)));
	connect(
		c_runCommandBeforeAction, SIGNAL(toggled(bool)),
		b_testCommandBeforeAction, SLOT(setEnabled(bool)));

	MTip *t_progs = new MTip(MTip::Warning, gb_south);
	t_progs->setTipText(
		"<qt>" +
		i18n("In most cases you need privileges to shut down system (e.g. run /sbin/shutdown)") + "<br>" \
		"<ul>" \
		"<li>" + i18n("If you are using <b>KDE</b> and <b>KDM</b> (KDE Display Manager), then set all methods to <i>KDE</i>") + "</li>" \
		"<li>" + i18n("If you are using <b>KDE</b> and display manager different than <b>KDM</b>, then set <i>Turn Off Computer</i> and <i>Restart Computer</i> methods to <i>/sbin/...</i>") + "</li>" \
		"</ul>" +
		i18n("Manuals:") +
		"<ul>" \
		"<li><a href=\"man:/halt\">halt, poweroff, reboot</a></li>" \
		"<li><a href=\"man:/shutdown\">shutdown</a></li>" \
		"<li><a href=\"man:/xlock\">xlock</a></li>" \
		"<li><a href=\"man:/xscreensaver-command\">xscreensaver-command</a></li>" \
		"</ul>" \
		"</qt>"
	);

	connect(this, SIGNAL(okClicked()), SLOT(slotOKClicked()));

	Action::Method m = Action::Method_KDE;
	QString c;
	QString group = ks_actions->actionToConfigGroup(_action);
	_methods->insertItem(ks_actions->getMethod(_action, m, c));

	QString s = ks_actions->getName(_action);
	l_title->setText(s);
	setCaption(s);

	_methods->insertItem(i18n("User Command"));

	// method
	_userCommand = c;
	setMethod(m);

	// command before action
	KConfig *conf = kshutdownrc->config();
	conf->setGroup(group);
	c_runCommandBeforeAction->setChecked(conf->readBoolEntry("RunCommandBeforeAction", false));
	i_commandBeforeAction->setText(conf->readEntry("CommandBeforeAction", ""));
	i_commandBeforeActionPause->setValue(conf->readNumEntry("CommandBeforeActionPause", 10));

	bool runCommand = c_runCommandBeforeAction->isChecked();
	i_commandBeforeAction->setEnabled(runCommand);
	i_commandBeforeActionPause->setEnabled(runCommand);
	b_testCommandBeforeAction->setEnabled(runCommand);
	_methods->setFocus();

	MiscUtils::pack(this);
}

MActionEditDialog::~MActionEditDialog()
{
}

void MActionEditDialog::setMethod(const Action::Method method) const
{
	_methods->setCurrentItem(method);
	l_command->setEnabled(method == Action::Method_UserCommand);
	in_command->setEnabled(method == Action::Method_UserCommand);
	switch (method)
	{
		case Action::Method_KDE:
			in_command->setText("");
			break;
		case Action::Method_DefaultCommand:
			in_command->setText(_methods->text(Action::Method_DefaultCommand));
			break;
		case Action::Method_UserCommand:
			in_command->setText(_userCommand);
			break;
	}
}

void MActionEditDialog::slotDefault()
{
	setMethod(Action::Method_KDE);
	c_runCommandBeforeAction->setChecked(false);
	i_commandBeforeAction->clear();
	i_commandBeforeActionPause->setValue(10);
}

void MActionEditDialog::slotMethodChange(int index)
{
	if (in_command->isEnabled())
		_userCommand = in_command->text();
	setMethod((Action::Method)index);
}

void MActionEditDialog::slotOKClicked()
{
	Action::Method m;
	QString c;
	QString group = ks_actions->actionToConfigGroup(_action);

	// method
	m = (Action::Method)_methods->currentItem();
	if (m == Action::Method_UserCommand)
		c = in_command->text();
	else
		c = "";
	ks_actions->setMethod(group, m, c);

	// command before action
	KConfig *conf = kshutdownrc->config();
	conf->setGroup(group);
	conf->writeEntry("RunCommandBeforeAction", c_runCommandBeforeAction->isChecked());
	conf->writeEntry("CommandBeforeAction", i_commandBeforeAction->text());
	conf->writeEntry("CommandBeforeActionPause", i_commandBeforeActionPause->value());
}

void MActionEditDialog::slotTestCommandBeforeAction()
{
	MiscUtils::runShellCommand(
		i_commandBeforeAction->text(),
		KProcess::DontCare,
		i_commandBeforeActionPause->value()
	);
}
