/***************************************************************************
 *   Copyright (C) 2005 by Danny Kukawka                                   *
 *                         <dkukawka@suse.de>, <danny.kukawka@web.de>      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

/*! \file settings.cpp
 * \brief 	In this file can be found the settings ( read ) related code. 
 * \author 	Danny Kukawka, <dkukawka@suse.de>, <danny.kukawka@web.de>
 * \date    	2005
 */

// KDE Header
#include <klocale.h>

// QT Header 


#include "settings.h"

/*! This is the default constructor of the class settings. */
settings::settings()
{
	kconfig = new KConfig("kpowersaverc", true );
	kde_settings = new KDE_Settings();
	load_kde_settings();
	load_general_settings();
	
	// dummys for settings.cpp
	QString dummy = "";
	dummy = i18n("Could not load the global configuration.");
	dummy = i18n("Could not load the requested scheme configuration.");
	dummy = i18n("Configure the current scheme.");
	dummy = i18n("Try loading the default configuration.");
	dummy = i18n("Maybe the global configuration file is empty or missing.");
}

/*! This is the default destructor of the class settings. */
settings::~settings()
{
}


/*!
 * Loads the scheme settings from kpowersaverc and fills the related variables.
 * \param schemeName QString with the name (realname not i18n() version) of the 
 *		     scheme which setting should be load. If the scheme could not
 *		     be loaded, this function try to load "default-scheme"
 * \return the result of the load
 * \retval true  if the settings could be loaded
 * \retval false if there was no group named like schemeName or named "default-scheme"
 */
bool settings::load_scheme_settings(QString schemeName){
	
	kconfig->reparseConfiguration();
	bool setToDefault = false;

	if( schemeName == "Performance" || schemeName == i18n("Performance"))
		 schemeName = "Performance";
	else if( schemeName == "Powersave" || schemeName == i18n("Powersave"))
		 schemeName = "Powersave";
	else if( schemeName == "Presentation" || schemeName == i18n("Presentation"))
		 schemeName = "Presentation";
	else if( schemeName == "Acoustic" || schemeName == i18n("Acoustic"))
		 schemeName = "Acoustic";
	
	if(kconfig->hasGroup(schemeName) || kconfig->hasGroup("default-scheme") ){
		if(kconfig->hasGroup(schemeName)) kconfig->setGroup(schemeName);
		else {
			// fallback to 'default-scheme'
			kconfig->setGroup("default-scheme");
			schemeName = "default-scheme";
			setToDefault = true;
		}
		currentScheme = schemeName;
		
		specSsSettings = kconfig->readBoolEntry("specSsSettings",false);
		disableSs = kconfig->readBoolEntry("disableSs",false);
		blankSs = kconfig->readBoolEntry("blankSs",false);
		specPMSettings = kconfig->readBoolEntry("specPMSettings",false);
		disableDPMS = kconfig->readBoolEntry("disableDPMS",false);
		
		int i_standby = kconfig->readNumEntry("standbyAfter", -1);
		if (i_standby >= 0) standbyAfter = i_standby;
		else {
			kconfig->setGroup("default-scheme");
			i_standby = kconfig->readNumEntry("standbyAfter", -1);
			if(i_standby >= 0) {
				standbyAfter = i_standby;
			}
			else standbyAfter = 0;
			// reset the group
			kconfig->setGroup(schemeName);
		}
		
		int i_suspend = kconfig->readNumEntry("suspendAfter", -1);
		if (i_suspend >= 0) suspendAfter = i_suspend;
		else {
			kconfig->setGroup("default-scheme");
			i_suspend = kconfig->readNumEntry("suspendAfter", -1);
			if(i_suspend >= 0) {
				suspendAfter = i_suspend;
			}
			else suspendAfter = 0;
			// reset the group
			kconfig->setGroup(schemeName);
		}
		
		int i_poweroff = kconfig->readNumEntry("powerOffAfter", -1);
		if (i_poweroff >= 0) powerOffAfter = i_poweroff;
		else {
			kconfig->setGroup("default-scheme");
			i_poweroff = kconfig->readNumEntry("powerOffAfter", -1);
			if(i_poweroff >= 0) {
				powerOffAfter = i_poweroff;
			}
			else powerOffAfter = 0;
			// reset the group
			kconfig->setGroup(schemeName);
		}
		
		brightness = kconfig->readBoolEntry("enableBrightness",false);
		brightnessValue = kconfig->readNumEntry("brightnessPercent", -1);
		if (brightnessValue == -1) {
			kconfig->setGroup("default-scheme");
			brightnessValue = kconfig->readNumEntry("brightnessPercent", 100);
			// reset the group
			kconfig->setGroup(schemeName);
		}

		int i_autoInactiveActionAfter = kconfig->readNumEntry("autoInactiveActionAfter", -1);
		if (i_autoInactiveActionAfter >= 0) autoInactiveActionAfter = i_autoInactiveActionAfter;
		else {
			kconfig->setGroup("default-scheme");
			i_autoInactiveActionAfter = kconfig->readNumEntry("autoInactiveActionAfter", -1);
			if(i_autoInactiveActionAfter >= 0) {
				autoInactiveActionAfter = i_autoInactiveActionAfter;
			}
			else autoInactiveActionAfter = 0;
			// reset the group
			kconfig->setGroup(schemeName);
		}
		
		QString _autoInactiveAction = kconfig->readEntry("autoInactiveAction", "NULL");
		if( _autoInactiveAction != "NULL") {
			autoInactiveAction = _autoInactiveAction;
		}
		else {
			kconfig->setGroup("default-scheme");
			_autoInactiveAction = kconfig->readEntry("autoInactiveAction", "NULL");
			if(_autoInactiveAction != "NULL") autoInactiveAction = _autoInactiveAction;
			else autoInactiveAction = "_NONE_";
			// reset the group
			kconfig->setGroup(schemeName);
		}
		
		autoSuspend = kconfig->readBoolEntry("autoSuspend",false);
		autoInactiveSBlistEnabled = kconfig->readBoolEntry("autoInactiveSchemeBlacklistEnabled",false);
		autoInactiveSBlist = kconfig->readListEntry("autoInactiveSchemeBlacklist", ',');
		
		disableNotifications = kconfig->readBoolEntry("disableNotifications",false);
		
		return true;
	}
	else return false;
}


/*!
 * Loads the general settings from kpowersaverc and fills the related variables.
 * \return the result of the load
 * \retval true  if the settings could be loaded
 * \retval false if there was no group named 'General'
 */
bool settings::load_general_settings(){
	
	kconfig->reparseConfiguration();

	if(kconfig->hasGroup("General")) {
		kconfig->setGroup("General");
		
		lockOnSuspend = kconfig->readBoolEntry("lockOnSuspend",false);
		lockOnLidClose = kconfig->readBoolEntry("lockOnLidClose",false);
		autostart = kconfig->readBoolEntry("Autostart",false);
		autostartNeverAsk = kconfig->readBoolEntry("AutostartNeverAsk",false);
		psMsgAsPassivePopup = kconfig->readBoolEntry("psMsgAsPassivePopup",false);
		forceDpmsOffOnLidClose  = kconfig->readBoolEntry("forceDpmsOffOnLidClose",false);

		lockmethod = kconfig->readEntry("lockMethod", "NULL");
		if(lockmethod == "NULL") lockmethod = "automatic";
	
		autoInactiveGBlist = kconfig->readListEntry("autoInactiveBlacklist", ',');
		
		timeToFakeKeyAfterLock = kconfig->readNumEntry("timeToFakeKeyAfterLock", 5000);

		return true;
	}
	else return false;
}

/*!
 * Loads the default KDE Settings from the different configfiles and store
 * them to a \ref KDE_Settings 'object'.
 * \retval true  if the settings could be loaded
 * \retval false if there was a error/problem 
 */
void settings::load_kde_settings(){
	KConfig *_kconfig = new KConfig("kcmdisplayrc", true );
	
	/* KDE settings [DisplayEnergy] from kcmdisplayrc */
	if(_kconfig->hasGroup("DisplayEnergy")) {
		_kconfig->setGroup("DisplayEnergy");
		kde_settings->displayEnergySaving = _kconfig->readBoolEntry("displayEnergySaving", false);
		kde_settings->displayStandby = _kconfig->readNumEntry("displayStandby", -1);
		kde_settings->displaySuspend = _kconfig->readNumEntry("displaySuspend", -1);
		kde_settings->displayPowerOff = _kconfig->readNumEntry("displayPowerOff", -1);
	}
	
	_kconfig = new KConfig("kdesktoprc", true );
	/* KDE settings [ScreenSaver] from kdesktoprc */
	if(_kconfig->hasGroup("ScreenSaver")) {
		_kconfig->setGroup("ScreenSaver");
		kde_settings->enabled = _kconfig->readBoolEntry("Enabled", false);
		kde_settings->lock = _kconfig->readBoolEntry("Lock", false);
	}
}

