/***************************************************************************
                          jabbercore.h  -  description
                             -------------------
    begin                : Thu Sep 30 1999
    copyright            : (C) 1999 by Seth Hartbecke
    email                : gandalf@netins.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef JABBERCORE_H
#define JABBERCORE_H

#include <config.h>
#include <stdlib.h>
#include <time.h>

#include <qstring.h>
#include <qobject.h>
#include <qsocketnotifier.h>
#include <qlist.h>
#include <qmap.h>
#include <qdatetime.h>

#include <kapp.h>

#include "xmlnode.h"
#include "xmlstream.h"
#include "messagebin.h"
#include "infoquery.h"
#include "dlgconnectinprogress.h"
#include <prefs.h>


/* These are used on the parameter passed from
   newXMLStreamStatus(int n) */
#define STREAM_READY			1
#define STREAM_DNS_FAILED    		2
#define STREAM_SERVER_REJECTED 		3
#define STREAM_CONNECT_TIMEOUT		4

/**
  *@author Seth Hartbecke

  The core stuff.....
  */

class MessageBin;
class QueryManager;
class User;
class Group;
class Agent;

/**
 * JabberCore is the center of all communications to and from the jabber server.  It manages the connection,
 * keeps track of groups, users, provies a pointer to the query manager, message bin, and preferences
 * database.
 *
 * There is only ever one copy of jabbercore for the entire run of the application, a pointer to it
 * is sored in the global variable jabber.
 */
class JabberCore : public XMLStream  {
   Q_OBJECT
public:
	JabberCore(QWidget *parent=0, const char *name=0);
	~JabberCore();

        /**
	 * Returns the pointer to the messagebin.
	 *
	 * @see MessageBin
	 */
	MessageBin *messageBin();
        /**
	 * Retruns a pointer to the query manager.
	 *
	 * @see InfoQuery
	 * @see QueryManager
         */
	QueryManager *queryManager();
        /**
	 * Retruns the pointer to the preferences database.
	 *
	 * @see PrefsDB
	 */
        PrefsDB *prefs();

	/**
	 * Returns the jabber id of the currently logged in user.  This would be both the
	 * login name, server, and resource.
	 */
	JabberID whoAmI();

	/**
	 * This function returns the number of users the user database knows about.
	 *
	 * @see User#getUser
	 */
	int numberUsers();

	/**
	 * Retruns the user object for the nth user in the user database.
	 *
	 * @see User#numberUsers
	 */
	User *getUser(int n);
	/**
	 * Returns the user object that matches the string
	 */
        User *getUser(QString s);
	/**
	 * Search like seth@jabber.org
         * please do not add the resource
         * because the search will fail
	 */
	User *getUser(JabberID s);	// Search like jabber:seth@jabber.org
					// please do not add the resource
					// because the search will fail

        /**
	 * This function will create a new user object
         * It will begin to send the necessary commands to the server
         * to retrieve information, though it may not have the responce
	 * back before you get the object
	 */
	User *fetchUser(QString s);

	enum StatusType {
        	Chat,
                Normal,
                Away,
                ExtendedAway,
                DND,
                Offline,
                Unknown
                 };

	void setStatus( StatusType show = Normal, int priority = 10, QString description = "Online with the world's buggies jabber client" );
        StatusType getShow() { return _show; };
        int getPriority() { return _priority; };
        QString getDescription() { return _description; };

        Group *getGroup( QString name, bool create = true );
        int numberGroups() { return _groups.count(); };
        Group *getGroup( int );

        Agent *getAgent( QString agent ) { return _agents[ agent ]; };
        int numberAgents() { return _agents.count(); };
        Agent *getAgent( int );

        static QDateTime convertFromJabberTime( QString &source );
        static QString convertToJabberTime( QDateTime &source );

public slots:
	// Connect to specified server
	void connectToServer(QString server, int port, QString user, QString pass, QString nick, bool create=false);

signals:
	// General purpose signals for indicating when the connection
	// is gained/lost
	void connected();
	void disconnected();

        void changedStatus();

	/* this signal is emitted when a new user is added to the user list */
	void newUser(User *u);

        void newGroup(Group *g);
        void deletedGroup(Group *g);
private slots:

        void connectCanceled();

        // Called when the settings show up
        void settingsRecieved();

	// This is called in the responce to a roster request
	void rosterRecieved(InfoQuery *request, InfoQuery *result);

        // This is called when we get the agents list
        void agentsRecieved(InfoQuery *result, InfoQuery *result);

	// This is called in responce to a roster _push_
	void rosterPushed(InfoQuery *result);

	// The login result has been recieved
	void loginResults(InfoQuery *request, InfoQuery *result);

        // Version request
        void versionRequest( InfoQuery *request );

        void deleteGroupNotifiy(Group *g);

        void userDeleted( User *u );
protected:
             void rootTagRecieved(XMLNode *t);		// Continue login  -- From XMLStream
             void connectionLost();			// From XMLStream
             void recievedTag(XMLNode *t);                    // From XMLStream

private:
	MessageBin *messageStore;
	QueryManager *infoQueryManager;

	QList<User> users;
	                   
	/* Stored for login stuff */
	QString _user, _pass, _nick, _server;
	bool createAccount;
	DlgConnectInProgress *connectStatusDlg;

        /* stored my status stuff */
        StatusType _show;
        int _priority;
        QString _description;
	QMap<QString, Group *> _groups;
        QMap<QString, Agent *> _agents;
        PrefsDB *_prefs;

};

extern JabberCore *jabber;

#endif























