/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/* parseconf.h

   author: Ron Lawrence <lawrence@missioncriticallinux.com>
   version: $Revision: 1.12 $
   Routines to parse a configuration file with the following format:

   [section]
   thing=some text here
   somthing=12334
   else=123,123,345,445

   doubles = 12.3,22.45,45.34345
   [othersection]

   name=your name goes here
   # this is a comment regarding the figfile:
   figfile=/usr/local/files/figfile
     network  =   n1

   The file is parsed and its data is stored somewhere.  There are
   several functions that allow the data, once stored to be looked up.
   Client code doesn't need to know how the configuration items are
   stored or accessed, so that information isn't exported.

   Blank lines are ignored.  Lines that begin with a '#', possibly
   preceded by whitespace, are considered to be comment lines, and are
   ignored.

   Configuration items are stored and retrieved as strings, with
   string keys.  The client code must perform any conversions to other
   types.

   Part of the motivation for writing these functions in this form is
   that they are relatively easy to wrap using SWIG for extending
   interpreters.

   Thu Mar 23 16:54:13 2000 RAL

   Thu May 4 10:11:34 2000 Added several functions to the interface of
   this module.  These are 1. to allow a configuration to be parsed
   out of a buffer of characters, and 2. to provide a convenient
   interface to callers who only want to read or write a configuration
   to/from a shared partition, and who want to save/restore a
   configuration to/from a well known file.

   Thu May 11 12:03:50 2000 Added iterators eith pattern matching and
   filters.

  */
#ifdef __cplusplus
extern "C" {
#endif

#ifndef _STDIO_H
#include <stdio.h>
#endif 

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifndef PARSECONF_H
#define PARSECONF_H

#define CFG_OK                1
#define CFG_FALSE             0

#define CFG_NO_SUCH_SECTION  -1
#define CFG_DEFAULT          -2
#define CFG_FAILED           -3
#define CFG_LINE_TOO_LONG    -4
#define CFG_PARSE_FAILED     -5
#define CFG_INIT_FAILURE     -6
#define CFG_BAD_KEY          -7
#define CFG_TOO_BIG          -8

typedef int CFG_status;

#define CFG_LOCK_LOCKED_THIS_SESSION       1
#define CFG_LOCK_LOCKED_OTHER_SESSION      2
#define CFG_LOCK_NOT_LOCKED                4
#define CFG_LOCK_LOST                      8
#define CFG_LOCK_EXPIRED                   16
#define CFG_LOCK_FAILURE                   32
#define CFG_LOCK_STATUS_UNKNOWN            64

typedef int CFG_lockStatus;

/* Return CFG_OK if CFG_Init has been called and CFG_Destroy has not,
   or CFG_Init has been called since the last time CFG_Destroy was
   called.  Otherwise, returns CFG_FALSE.  */

CFG_status CFG_Initialized(void);

/* Return CFG_OK if CFG_Initialized would return CFG_OK and
   CFG_ReadFile has been called and CFG_Destroy has not been called,
   or CFG_Init, and CFG_ReadFile have been called since the last time
   CFG_Destroy was called.  Otherwise, returns CFG_FALSE.  */

CFG_status CFG_Loaded(void);

/* Undo the initialization of the configuration data structures,
   returning any used memory to the heap. */

CFG_status CFG_Destroy(void);

/** Advisory locking on the cluster database *****************/

/* Sieze the advisory session lock on the cluster database.  This
   should be called at the beginning of a sequence of commands that
   will potentially modify the configuration database. 

   Session is a string that names the session that is attempting to
   manipulate the lock.  It should be unique across processes, and
   across cluster nodes, but should be the same for sessions that
   conceptually live across multiple process lifetimes.  The session
   string should not contain any whitespace characters.

   The forced flag indicates whether the lock should be taken, even if
   it is already claimed by another session.  */

CFG_lockStatus CFG_Lock(char *session, int forced);

/* Release the advisory session lock on the cluster database.  This
   should be called at the end of a series of operations that might
   modify the configuration database. */

CFG_lockStatus CFG_Unlock(char *session, int forced);

/* Check the condition of the advisory session lock.  Call this before
   calling CFG_Read, or CFG_Write to determine the current session
   lock status of the configuration database. */

CFG_lockStatus CFG_CheckLock(char *session);

/* Return the lock string that was most recently read out of the
   database on disk.  We need this so we can tell the user who has the
   lock if is held. */
char * CFG_LockString(void);

/* Given a numeric string that represents the number of seconds in
   epoch time, return a nicely formatted string for the user to
   view. */
char * CFG_UserTimeString(char *timestring);

/** end locking sub-module ***********************************/

/* Read configuration data from the file named by file. */

CFG_status CFG_ReadFile(char *filename);

/* Read configuration data from the open file pointed to by f */

CFG_status CFG_ReadFD(FILE *f);

/* Write out the table of keys/values in such a way that it can be
   read back in. */

CFG_status CFG_WriteFile(char *filename);

/* Write out the key/value pairs to the open file pointed to by f */

CFG_status CFG_WriteFD(FILE *f);

/* Write the contents of the database structures to the cluster's
   shared disk partition. */

CFG_status CFG_Write(void);

/* Read the contents of the cluster's shared disk partition into the
   database structures.  Cleans the database structures prior to
   parsing the contents of the shared partition. */

CFG_status CFG_Read(void);

/* Copy the contents of the cluster's shared disk partition out to a
   backup file.  The backup file's name and location are specified in
   the global configuration file's "config" section in a "backupFile"
   entry. */

CFG_status CFG_Backup(void);

/* Restore a backup configuration file to the cluster's share disk
   partition. */

CFG_status CFG_Restore(void);

/* Get an entry from the configuration data tables. Section is the
   major section in the configuration file where the datum is to be
   found, ident is the identifier to read.  Default is a character
   string to return if there is no corresponding entry in the data
   tables.  result will be set to point to memory owned by the
   implementation.  Don't delete it!

   The ident argument is of the form "section%subsections%identifier".
   Where '%' is the path separator character, and subsections are zero
   or more separator separated subsection identifiers.  Path separator
   defaults to '%'.  Identifiers must be "c-style identifiers".  If
   ident is not a series of "c-style identifiers" separated by the
   path separator character, returns CFG_BAD_KEY. */

CFG_status CFG_Get(char* ident, char* dflt, char **result);

/* Set an entry into the configuration data tables. See above. */

CFG_status CFG_Set(char* ident, char* value);

/* Remove a single element from the in-memory configuration
   database. To update the database on disk, call one of the
   CFG_Write* entry points. */

CFG_status CFG_Remove(char *ident);

/* Remove all of the elements from the in-memory database that match
   the given pattern.  The pattern is a shell-style glob pattern.  To
   update the database on disk, call one of the CFG_Write* entry
   points. */

CFG_status CFG_RemoveMatch(char *pattern);

/* Read the file named by filename, and insert its contents at the key
   given by ident.  When CFG_WriteFile writes out the config file, the
   entry for the file will have all of its trailing newlines escaped.
   The newline escapes are removed when the config file is later read
   in. Returns CFG_FAILED if the given file cannot be opened or read.
   Returns CFG_ZERO_LENGTH_FILE if the file exists and is readable,
   but has zero length. */

CFG_status CFG_InsertEntryFromFile(char *ident, char *filename);

/* Write the entry in the database at the key given by ident to the
   file given by filename.  If the file already exists it is
   overwritten.  If there is no entry corresponding to ident, returns
   CFG_FAILED without affecting the file named by filename.  Returns
   CFG_FAILED if the given file cannot be opened or written. */

CFG_status CFG_ExtractEntryToFile(char *ident, char *filename);

/**********************************************************************/
/* Iterator functions */

/* An opaque data type for CFG iterators. */
struct CFG_Iter;

/* A typedef for function prototypes that filter functions passed to
   CFG_CreateFilterIterator must conform to, */
typedef int (*IterFilter)(char *key, char *value);

/* Iterator creation routines. */

/* Create an iterator that iterates over all of the keys in the
   table. */
CFG_status CFG_CreateIterator(struct CFG_Iter **iter);

/* Create an iterator that iterates over all of the keys in the table
   that match the given glob pattern string. */
CFG_status CFG_CreateGlobIterator(char *glob_pattern,
                                  struct CFG_Iter **iter);

/* Create an iterator that iterates over all of the keys in the table
   for which the keys and values satisfy the given filter function.
   To satisfy the given filter function means that the function
   returns non-zero for the given key, value pair. */
CFG_status CFG_CreateFilterIterator(IterFilter filter,
                                    struct CFG_Iter **iter);

/* Iterator destruction */
CFG_status CFG_DestroyIterator(struct CFG_Iter *iter);

/* Iterator operations */

/* Test the given iterator.  Return CFG_OK if there are more keys left
   to iterate over, return CFG_FALSE if the iterator has been
   exhausted. */
CFG_status CFG_IteratorMore(struct CFG_Iter *iter);

/* Get the next key, value pair from the database, as pointed to by
   the given iterator.  Sets key and value to point to character
   arrays within the storage space of the hash table.  Do not modify
   them! 

   The iterator contains a list of keys that were present in the table
   at the time the iterator was created.  The table may have been
   subsequently modified.  The function will return CFG_OK if the key
   had a corresponding value in the database.  It will return
   CFG_DEFAULT if not.  Other values returned, esp. those returned for
   various errors, are those that are returned by CFG_Get. */
CFG_status CFG_IteratorNext(struct CFG_Iter *iter,
                            char** key,
                            char** value);

/* Rewind the given iterator so that the next call to CFG_IteratorNext
   will return the first key, value pair in alphabetical order. */
CFG_status CFG_IteratorRewind(struct CFG_Iter *iter);

/* Return the number of items contained in the iterator in count. */
CFG_status CFG_IteratorCount(struct CFG_Iter *iter, int * count);

/* The following routines turn cluster locking on and off for accesses
to the shared partition.  This is necessary to allow the shared
database to be updated before the cluster is operational.  Note that
the sort of locking that is referred to here is entirely different
than the locking for which routine are provided above.  The locking
dealt with here is the strict, low level file locking on the cluster
shared database, the locking above is a more casual, advisory, session
locking mechanism. */

/* Turn off locking and unlocking of the cluster shared database.*/

CFG_status CFG_cluster_locking_off(void);

/* Turn locking back on for the cluster shared database. */

CFG_status CFG_cluster_locking_on(void);

#ifdef __cplusplus
}
#endif

#endif

