#include <qdialog.h>
#include <qwidget.h>
#include <qlistbox.h>
#include <qlineedit.h>
#include <qmultilineedit.h>
#include <qpushbutton.h>
#include <qtabwidget.h>
#include <qevent.h>
#include <qapplication.h>
#include <iostream>

#include "chatbox.h"
#include "globals.h"
#include "preferences.h"
#include "utility.h"

ChatBox::ChatBox() : QDialog()
{
	setCaption( "Kascade IRC Client" );
	resize( 650, 400 );
	setMinimumSize( 350, 200 );

	names = new QListBox( this );
	input = new QLineEdit( this );
	closeButton = new QPushButton( this );
	channels = new QTabWidget( this );

	closeButton->setText( "close" );

	sysmsgarea = new QMultiLineEdit( this );
	channels->addTab( sysmsgarea, "system" );
	sysmsgarea->setWordWrap( QMultiLineEdit::WidgetWidth );
	sysmsgarea->setReadOnly( TRUE );

	QObject::connect( input, SIGNAL( returnPressed() ), this, SLOT( textEntered() ) );
	QObject::connect( closeButton, SIGNAL( clicked() ), this, SLOT( closeChannel() ) );
	QObject::connect( names, SIGNAL( selected( int ) ), this, SLOT( createPrivateChannel( int ) ) );

	socket = new QSocket();
	QObject::connect( socket, SIGNAL( connected() ), this, SLOT( connected() ) );
	QObject::connect( socket, SIGNAL( readyRead() ), this, SLOT( incoming() ) );

	QObject::connect( channels, SIGNAL( selected( const QString & ) ), this, SLOT( selected( const QString & ) ) );

	setFont( config->ircFont() );

	state = DISCONNECTED;
	nicknamenr = 1;
}

void ChatBox::selected( const QString &title )
{
	if( title != "system" )	
		gotoPage( pageForChannel( channelForTitle( (const char *)title ) ) );
}

void ChatBox::gotoPage( QWidget *page )
{
	names->clear();

	if( page != sysmsgarea )
	{
		string name;

		for( unsigned int n = 0; n < namevv[vectorIndex( page )].size(); n++ )
		{
			name = namevv[ vectorIndex( page ) ][n];
			if( name[0] == '@' )
				names->insertItem( name.c_str(), 0 );
			else
				names->insertItem( name.c_str() );
		
		}
	}
	
	channels->showPage( page );
}

void ChatBox::login()
{
	if( state == DISCONNECTED )
	{
		printLine( "** connecting to server " + config->ircServer() + ", port 6666", currentArea() );
		myapp->processEvents();

		// socket->setMode( QSocket::Ascii );
		socket->connectToHost( config->ircServer().c_str(), 6666 );
	}	
}

void ChatBox::join( string title, string channel = "" )
{
	if( channel == "" )
		channel = title;

	input->setFocus();
	show();
	setActiveWindow();

	if( vectorIndex( channel ) != -1 )
		return;

	// determine title 

	int exists = TRUE;
	while( exists )
	{
		exists = FALSE;

		for( unsigned int n = 0; n < channelandtitlev.size(); n += 2 )
			if( channelandtitlev[n+1] == title )
			{
				title += "*";

				exists = TRUE;
				break;
			}
	} 
	
	channelandtitlev.push_back( channel );
	channelandtitlev.push_back( title );

	if( state == CONNECTED ) 
		transmitLine( "JOIN " + channel );
	else	
	{
		autojoin = channel;
		login();
	}	
}

void ChatBox::disconnect()
{
	if( state == DISCONNECTED )
		return;

	socket->close();
	state = DISCONNECTED; 
	nicknamenr = 1;

	for( int n = pagev.size()-1; n >= 0; n-- )
		removePage( pagev[n] );
		
	printLine( "** disconnected from server", sysmsgarea );
}

int ChatBox::vectorIndex( string channel )
{
	for( unsigned int n = 0; n < channelv.size(); n++ )
		if( channelv[n] == channel )
			return n;
	return -1;
}

int ChatBox::vectorIndex( QWidget *page )
{
	for( unsigned int n = 0; n < pagev.size(); n++ )
		if( pagev[n] == page )
			return n;
	return -1;
}

string ChatBox::currentChannel()
{
	if( vectorIndex( channels->currentPage() ) != -1 )
		return channelv[vectorIndex( channels->currentPage() )];
	return "";	
}

QWidget *ChatBox::pageForChannel( string channel )
{
	if( vectorIndex( channel ) != -1 )
		return pagev[vectorIndex( channel )];

	return 0;	
}

string ChatBox::titleForChannel( string channel )
{
	for( unsigned int n = 0; n < channelandtitlev.size(); n += 2 )
		if( channelandtitlev[n] == channel )	
			return channelandtitlev[n+1];
		
	return channel;		
}

string ChatBox::channelForTitle( string title )
{
	for( unsigned int n = 0; n < channelandtitlev.size(); n += 2 )
		if( channelandtitlev[n+1] == title )
			return channelandtitlev[n];

	return title;		
}

string ChatBox::tfc( string channel )
{
	string title = titleForChannel( channel );

	if( title[0] == '#' )
		return title;
	return "'" + title + "'";	
}

QMultiLineEdit *ChatBox::areaForChannel( string channel )
{
	return (QMultiLineEdit *)pageForChannel( channel );
}

QMultiLineEdit *ChatBox::currentArea()
{
	return (QMultiLineEdit *)(channels->currentPage());
}

void ChatBox::connected()
{
	printLine( "** connected, logging in..", currentArea() );

	if( !tryNickUser() )
	{
		printLine( "** no nicknames specified. edit preferences.", sysmsgarea ); 	
		printLine( "** disconnected", sysmsgarea ); 	
		nicknamenr = 1; 
	}	
}

int ChatBox::tryNickUser()
{
	if( nicknamenr == 5 )
		return FALSE;

	while( ( nicknamenr < 5 ) && 
	       ( ( user = config->ircNickname( nicknamenr++ ) ) == "" ) );

	if( user == "" )
		return FALSE;
	
	transmitLine( "NICK " + user );
	transmitLine( "USER " + user + " localhost " + config->ircServer() + " :" + user );

	return TRUE;	
}

int ChatBox::transmitLine( string s )
{
	//cout << "sending " + s << endl; 
	return (socket->writeBlock( ( s + "\n\r" ).c_str(), s.length()+2 ) != -1 );
}

void ChatBox::incoming()
{
	while( socket->canReadLine() )
	{
		string s = (const char *)( socket->readLine() );
		while( ( s[s.length()-1] == 0xd ) ||
		       ( s[s.length()-1] == 0xa ) )
	        	s = s.substr( 0, s.length()-1 );

		s = trim( s );

		//cout << "## " << s << endl;
		
		int k;

		string origin, code, params;

		origin = s.substr( 0, k = s.find_first_of( ' ' ) ); 
		s = s.substr( k+1, s.length()-k-1 );

		if( ( k = s.find_first_of( ' ' ) ) != -1 )
		{
			code = s.substr( 0, k ); 
			params = s.substr( k+1, s.length()-k-1 ); 
		}
		else
	   	    code = s;
		    
		processMessage( origin, code, params );
	}
}

void ChatBox::processMessage( string origin, string code, string params )
{
	string person;
	if( origin.find_first_of( '!' ) != -1 )
		person = origin.substr( 1, origin.find_first_of( '!' )-1 );

	if( code == "001" )
	{
		state = CONNECTED;
		printLine( "** you are logged in as " + user, sysmsgarea );

		if( autojoin != "" )
		{
			transmitLine( "JOIN " + autojoin );
			autojoin = "";
		}	
	}

	else if( origin == "PING" )
		transmitLine( "PONG " + code );
	
	else if( code == "JOIN" )
	{
		string channel = params.substr( 1, params.find_first_of( ' ' )-1 );

		if( person == user )
		{
			int exists = FALSE;
			unsigned int n;
			
			for( n = 0; n < channelv.size(); n++ )
				if( channelv[n] == channel )
				{
					exists = TRUE;
					break;
				}
		
			if( !exists )
			{
				channels->removePage( sysmsgarea );
				createChannel( channel );
				printLine( "** you have joined channel " + tfc( channel ), areaForChannel( channel ) ); 
			}
			else
			{
				joinedv[n] = TRUE;
				gotoPage( pageForChannel( channel ) );
				printLine( "** you have rejoined channel " + tfc( channel ), areaForChannel( channel ) );
			}	
		}	
		else		
		{
			namevv[ vectorIndex( channel ) ].push_back( person );
			gotoPage( channels->currentPage() );
			broadcast( "** " + person + " has joined channel " + tfc( channel ) );
		}	
	}

	else if( code == "KICK" )
	{
		int k;
		string channel = params.substr( 0, k = params.find_first_of( ' ' ) );
		string kicked = params.substr( k+1, params.find_first_of( ' ', k+1 )-k-1 );
		
		if( kicked == user )
		{
			joinedv[ vectorIndex( channel )	] = FALSE;
			namevv[ vectorIndex( channel ) ].clear();
			gotoPage( channels->currentPage() );

			if( person == user )
				broadcast( "** you have kicked yourself from channel " + tfc( channel ) );
			else	
				broadcast( "** " + person + " has kicked you from channel " + tfc( channel ) );
		}
		else
		{
			removeName( kicked, channel );	
			if( kicked == person ) 
				broadcast( "** " + person + " has kicked themselves from channel " + tfc( channel ) );
			else if( person == user )
				broadcast( "** you have kicked " + kicked + " from channel " + tfc( channel ) );
			else
				broadcast( "** " + kicked + " has been kicked by " + person + " from channel " + tfc( channel ) ); 
		}	
	}

	else if( code == "NICK" )
	{
		string nick = params.substr( 1, params.length()-1 );

		if( person == user )
		{
			broadcast( "** you are now known as " + nick );
			user = nick;
		}
		else
			broadcast( "** " + person + " is now known as " + nick );

		for( unsigned int m = 0; m < namevv.size(); m++ )
			for( unsigned int n = 0; n < namevv[m].size(); n++ )
				if( namevv[m][n] == person ) 
				{
					namevv[m][n] = nick;	
					break;
				}
				else if( namevv[m][n] == "@" + person )
				{
					namevv[m][n] = "@" + nick;
					break;
				}
			
		for( unsigned int n = 0; n < channelv.size(); n++ )
			if( channelv[n] == "$" + person )
			{
				channels->changeTab( pageForChannel( "$" + person ), ("$" + nick).c_str() );
				channelv[n] = "$" + nick;	
			}
			
		gotoPage( channels->currentPage() );
	}
	
	else if( code == "PART" )
	{
		string channel = params.substr( 0, params.find_first_of( ' ' ) );

//		cout << "channel" << channel << "!" << endl;

		if( person == user ) 
			removePage( pageForChannel( channel ) );
		else
		{
			removeName( person, channel );
			broadcast( "** " + person + " has left channel " + tfc( channel ) );
		}	
	}

	else if( code == "QUIT" )
	{
		for( unsigned int n = 0; n < channelv.size(); n++ )
			removeName( person, channelv[n] );

		broadcast( "** " + person + " has quit irc" );	
	}
	
	else if( code == "MODE" )
	{
		int k;
		string channel = params.substr( 0, k = params.find_first_of( ' ' ) );
		
		if( channel[0] == '#' )
		{
			origin = origin.substr( 1, origin.find_first_of( '!' )-1 );
			char flag = params[k+1];
			string modes = params.substr( k+2, params.find_first_of( ' ', k+2 )-k-2 );
			k = params.find_first_of( ' ', k+2 );
			string person = params.substr( k+1, params.length()-k-1 );
	
			/*cout << "origin " << origin << endl;
			cout << "channel " << channel << endl;
			cout << "flag " << flag << endl;
			cout << "modes " << modes << endl;
			cout << "person " << person << endl; */
			
			vector<string> personv;
			dirToVec( person, personv, ' ', FALSE );
			vector<string> &namev = namevv[ vectorIndex( channel ) ];

			if( flag == '+' )
			{
			  	if( modes.find_first_of( 'o' ) != -1 )
					for( unsigned int n = 0; n < personv.size(); n++ )
						giveOpStatus( origin, personv[n], namev );
			}
			else
			{
			  	if( modes.find_first_of( 'o' ) != -1 )
					for( unsigned int n = 0; n < personv.size(); n++ )
						removeOpStatus( origin, personv[n], namev );
			} 

			gotoPage( channels->currentPage() );
		}	
	}

	else if( code == "PRIVMSG" )
	{
		int k;
			
		string channel = params.substr( 0, k = params.find_first_of( ' ' ) );
		string msg = params.substr( k+2, params.length()-k-2 );
		
		string finalmsg;
		
		if( ( msg[0] == 1 ) &&
		    ( msg.find( "ACTION " ) == 1 ) )
			finalmsg = "** " + person + msg.substr( 7, msg.length()-8 );
		else
			finalmsg = person + "> " + msg;

		if( channel[0] == '#' )
			printLine( finalmsg, areaForChannel( channel ) );	
		else
		{
			if( vectorIndex( "$" + person ) == -1 )
				createChannel( "$" + person );
		
			if( person != user )
				printLine( finalmsg, areaForChannel( "$" + person ) );	
		}
	}

	else if( code == ERR_NICKNAMEINUSE )
	{
		int i = params.find_first_of( ' ' )+1;
		int j = params.find_first_of( ' ', i );
		string nick = params.substr( i, j-i );

		printLine( "** nickname " + nick + " already in use", currentArea() );

		if( ( state == DISCONNECTED ) &&
		    ( !tryNickUser() ) )
		{
			nicknamenr = 1;  
			printLine( "** all specified nicknames are in use. edit preferences.", sysmsgarea ); 
			printLine( "** disconnected", sysmsgarea ); 
		}	
	}
	
	else if( code == RPL_NAMREPLY )
	{
		int k = params.find_first_of( '#' );
		string chann = params.substr( k, params.find_first_of( ' ', k+2 )-k );
		int index = vectorIndex( chann );

		k = params.find_first_of( ':' );
		params = params.substr( k+1, params.length()-k-1 ); 
	
		namevv[index].erase( namevv[index].begin(), namevv[index].end() );

		while( ( k = params.find_first_of( ' ' ) ) != -1 )
		{
			namevv[index].push_back( params.substr( 0, k ) );	
			params = params.substr( k+1, params.length()-k-1 );
		}
		namevv[index].push_back( params );

		gotoPage( pageForChannel( chann ) );
	}
}
	

void ChatBox::giveOpStatus( string origin, string person, vector<string> &namev )
{
	for( unsigned int n = 0; n < namev.size(); n++ )
		if( namev[n] == person ) 
		{
			namev[n] = "@" + person;	
			break;
		}

	if( origin == user )
		printLine( "** you give operator status to " + person, currentArea() );
	else if( person == user )
		printLine( "** " + origin + " gives you operator status", currentArea() );
	else
		printLine( "** " + origin + " gives operator status to " + person, currentArea() );
}

void ChatBox::removeOpStatus( string origin, string person, vector<string> &namev )
{
	for( unsigned int n = 0; n < namev.size(); n++ )
		if( namev[n] == "@" + person ) 
		{
			namev[n] = person;	
			break;
		}	

	if( ( origin == user ) && ( person == user ) )
		printLine( "** you remove operator status from yourself", currentArea() );
	else if( origin == user )
		printLine( "** you remove operator status from " + person, currentArea() );
	else if( person == user )
		printLine( "** " + origin + " removes your operator status", currentArea() );
	else
		printLine( "** " + origin + " removes operator status from " + person, currentArea() );
}

void ChatBox::removeName( string person, string channel )
{
	//cout << person << "!" << channel << "!" << endl;

	vector<string> &namev  = namevv[ vectorIndex( channel ) ];

	for( unsigned int n = 0; n < namev.size(); n++ )
	{
		//cout << "!" << namev[n] << endl;

		if( ( namev[n] == person ) ||
		    ( namev[n] == "@" + person ) )
		{
			namev.erase( namev.begin()+n, namev.begin()+n+1 );
			break;
		}
	}
	
	gotoPage( channels->currentPage() );
}

void ChatBox::broadcast( string msg )
{
	printLine( msg, sysmsgarea );

	for( unsigned int n = 0; n < pagev.size(); n++ )
		printLine( msg, (QMultiLineEdit *)pagev[n] );
}

void ChatBox::textEntered()
{
	if( state == DISCONNECTED )
	{
		printLine( "** you are not connected", sysmsgarea );
		input->clear();
		return;
	}
	
	string s = (const char *)input->text();
	if( s == "" )
	    return;

	if( ( s == "/help" ) || ( s == "/HELP" ) )
	{
		QMultiLineEdit *area = (QMultiLineEdit *)(channels->currentPage() );
		
		printLine( "", area ); 
		printLine( "** Kascade IRC Client - version 1.0", area );
		printLine( "**", area );
		printLine( "** This is a very simple IRC Client. You can use another IRC Client", area );
		printLine( "** with the Kascade Browser if you like, by editing the preferences.", area );
		printLine( "**", area );
		printLine( "** Command-list: join, part, kick, op(er), deop, me, msg, nick", area );	
		printLine( "** Examples: /join #startrek, /op janeway, /kick neelix, /me rules, /part", area );
		printLine( "** Double-click on someone's name to chat privately.", area );
		printLine( "**", area );	
		printLine( "** Please send bug-reports and suggestions to: m.dufour@student.tudelft.nl", area );	
		printLine( "** Have fun!", area );	
	}

	else if( s[0] == '/' ) 
	{
		s.erase( 0, 1 );
		
		int k;
		string command;
		string params;
		
		if( ( k = s.find_first_of( ' ' ) ) == -1 )
			command = toUC( s );
		else
		{
			command = toUC( s.substr( 0, k ) );
			params = s.substr( k+1, s.length()-k-1 );
		}
	
		if( ( command == "OP" ) ||
		    ( command == "OPER" ) )
			transmitLine( "MODE " + currentChannel() + " +o " + params ); 

		if( command == "DEOP" ) 
			transmitLine( "MODE " + currentChannel() + " -o " + params );

		else if( command == "PART" ) 
		{
			if( currentChannel()[0] == '$' )
				removePage( channels->currentPage() );			
			else
				transmitLine( command + " " + currentChannel() + " " + params );	
		}
		
		else if( command == "KICK" )
			transmitLine( command + " " + currentChannel() + " " + params );

		else if( command == "ME" )
		{
			printLine( "** " + user + " " + params, currentArea() );
			if( currentChannel()[0] == '$' )	
				transmitLine( "PRIVMSG " + currentChannel().substr( 1, currentChannel().length()-1 ) + 
					" :" + (char)1 + "ACTION " + params + (char)1 );
			else
				transmitLine( "PRIVMSG " + currentChannel() + " :" + (char)1 + "ACTION " + params + (char)1 );
		}
		
		else if( command == "MSG" )
		{
			int k = params.find_first_of( ' ' ); 
			if( k != -1 )
			{
				string person = params.substr( 0, k );
				string msg = params.substr( k+1, params.length()-k-1 );

				transmitLine( "PRIVMSG " + person + " :" + msg ); 
			}	
		}
		
		else if( ( command == "JOIN" ) || ( command == "NICK" ) )
			transmitLine( s );
	}
	
	else if ( channels->currentPage() != sysmsgarea )
	{
		if( joinedv[ vectorIndex( channels->currentPage() ) ] == TRUE ) 
		{
			printLine( user + "> " + s, currentArea() );
			
			if( currentChannel()[0] == '$' )
				transmitLine( "PRIVMSG " + currentChannel().substr( 1, currentChannel().length()-1 ) + " :" + s );
			else	
				transmitLine( "PRIVMSG " + currentChannel() + " :" + s );
		}	
		else
			printLine( "** try rejoining the channel first", currentArea() );
			
	}

	else
		printLine( "** try to join a channel first", sysmsgarea );

	input->clear();
}

void ChatBox::printLine( string s, QMultiLineEdit *textarea )
{

	if( textarea != 0 )
	{
		//textarea->insertLine( s.c_str() );

		if( textarea->text() == "" )
			textarea->setText( s.c_str() );
		else
			textarea->append( s.c_str() );

		textarea->setCursorPosition( textarea->numLines(), 0 );
	}
//	else
//		cout << s << endl;
}

void ChatBox::closeChannel()
{
	if( channels->currentPage() == sysmsgarea )
		hide();

	if( currentChannel()[0] == '#' ) 
		transmitLine( "PART " + currentChannel() );

	removePage( channels->currentPage() );
}

void ChatBox::removePage( QWidget *page )
{
	if( page == 0 )
		return;

	for( unsigned int n = 0; n < pagev.size(); n++ )
		if( pagev[n] == page )
		{
			pagev.erase( pagev.begin()+n, pagev.begin()+n+1 );
			channelv.erase( channelv.begin()+n, channelv.begin()+n+1 );
			namevv.erase( namevv.begin()+n, namevv.begin()+n+1 );
			joinedv.erase( joinedv.begin()+n, joinedv.begin()+n+1 );
			break;
		}
		
	channels->removePage( page );

	if( channels->currentPage() == 0 )
	{
		channels->addTab( sysmsgarea, "system" );
		gotoPage( sysmsgarea );
		printLine( "** you have no channels open. use /join #channel to join one.", sysmsgarea );
	}	
}

void ChatBox::createPrivateChannel( int name ) 
{
	string channel = (const char *)(names->item( name )->text());
	if( channel[0] == '@' )
		channel.erase( 0, 1 );
	channel = "$" + channel;	

	createChannel( channel );
}

void ChatBox::createChannel( string channel ) 
{
	QMultiLineEdit *chatarea = new QMultiLineEdit( this );
	chatarea->setWordWrap( QMultiLineEdit::WidgetWidth );
	chatarea->setReadOnly( TRUE );

	if( channel[0] == '$' )
		channels->addTab( chatarea, channel.c_str() );
	else
		channels->addTab( chatarea, titleForChannel( channel ).c_str() );

	channelv.push_back( channel );
	pagev.push_back( chatarea );
	joinedv.push_back( TRUE );
	vector<string> namev;
	namevv.push_back( namev );

	gotoPage( chatarea );
	closeButton->setText( "close" );
}

void ChatBox::resizeEvent( QResizeEvent * )
{
	int h = this->height(), w = this->width();

	closeButton->setGeometry( w-120, h-70, 110, 30);
	names->setGeometry( w-120, 33, 110, h-113 );			
	if( channels != 0 )
		 channels->setGeometry( 10, 10, w-140, h-50  );
	input->setGeometry( 10, h-30, w-20, 20 );			
}
