// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the Light K-3D object, which encapsulates a light source within the 3D workspace
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/classes.h>
#include <k3dsdk/glutility.h>
#include <k3dsdk/iapplication.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/module.h>
#include <k3dsdk/property.h>
#include <k3dsdk/renderman.h>
#include <k3dsdk/transformable.h>
#include <k3dsdk/vectors.h>
#include <k3dsdk/viewport.h>

namespace libk3drenderman
{

/////////////////////////////////////////////////////////////////////////////
// light_implementation

class light_implementation :
	public k3d::viewport::drawable<k3d::transformable<k3d::persistent<k3d::object> > >,
	public k3d::viewport::ilight,
	public k3d::ri::ilight
{
	typedef k3d::viewport::drawable<k3d::transformable<k3d::persistent<k3d::object> > > base;

public:
	light_implementation(k3d::idocument& Document) :
		base(Document),
		m_shader(k3d::init_name("shader") + k3d::init_description("Light shader [object]") + k3d::init_object_value(0) + k3d::init_document(Document)),
		m_raytraced_shadows(k3d::init_name("raytraced-shadows") + k3d::init_description("Render raytraced shadows from this light [boolean]") + k3d::init_value(false) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::object_proxy(m_shader));
		enable_serialization(k3d::persistence::proxy(m_raytraced_shadows));

		register_property(m_shader);
		register_property(m_raytraced_shadows);
		
		m_position.changed_signal().connect(SigC::slot(*this, &light_implementation::async_redraw_all));
		m_orientation.changed_signal().connect(SigC::slot(*this, &light_implementation::async_redraw_all));
		m_scale.changed_signal().connect(SigC::slot(*this, &light_implementation::async_redraw_all));
	}

	void on_viewport_draw(const k3d::viewport::render_state& State)
	{
		glDisable(GL_LIGHTING);
		glDisable(GL_TEXTURE_1D);
		glDisable(GL_TEXTURE_2D);

		glColor3d(1.0, 1.0, 1.0);
		glLineWidth(1.0f);
		glDisable(GL_LINE_STIPPLE);

		EditorRenderPoint();
	}

	void on_viewport_select(const k3d::viewport::render_state& State)
	{
		k3d::glPushName(this);

		viewport_draw(State);

		k3d::glPopName();
	}

	void setup_viewport_light(unsigned long Light)
	{
		// Setup our own transformation matrix ...
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		k3d::push_opengl_matrix(matrix());

		// Offset our light index ...
		GLenum light = GLenum(Light + GL_LIGHT0);

		// Enable the light ...
		glEnable(light);

		// Setup intensity / attenuation ...
		double intensity = 1.0 / 1200.0;
		glLightf(light, GL_CONSTANT_ATTENUATION, 0.0f);
		glLightf(light, GL_LINEAR_ATTENUATION, 0.0f);
		glLightf(light, GL_QUADRATIC_ATTENUATION, GLfloat(intensity));

		// Setup color ...
		k3d::vector3 color(1.0, 1.0, 1.0);

		GLfloat glcolor[4];
		glcolor[0] = (GLfloat)color[0];
		glcolor[1] = (GLfloat)color[1];
		glcolor[2] = (GLfloat)color[2];
		glcolor[3] = 1.0f;

		glLightfv(light, GL_AMBIENT, glcolor);
		glLightfv(light, GL_DIFFUSE, glcolor);
		glLightfv(light, GL_SPECULAR, glcolor);

	/*
		// Setup light based on type
		if(DISTANT == m_Type.Value())
			{
				GLfloat gllightpos[] = { 0.0f, 0.0f, -1.0f, 0.0f };
				glLightfv(light, GL_POSITION, gllightpos);
			}
		else if(SPOTLIGHT == m_Type.Value())
			{
				// Set spotlight position
				GLfloat gllightpos[4];
				gllightpos[0] = 0.0f;
				gllightpos[1] = 0.0f;
				gllightpos[2] = 0.0f;
				gllightpos[3] = 1.0f;
				glLightfv(light, GL_POSITION, gllightpos);

				// Set spotlight angle & exponent
	//			glLightf(Light, GL_SPOT_CUTOFF, (GLfloat)m_Coneangle->GetValue1(time(), delta_time()));
	//			glLightf(Light, GL_SPOT_EXPONENT, (GLfloat)m_Spotlightfalloff->GetValue1(time(), delta_time()));

				const static GLfloat lightdirection[3] = { 0.0f, 0.0f, -1.0f };
				glLightfv(light, GL_SPOT_DIRECTION, lightdirection);
			}
		else if(POINT == m_Type.Value())
			{
	*/
				// Set point light position
				GLfloat gllightpos[4];
				gllightpos[0] = 0.0f;
				gllightpos[1] = 0.0f;
				gllightpos[2] = 0.0f;
				gllightpos[3] = 1.0f;
				glLightfv(light, GL_POSITION, gllightpos);

				// Make sure it doesn't become a spotlight
				glLightf(light, GL_SPOT_CUTOFF, 180.0f);
	/*
			}
	*/

		glPopMatrix();
	}

	void setup_renderman_light(const k3d::ri::render_state& State)
	{
		// If this is a shadow pass, we're done ...
		if(k3d::ri::SHADOW_MAP == State.render_context)
			return;

		// We only generate RIB on the last sample ...
		if(!k3d::ri::last_sample(State))
			return;

		// If the user hasn't selected a shader, we're done ...
		k3d::ri::ilight_shader* const shader = m_shader.interface();
		if(!shader)
			return;

		// Make it happen ...
		State.engine.RiTransformBegin();
		State.engine.RiTransform(k3d::ri::convert(matrix()));
		k3d::ri::parameter_list light_attributes;
		light_attributes.push_back(k3d::ri::parameter("shadows", k3d::ri::UNIFORM, k3d::ri::string(m_raytraced_shadows.value() ? "on" : "off")));
		State.engine.RiAttributeV("light", light_attributes);
		shader->setup_renderman_light_shader(State);
		State.engine.RiTransformEnd();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<light_implementation>,
			k3d::interface_list<k3d::viewport::ilight,
			k3d::interface_list<k3d::ri::ilight,
			k3d::interface_list<k3d::itransform_source,
			k3d::interface_list<k3d::itransform_sink > > > > > factory(
			k3d::classes::RenderManLight(),
			"RenderManLight",
			"RenderMan light source",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	void EditorRenderAmbient()
	{
		sdpgl::trace(__PRETTY_FUNCTION__);

		EditorRenderPoint();
	}

	void EditorRenderPoint()
	{
		sdpgl::trace(__PRETTY_FUNCTION__);

		// Draw a nice "furball"
		glBegin(GL_LINES);

		k3d::vector3 coords(0, 0, 0);

		glVertex3d(coords[0] + 1.0, coords[1], coords[2]);
		glVertex3d(coords[0] - 1.0, coords[1], coords[2]);
		glVertex3d(coords[0], coords[1] + 1.0, coords[2]);
		glVertex3d(coords[0], coords[1] - 1.0, coords[2]);
		glVertex3d(coords[0], coords[1], coords[2] + 1.0);
		glVertex3d(coords[0], coords[1], coords[2] - 1.0);

		glVertex3d(coords[0] + 0.4, coords[1] + 0.4, coords[2] + 0.4);
		glVertex3d(coords[0] - 0.4, coords[1] - 0.4, coords[2] - 0.4);
		glVertex3d(coords[0] - 0.4, coords[1] + 0.4, coords[2] + 0.4);
		glVertex3d(coords[0] + 0.4, coords[1] - 0.4, coords[2] - 0.4);
		glVertex3d(coords[0] + 0.4, coords[1] + 0.4, coords[2] - 0.4);
		glVertex3d(coords[0] - 0.4, coords[1] - 0.4, coords[2] + 0.4);
		glVertex3d(coords[0] - 0.4, coords[1] + 0.4, coords[2] - 0.4);
		glVertex3d(coords[0] + 0.4, coords[1] - 0.4, coords[2] + 0.4);

		glEnd();
	}

	void EditorRenderSpotlight()
	{
		sdpgl::trace(__PRETTY_FUNCTION__);

		GLUquadricObj* quadric = gluNewQuadric();
		assert_warning(quadric);

		// Draw the cone of illumination ...
	//			glScaled(m_CurrentWidthPlane, m_CurrentHeightPlane, m_CurrentNearPlane);
	//			glRotated(45.0, 0.0, 0.0, 1.0);
		glScaled(1.0, 1.0, -1.0);
		glTranslated(0.0, 0.0, -1.0);

		gluQuadricDrawStyle(quadric, GLenum(GLU_LINE));
		gluCylinder(quadric, sqrt(2.0), 0.001, 1.0, 8, 1);
		gluDeleteQuadric(quadric);

		// Draw the axis ...
		glBegin(GL_LINES);

		glVertex3d(0.0, 0.0, 0.0);
		glVertex3d(0.0, 0.0, -5.0);

		glEnd();
	}

	void EditorRenderDistant()
	{
		sdpgl::trace(__PRETTY_FUNCTION__);

		// Draw the "tail" of our arrow ...
		glBegin(GL_LINES);

		glVertex3d(0, 0, 0);
		glVertex3d(0, 0, -5);

		glEnd();

		// Draw the "head" of our arrow as a cone ...
		GLUquadricObj* quadric = gluNewQuadric();
		assert_warning(quadric);

		gluQuadricDrawStyle(quadric, GLenum(GLU_LINE));
		gluCylinder(quadric, 0.4, 0.001, 0.8, 8, 1);
		gluDeleteQuadric(quadric);
	}

	k3d_object_property(k3d::ri::ilight_shader, k3d::immutable_name, k3d::with_undo, k3d::local_storage) m_shader;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_raytraced_shadows;
};

/////////////////////////////////////////////////////////////////////////////
// light_factory

k3d::iplugin_factory& light_factory()
{
	return light_implementation::get_factory();
}

} // namespace libk3drenderman


