// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/color.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/utility.h>

#include <boost/multi_array.hpp>
#include <iomanip>
#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// triangulate_faces_implementation

class triangulate_faces_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	triangulate_faces_implementation(k3d::idocument& Document) :
		base(Document)
	{
		m_input_mesh.changed_signal().connect(SigC::slot(*this, &triangulate_faces_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &triangulate_faces_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		update_geometry(*output);

		return output;
	}

	/// Returns true iff the given object is selected
	struct is_selected
	{
		bool operator()(k3d::face* const Face)
		{
			return Face->selected;
		}
	};

	void update_geometry(k3d::mesh& Mesh)
	{
		// For each polyhedron ...
		for(k3d::mesh::polyhedra_t::iterator polyhedron_iterator = Mesh.polyhedra.begin(); polyhedron_iterator != Mesh.polyhedra.end(); ++polyhedron_iterator)
			{
				k3d::polyhedron& polyhedron = **polyhedron_iterator;

				// Remove all faces and edges ...
				k3d::polyhedron::faces_t faces;
				faces.swap(polyhedron.faces);

				k3d::polyhedron::edges_t edges;
				edges.swap(polyhedron.edges);
					
				// For each face ...
				k3d::triangulate(faces.begin(), faces.end(), std::back_inserter(polyhedron.faces), std::back_inserter(polyhedron.edges), std::back_inserter(Mesh.points));

				// Get rid of the leftovers ...
				std::for_each(faces.begin(), faces.end(), k3d::delete_object());
				std::for_each(edges.begin(), edges.end(), k3d::delete_object());

				assert_warning(is_valid(polyhedron));
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<triangulate_faces_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x3796ae29, 0xbcc84a5c, 0xb6bae620, 0x87a9b1bd),
				"TriangulateFaces",
				"Converts input faces into triangles",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// triangulate_faces_factory

k3d::iplugin_factory& triangulate_faces_factory()
{
	return triangulate_faces_implementation::get_factory();
}

} // namespace libk3dmesh

