// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Bart Janssens <bart.janssens@lid.kviv.be>
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/utility.h>

#include <boost/multi_array.hpp>
#include <iomanip>
#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// split_faces_implementation

class split_faces_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	split_faces_implementation(k3d::idocument& Document) :
		base(Document)
	{
		m_input_mesh.changed_signal().connect(SigC::slot(*this, &split_faces_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &split_faces_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		// For each polyhedron ...
		for(k3d::mesh::polyhedra_t::iterator polyhedron = output->polyhedra.begin(); polyhedron != output->polyhedra.end(); ++polyhedron)
			{
				const bool was_solid = k3d::is_solid(**polyhedron);

				k3d::polyhedron::faces_t new_faces;
				for(k3d::polyhedron::faces_t::iterator f = (*polyhedron)->faces.begin(); f != (*polyhedron)->faces.end(); ++f)
					{
						k3d::face& face = **f;

						std::vector<k3d::split_edge*> edges;
						for(k3d::split_edge* edge = face.first_edge; edge; edge = edge->face_clockwise)
							{
								if(edge->face_clockwise && edge->face_clockwise->vertex && edge->face_clockwise->vertex->selected)
									edges.push_back(edge);

								if(edge->face_clockwise == face.first_edge)
									break;
							}

						if(2 != edges.size())
							continue;

						k3d::split_edge* const edge1 = new k3d::split_edge(edges[0]->face_clockwise->vertex);
						(*polyhedron)->edges.push_back(edge1);

						k3d::split_edge* const edge2 = new k3d::split_edge(edges[1]->face_clockwise->vertex);
						(*polyhedron)->edges.push_back(edge2);

						edge1->face_clockwise = edges[1]->face_clockwise;
						edge2->face_clockwise = edges[0]->face_clockwise;

						edges[0]->face_clockwise = edge1;
						edges[1]->face_clockwise = edge2;

						k3d::join_edges(*edge1, *edge2);

						face.first_edge = edge1;

						k3d::face* const new_face = new k3d::face(edge2);
						new_faces.push_back(new_face);
					}

				(*polyhedron)->faces.insert((*polyhedron)->faces.end(), new_faces.begin(), new_faces.end());

				assert_warning(is_valid(**polyhedron));

				// If the polyhedron was solid when we started, it better be solid when we're done ...
				if(was_solid)
					assert_warning(is_solid(**polyhedron));
			}

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<split_faces_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x8f54ddc8, 0xee5c46f4, 0x98a35f72, 0xb4127f52),
				"SplitFaces",
				"Splits selected faces by connecting selected edges at their midpoints",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// split_faces_factory

k3d::iplugin_factory& split_faces_factory()
{
	return split_faces_implementation::get_factory();
}

} // namespace libk3dmesh


