// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/axis.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// set_material_implementation

class set_material_implementation :
	public k3d::material_collection<k3d::mesh_filter<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_filter<k3d::persistent<k3d::object> > > base;

public:
	set_material_implementation(k3d::idocument& Document) :
		base(Document)
	{
		m_input_mesh.changed_signal().connect(SigC::slot(*this, &set_material_implementation::on_reset_geometry));
		m_material.changed_signal().connect(SigC::slot(*this, &set_material_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &set_material_implementation::on_create_geometry));
	}
	
	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}
	
	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input mesh and modify the copy ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		reshape_geometry(*input, *output);

		return output;
	}

	template<typename T>
	void set_material(const bool Selected, const bool IgnoreSelection, k3d::imaterial* const Material, T* RHS)
	{
		if(Selected || IgnoreSelection)
			RHS->material = Material;
	}

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		const bool ignore_selection = !k3d::contains_selection(Source);
		k3d::imaterial* const material = m_material.interface();

/*		
		for(k3d::mesh::points_t::iterator point = Target.points.begin(); point != Target.points.end(); ++point)
			set_material((*point)->selected, ignore_selection, new_color, default_color, (*point)->vertex_data);
*/
			
		for(k3d::mesh::point_groups_t::iterator group = Target.point_groups.begin(); group != Target.point_groups.end(); ++group)
			set_material((*group)->selected, ignore_selection, material, *group);
			
		for(k3d::mesh::linear_curve_groups_t::iterator group = Target.linear_curve_groups.begin(); group != Target.linear_curve_groups.end(); ++group)
			set_material((*group)->selected, ignore_selection, material, *group);
			
		for(k3d::mesh::cubic_curve_groups_t::iterator group = Target.cubic_curve_groups.begin(); group != Target.cubic_curve_groups.end(); ++group)
			set_material((*group)->selected, ignore_selection, material, *group);
			
		for(k3d::mesh::bilinear_patches_t::iterator patch = Target.bilinear_patches.begin(); patch != Target.bilinear_patches.end(); ++patch)
			set_material((*patch)->selected, ignore_selection, material, *patch);
			
		for(k3d::mesh::bicubic_patches_t::iterator patch = Target.bicubic_patches.begin(); patch != Target.bicubic_patches.end(); ++patch)
			set_material((*patch)->selected, ignore_selection, material, *patch);

		for(k3d::mesh::polyhedra_t::iterator polyhedron = Target.polyhedra.begin(); polyhedron != Target.polyhedra.end(); ++polyhedron)
			{
				set_material((*polyhedron)->selected, ignore_selection, material, *polyhedron);
				
/*
				for(k3d::polyhedron::edges_t::const_iterator edge = (*polyhedron)->edges.begin(); edge != (*polyhedron)->edges.end(); ++edge)
					set_material((*edge)->selected, ignore_selection, new_color, default_color, (*edge)->facevarying_data);
					
				for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
					set_material((*face)->selected, ignore_selection, new_color, default_color, (*face)->uniform_data);
*/
			}
	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<set_material_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x7743a7f2, 0x995345b9, 0x8d6dacc7, 0x3fa37029),
				"SetMaterial",
				"Sets geometry materials",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
};

/////////////////////////////////////////////////////////////////////////////
// set_material_factory

k3d::iplugin_factory& set_material_factory()
{
	return set_material_implementation::get_factory();
}

} // namespace libk3dmesh

