// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// poly_grid_implementation

class poly_grid_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	poly_grid_implementation(k3d::idocument& Document) :
		base(Document),
		m_columns(k3d::init_name("columns") + k3d::init_description("Columns [integer]") + k3d::init_value(5) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_rows(k3d::init_name("rows") + k3d::init_description("Rows [integer]") + k3d::init_value(5) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_width(k3d::init_name("width") + k3d::init_description("Width [number]") + k3d::init_document(Document) + k3d::init_value(10.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_height(k3d::init_name("height") + k3d::init_description("Height [number]") + k3d::init_document(Document) + k3d::init_value(10.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)))
	{
		enable_serialization(k3d::persistence::proxy(m_rows));
		enable_serialization(k3d::persistence::proxy(m_columns));
		enable_serialization(k3d::persistence::proxy(m_width));
		enable_serialization(k3d::persistence::proxy(m_height));

		register_property(m_columns);
		register_property(m_rows);
		register_property(m_width);
		register_property(m_height);
		

		m_material.changed_signal().connect(SigC::slot(*this, &poly_grid_implementation::on_reset_geometry));
		
		m_columns.changed_signal().connect(SigC::slot(*this, &poly_grid_implementation::on_reset_geometry));
		m_rows.changed_signal().connect(SigC::slot(*this, &poly_grid_implementation::on_reset_geometry));
		m_width.changed_signal().connect(SigC::slot(*this, &poly_grid_implementation::on_reset_geometry));
		m_height.changed_signal().connect(SigC::slot(*this, &poly_grid_implementation::on_reset_geometry));
	
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &poly_grid_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}
	
	k3d::mesh* on_create_geometry()
	{
		std::auto_ptr<k3d::mesh> mesh(new k3d::mesh());
		
		mesh->polyhedra.push_back(new k3d::polyhedron());
		k3d::polyhedron& polyhedron = *mesh->polyhedra.back();
		polyhedron.material = m_material.interface();
		
		const k3d::grid_results_t grid = k3d::add_grid(*mesh, polyhedron, m_rows.property_value(), m_columns.property_value(), false, false);
		const boost::multi_array<k3d::point*, 2>& points = boost::get<0>(grid);

		const unsigned long point_rows = m_rows.property_value() + 1;
		const unsigned long point_columns = m_columns.property_value() + 1;
		const double width = m_width.property_value();
		const double height = m_height.property_value();
		
		for(unsigned long row = 0; row != point_rows; ++row)
			{
				const double row_percent = 0.5 - (static_cast<double>(row) / static_cast<double>(point_rows-1));
				
				for(unsigned long column = 0; column != point_columns; ++column)
					{
						const double column_percent = (static_cast<double>(column) / static_cast<double>(point_columns-1)) - 0.5;
						
						points[row][column]->position = k3d::vector3(width * column_percent, height * row_percent, 0);
					}
			}

		return_val_if_fail(is_valid(polyhedron), 0);
																																				
		return mesh.release();
	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<poly_grid_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
			k3d::uuid(0xacb3b4f8, 0x5cd6471c, 0xaed72686, 0xc576987c),
			"PolyGrid",
			"Generates a polygonal grid",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_columns;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_rows;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_width;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_height;
};

/////////////////////////////////////////////////////////////////////////////
// poly_grid_factory

k3d::iplugin_factory& poly_grid_factory()
{
	return poly_grid_implementation::get_factory();
}

} // namespace libk3dmesh

