// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// poly_cylinder_implementation

class poly_cylinder_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	poly_cylinder_implementation(k3d::idocument& Document) :
		base(Document),
		m_u_segments(k3d::init_name("u_segments") + k3d::init_description("Columns [integer]") + k3d::init_value(32) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(3UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_v_segments(k3d::init_name("v_segments") + k3d::init_description("Rows [integer]") + k3d::init_value(5) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_top(k3d::init_name("top") + k3d::init_description("Cap cylinder top [boolean]") + k3d::init_document(Document) + k3d::init_value(true)),
		m_bottom(k3d::init_name("bottom") + k3d::init_description("Cap cylinder bottom [boolean]") + k3d::init_document(Document) + k3d::init_value(true)),
		m_radius(k3d::init_name("radius") + k3d::init_description("Radius [number]") + k3d::init_document(Document) + k3d::init_value(5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_zmax(k3d::init_name("zmax") + k3d::init_description("Z Max [number]") + k3d::init_document(Document) + k3d::init_value(5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_zmin(k3d::init_name("zmin") + k3d::init_description("Z Min [number]") + k3d::init_document(Document) + k3d::init_value(-5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_u_power(k3d::init_name("u_power") + k3d::init_description("Radial Power [number]") + k3d::init_document(Document) + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar)))
	{
		enable_serialization(k3d::persistence::proxy(m_u_segments));
		enable_serialization(k3d::persistence::proxy(m_v_segments));
		enable_serialization(k3d::persistence::proxy(m_top));
		enable_serialization(k3d::persistence::proxy(m_bottom));
		enable_serialization(k3d::persistence::proxy(m_radius));
		enable_serialization(k3d::persistence::proxy(m_zmax));
		enable_serialization(k3d::persistence::proxy(m_zmin));
		enable_serialization(k3d::persistence::proxy(m_u_power));

		register_property(m_u_segments);
		register_property(m_v_segments);
		register_property(m_top);
		register_property(m_bottom);
		register_property(m_radius);
		register_property(m_zmax);
		register_property(m_zmin);
		register_property(m_u_power);

		m_material.changed_signal().connect(SigC::slot(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_u_segments.changed_signal().connect(SigC::slot(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_v_segments.changed_signal().connect(SigC::slot(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_top.changed_signal().connect(SigC::slot(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_bottom.changed_signal().connect(SigC::slot(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_radius.changed_signal().connect(SigC::slot(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_zmax.changed_signal().connect(SigC::slot(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_zmin.changed_signal().connect(SigC::slot(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_u_power.changed_signal().connect(SigC::slot(*this, &poly_cylinder_implementation::on_reset_geometry));
	
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &poly_cylinder_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}
	
	k3d::mesh* on_create_geometry()
	{
		std::auto_ptr<k3d::mesh> mesh(new k3d::mesh());

		mesh->polyhedra.push_back(new k3d::polyhedron());
		k3d::polyhedron& polyhedron = *mesh->polyhedra.back();
		polyhedron.material = m_material.interface();

		// Create the cylinder body ...
		const k3d::grid_results_t grid = k3d::add_grid(*mesh, polyhedron, m_v_segments.property_value(), m_u_segments.property_value(), false, true);
		const boost::multi_array<k3d::point*, 2>& points = boost::get<0>(grid);
		const boost::multi_array<k3d::split_edge*, 3>& edges = boost::get<1>(grid);
//		const boost::multi_array<k3d::face*, 2>& faces = boost::get<2>(grid);

/*
for(unsigned long v = 0; v != edges.shape()[0]; ++v)
{
	for(unsigned long col = 0; col != edges.shape()[1]; ++col)
		{
			for(unsigned long i = 0; i != 4; ++i)
				std::cerr << v << ", " << col << ", " << i << ": " << edges[v][col][i]->companion << std::endl;
		}
}
*/

		// Define the shape of the cylinder ...
		const unsigned long point_v_segments = points.shape()[0];
		const unsigned long point_u_segments = points.shape()[1];
		
		// Optionally cap the top of the cylinder ...
		const bool top = m_top.property_value();
		if(top)
			{
				std::vector<k3d::split_edge*> new_edges;
				for(unsigned long u = 0; u != point_u_segments; ++u)
					new_edges.push_back(new k3d::split_edge(edges[0][(point_u_segments - u) % point_u_segments][0]->vertex));

				for(unsigned long u = 0; u != point_u_segments; ++u)
					{
						new_edges[u]->face_clockwise = new_edges[(u + 1) % point_u_segments];
						k3d::join_edges(*new_edges[u], *edges[0][(point_u_segments-1)-u][0]);
					}

				polyhedron.edges.insert(polyhedron.edges.end(), new_edges.begin(), new_edges.end());
				polyhedron.faces.push_back(new k3d::face(new_edges[0]));
			}

		// Optionally cap the bottom of the cylinder ...
		const bool bottom = m_bottom.property_value();
		if(bottom)
			{
				std::vector<k3d::split_edge*> new_edges;
				for(unsigned long u = 0; u != point_u_segments; ++u)
					new_edges.push_back(new k3d::split_edge(edges[point_v_segments-2][u][3]->vertex));

				for(unsigned long u = 0; u != point_u_segments; ++u)
					{
						new_edges[u]->face_clockwise = new_edges[(u + 1) % point_u_segments];
						k3d::join_edges(*new_edges[u], *edges[point_v_segments-2][u][2]);
					}

				polyhedron.edges.insert(polyhedron.edges.end(), new_edges.begin(), new_edges.end());
				polyhedron.faces.push_back(new k3d::face(new_edges[0]));
			}

		// Shape the cylinder points
		const double radius = m_radius.property_value();
		const double zmax = m_zmax.property_value();
		const double zmin = m_zmin.property_value();
		const double u_power = m_u_power.property_value();
		
		const double inv_u_power = u_power ? 1.0 / u_power : 1.0;
		
		for(unsigned long v = 0; v != point_v_segments; ++v)
			{
				const double height = static_cast<double>(v) / static_cast<double>(point_v_segments - 1);
				
				for(unsigned long u = 0; u != point_u_segments; ++u)
					{
						const double theta = k3d::pi_times_2() * static_cast<double>(u) / static_cast<double>(point_u_segments);
						
						double x = cos(theta);
						double y = -sin(theta);
						double z = k3d::mix(zmax, zmin, height);

						x = radius * k3d::sign(x) * std::pow(std::abs(x), inv_u_power);
						y = radius * k3d::sign(y) * std::pow(std::abs(y), inv_u_power);
						
						points[v][u]->position = k3d::vector3(x, y, z);
					}
			}

		return_val_if_fail(is_valid(polyhedron), 0);

		if(top && bottom)
			assert_warning(is_solid(polyhedron));
						
		return mesh.release();
	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<poly_cylinder_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
			k3d::uuid(0xd8c4d9fd, 0x42334a54, 0xa4b48185, 0xd8506489),
			"PolyCylinder",
			"Generates a polygonal cylinder with optional endcaps",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_u_segments;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_v_segments;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_top;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_bottom;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_radius;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_zmax;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_zmin;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_u_power;
};

/////////////////////////////////////////////////////////////////////////////
// poly_cylinder_factory

k3d::iplugin_factory& poly_cylinder_factory()
{
	return poly_cylinder_implementation::get_factory();
}

} // namespace libk3dmesh

