// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/renderman.h>
#include <k3dsdk/transform.h>

#include <boost/random/lagged_fibonacci.hpp>
#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// lightning_implementation

class lightning_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	lightning_implementation(k3d::idocument& Document) :
		base(Document),
		m_segments(k3d::init_name("segments") + k3d::init_description("Segments [integer]") + k3d::init_value(20) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(0L)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_width(k3d::init_name("width") + k3d::init_description("Width [number]") + k3d::init_document(Document) + k3d::init_value(0.2) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_radius(k3d::init_name("radius") + k3d::init_description("Radius [number]") + k3d::init_document(Document) + k3d::init_value(0.2) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance)))
	{
		enable_serialization(k3d::persistence::proxy(m_segments));
		enable_serialization(k3d::persistence::proxy(m_width));
		enable_serialization(k3d::persistence::proxy(m_radius));

		register_property(m_segments);
		register_property(m_width);
		register_property(m_radius);
		
		m_material.changed_signal().connect(SigC::slot(*this, &lightning_implementation::on_reset_geometry));
		
		m_segments.changed_signal().connect(SigC::slot(*this, &lightning_implementation::on_reset_geometry));
		m_width.changed_signal().connect(SigC::slot(*this, &lightning_implementation::on_reset_geometry));
		m_radius.changed_signal().connect(SigC::slot(*this, &lightning_implementation::on_reset_geometry));
		
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &lightning_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}
	
	k3d::mesh* on_create_geometry()
	{
		k3d::mesh* const mesh = new k3d::mesh();

		const unsigned long segments = m_segments.property_value();
		const double width = m_width.property_value();
		const double radius= m_radius.property_value();
		
		k3d::linear_curve_group* const group = new k3d::linear_curve_group();
		group->material = m_material.interface();
		group->constant_data["constantwidth"] = static_cast<k3d::ri::real>(width);

		k3d::linear_curve* const curve = new k3d::linear_curve();

		boost::lagged_fibonacci607 random;
		mesh->points.resize(segments+1);
		for(unsigned long i = 0; i <= segments; ++i)
			{
				mesh->points[i] = new k3d::point(radius * (random() - 0.5), k3d::mix(0.5, -0.5, static_cast<double>(i) / static_cast<double>(segments)), radius * (random() - 0.5));
			}
		curve->control_points = mesh->points;

		group->curves.push_back(curve);
		mesh->linear_curve_groups.push_back(group);

		return mesh;
	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<lightning_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
			k3d::uuid(0x7bc13682, 0xfb5f44ad, 0x97a524ad, 0x5f3d5c6f),
			"Lightning",
			"Simulates a lightning stroke using curves",
			"Objects",
			k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d_measurement_property(long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_segments;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_width;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_radius;
};

/////////////////////////////////////////////////////////////////////////////
// lightning_factory

k3d::iplugin_factory& lightning_factory()
{
	return lightning_implementation::get_factory();
}

} // namespace libk3dmesh
