// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements a deformation bone primitive for use with skeletons & kinematics
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Brett W. McCoy (bmccoy@chapelperilous.net)
*/

#include <k3dsdk/bounded.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/transformable.h>
#include <k3dsdk/viewport.h>

namespace libk3ddeformation
{

namespace detail
{

/// Functor object for applying a transformation to a collection of points
struct transform_points
{
	transform_points(const k3d::matrix4& Matrix) :
		matrix(Matrix)
	{
	}

	void operator()(k3d::point* const Point)
	{
		Point->position = matrix * Point->position;
	}

	const k3d::matrix4 matrix;
};

}

/////////////////////////////////////////////////////////////////////////////
// deformation_bone_implementation

class deformation_bone_implementation :
	public k3d::bounded<k3d::viewport::drawable<k3d::mesh_filter<k3d::transformable<k3d::persistent<k3d::object> > > > >
{
	typedef k3d::bounded<k3d::viewport::drawable<k3d::mesh_filter<k3d::transformable<k3d::persistent<k3d::object> > > > > base;

public:
	deformation_bone_implementation(k3d::idocument& Document) :
		base(Document),
		m_length(k3d::init_name("length") + k3d::init_description("Length [number]") + k3d::init_value(5.0) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)))

	{
		enable_serialization(k3d::persistence::proxy(m_length));

		register_property(m_length);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &deformation_bone_implementation::on_reset_geometry));
		m_input_matrix.changed_signal().connect(SigC::slot(*this, &deformation_bone_implementation::on_reset_geometry));

		m_position.changed_signal().connect(SigC::slot(*this, &deformation_bone_implementation::on_reset_geometry));
		m_orientation.changed_signal().connect(SigC::slot(*this, &deformation_bone_implementation::on_reset_geometry));
		m_scale.changed_signal().connect(SigC::slot(*this, &deformation_bone_implementation::on_reset_geometry));

		m_length.changed_signal().connect(SigC::slot(*this, &deformation_bone_implementation::async_redraw_all));
		m_position.changed_signal().connect(SigC::slot(*this, &deformation_bone_implementation::async_redraw_all));
		m_orientation.changed_signal().connect(SigC::slot(*this, &deformation_bone_implementation::async_redraw_all));
		m_scale.changed_signal().connect(SigC::slot(*this, &deformation_bone_implementation::async_redraw_all));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &deformation_bone_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		// Transform the points ...
		std::for_each(output->points.begin(), output->points.end(), detail::transform_points(matrix()));

		return output;
	}

	const k3d::bounding_box extents()
	{
		const double length = m_length.property_value();
		return k3d::bounding_box(.5, -.5, length, -.5, .5, -.5);
	}

	void draw(const double length)
	{
		glBegin(GL_TRIANGLE_FAN);
		glVertex3d(0, 0, length);
		glVertex3d(0, 0.5, 0);
		glVertex3d(0.5, 0, 0);
		glVertex3d(0, -0.5, 0);
		glVertex3d(-0.5, 0, 0);
		glVertex3d(0, 0.5, 0);
		glEnd();

		glBegin(GL_TRIANGLE_FAN);
		glVertex3d(0, 0, length * -0.1);
		glVertex3d(0, 0.5, 0);
		glVertex3d(0.5, 0, 0);
		glVertex3d(0, -0.5, 0);
		glVertex3d(-0.5, 0, 0);
		glVertex3d(0, 0.5, 0);
		glEnd();
	}

	void on_viewport_draw(const k3d::viewport::render_state& State)
	{
		glColor3dv(is_selected() ? k3d::vector3(1, 1, 1) : k3d::vector3(0, 0, 0));
		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
		glDisable(GL_TEXTURE_1D);
		glDisable(GL_TEXTURE_2D);
		glDisable(GL_LIGHTING);
		glDisable(GL_AUTO_NORMAL);
		glDisable(GL_CULL_FACE);

		draw(m_length.property_value());
	}

	void on_viewport_select(const k3d::viewport::render_state& State)
	{
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		glDisable(GL_TEXTURE_1D);
		glDisable(GL_TEXTURE_2D);
		glDisable(GL_LIGHTING);
		glDisable(GL_AUTO_NORMAL);
		glDisable(GL_CULL_FACE);

		k3d::glPushName(this);
		draw(m_length.property_value());
		k3d::glPopName();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<deformation_bone_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink> > > factory(
				k3d::uuid(0x54e46452, 0x681f4ed6, 0xb30832d5, 0x06a69295),
				"DeformationBone",
				"Deformation Bone",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_length;
};

/////////////////////////////////////////////////////////////////////////////
// deformation_bone_factory

k3d::iplugin_factory& deformation_bone_factory()
{
	return deformation_bone_implementation::get_factory();
}

} // namespace libk3ddeformation


