// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/object.h>
#include <k3dsdk/computed_property.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/module.h>

namespace libk3dcore
{

class scalar_rand_implementation : 
	public k3d::persistent<k3d::object> 
{
	typedef k3d::persistent<k3d::object>  base;
public:
	scalar_rand_implementation(k3d::idocument& Document) : 
		base(Document),
		m_seed(k3d::init_name("seed") + k3d::init_description("seed [double]") + k3d::init_value(0) + k3d::init_document(Document)),
		m_output("output", "pseudo-random output [double]", k3d::method_call(*this, &scalar_rand_implementation::get_value))
	{
		register_property(m_seed);
		register_property(m_output);

		enable_serialization(k3d::persistence::proxy(m_seed));

		m_seed.changed_signal().connect(SigC::slot(*this, &scalar_rand_implementation::reset_rand));
		m_seed.changed_signal().connect(m_output.changed_signal().slot());
	}

	void reset_rand()
	{
		srand(m_seed.property_value());
	}
	
	double get_value()
	{
		return std::rand();
	}
	
	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<scalar_rand_implementation> > factory(
			k3d::uuid(0x14d8ab55, 0x1a014938, 0x9e1ad629, 0x86181375),
			"ScalarPseudoRand", 
			"Produce pseudo-random output", 
			"Objects", 
			k3d::iplugin_factory::EXPERIMENTAL);
		return factory;
	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}
	
private:
	k3d_data_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_seed;
	k3d::computed_property<double, k3d::method_call_t<scalar_rand_implementation, double> > m_output;
	
};

k3d::iplugin_factory& scalar_rand_factory()
{
	return scalar_rand_implementation::get_factory();
}

} //namespace libk3dcore
