// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/algebra.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/computed_property.h>
#include <k3dsdk/itransform_sink.h>
#include <k3dsdk/itransform_source.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/module.h>

namespace libk3dcore
{

/////////////////////////////////////////////////////////////////////////////
// position_implementation

class position_implementation :
	public k3d::persistent<k3d::object>,
	public k3d::itransform_source,
	public k3d::itransform_sink
{
	typedef k3d::persistent<k3d::object>  base;

public:
	position_implementation(k3d::idocument& Document) :
		base(Document),
		m_input(k3d::init_name("input_matrix") + k3d::init_description("Input matrix [matrix4]") + k3d::init_value(k3d::identity3D()) + k3d::init_document(Document)),
		m_x(k3d::init_name("x") + k3d::init_description("X [number]") + k3d::init_document(Document) + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_y(k3d::init_name("y") + k3d::init_description("Y [number]") + k3d::init_document(Document) + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_z(k3d::init_name("z") + k3d::init_description("Z [number]") + k3d::init_document(Document) + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_output("output_matrix", "Output matrix [matrix4]", k3d::method_call(*this, &position_implementation::output_value))
	{
		enable_serialization(k3d::persistence::proxy(m_x));
		enable_serialization(k3d::persistence::proxy(m_y));
		enable_serialization(k3d::persistence::proxy(m_z));

		register_property(m_input);
		register_property(m_x);
		register_property(m_y);
		register_property(m_z);
		register_property(m_output);
		
		m_input.changed_signal().connect(m_output.changed_signal().slot());
		m_x.changed_signal().connect(m_output.changed_signal().slot());
		m_y.changed_signal().connect(m_output.changed_signal().slot());
		m_z.changed_signal().connect(m_output.changed_signal().slot());
	}

	k3d::iproperty& transform_source_output()
	{
		return m_output;
	}
	
	k3d::iproperty& transform_sink_input()
	{
		return m_input;
	}

	k3d::matrix4 output_value()
	{
		return m_input.property_value() * translation3D(k3d::vector3(m_x.property_value(), m_y.property_value(), m_z.property_value()));
	}
			
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<position_implementation>,
			k3d::interface_list<k3d::itransform_source,
			k3d::interface_list<k3d::itransform_sink > > > factory(
			k3d::classes::Position(),
			"Position",
			"Creates a position transform matrix",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

	k3d_data_property(k3d::matrix4, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_input;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_x;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_y;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_z;
	k3d::computed_property<k3d::matrix4, k3d::method_call_t<position_implementation, k3d::matrix4> > m_output;
};

/////////////////////////////////////////////////////////////////////////////
// position_factory

k3d::iplugin_factory& position_factory()
{
	return position_implementation::get_factory();
}

} // namespace libk3dcore




