#ifndef K3DUI_SPIN_BUTTON_H
#define K3DUI_SPIN_BUTTON_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares the k3dSpinButton class, which provides a standard MVC UI for scalar values
		\author Tim Shead (tshead@k-3d.com)
		\author Dan Erikson <derikson@montana.com>
*/

#include "k3dcontrol.h"
#include <k3dsdk/iproperty.h>

// Forward declarations
class sdpGtkEventTimeout;

namespace k3d
{

namespace spin_button
{

/////////////////////////////////////////////////////////////////////////////
// idata_proxy

/// Abstract interface for an object that proxies a data source for a spin-button control (i.e. the "model" in model-view-controller)
class idata_proxy
{
public:
	virtual ~idata_proxy() {}

	/// Returns true iff the underlying data source is writable
	virtual bool writable() = 0;
	/// Called to return the underlying data value
	virtual double value() = 0;
	/// Called to set a new data value
	virtual void set_value(const double Value) = 0;
	/// Signal emitted if the underlying data changes
	typedef SigC::Signal0<void> changed_signal_t;
	/// Signal emitted if the underlying data changes
	virtual changed_signal_t& changed_signal() = 0;

protected:
	idata_proxy() {}
	idata_proxy(const idata_proxy& RHS) {}
	idata_proxy& operator=(const idata_proxy& RHS) { return *this; }
};

/////////////////////////////////////////////////////////////////////////////
// data_proxy

/// Provides an implementation of k3d::spin_button::idata_proxy that supports any data source that supports the value(), set_value(), and changed_signal() concepts
template<typename data_t>
class data_proxy :
	public idata_proxy
{
public:
	data_proxy(data_t& Data) :
		m_data(Data)
	{
	}

	bool writable()
	{
		return true;
	}

	double value()
	{
		return m_data.value();
	}

	void set_value(const double Value)
	{
		m_data.set_value(static_cast<typename data_t::value_t>(Value));
	}

	changed_signal_t& changed_signal()
	{
		return m_data.changed_signal();
	}

private:
	data_proxy(const data_proxy& RHS);
	data_proxy& operator=(const data_proxy& RHS);
	~data_proxy() {}

	data_t& m_data;
};

/////////////////////////////////////////////////////////////////////////////
// proxy

/// Convenience factory function for creating k3d::spin_button::idata_proxy objects
template<typename data_t>
std::auto_ptr<idata_proxy> proxy(data_t& Data)
{
	return std::auto_ptr<idata_proxy>(new data_proxy<data_t>(Data));
}

/// Convenience factory function for creating k3d::spin_button::idata_proxy objects, specialized for k3d::iproperty
std::auto_ptr<idata_proxy> proxy(iproperty& Data);

/////////////////////////////////////////////////////////////////////////////
// control

/// Provides a UI for manipulating boolean quantities (i.e. the view and the controller from model-view-controller)
class control :
	public k3dControl
{
public:
	control(k3d::iunknown* const CommandNodeParent, const std::string CommandNodeName);
	~control();

	/// Called by the framework when instantiating the button from a GTKML (XML) document
	bool Create(sdpGtkIObjectContainer* const ObjectContainer, sdpxml::Document& Document, sdpxml::Element& Element);

	/// Attaches the button to the data it fronts for
	bool attach(std::auto_ptr<idata_proxy> Data, k3d::istate_recorder* const StateRecorder, const std::string StateChangeName);

	/// Sets the precision of the value display, in digits to the right of the decimal point
	void set_precision(const unsigned long Precision);
	/// Sets the step increment between values when the user clicks on the up or down arrows
	void set_step_increment(const double StepIncrement);
	/// Sets the physical unit-of-measure that should be used to display values
	void set_units(const std::type_info& Units);

	const std::string CustomType() const;

	// k3d::icommand_node implementation
	bool execute_command(const std::string& Command, const std::string& Arguments);

private:
	typedef k3dControl base;

	/// Enumerates scroll "modes"
	typedef enum
	{
		NONE,
		INCREMENT,
		DECREMENT
	} scroll_t;

	/// Called whenever the the button state needs to be updated
	void update();

	/// Called to handle SDPGTK events
	void OnEvent(sdpGtkEvent* Event);
	void on_scrolling();
	void on_start_scrolling();
	void on_faster_scrolling();
	void on_fastest_scrolling();
	/// Called when the user tabs into / clicks on the edit control
	void on_value_focus_in();
	/// Called when the user tabs out of the edit control
	void on_value_focus_out();
	/// Called when the user hits the "enter" key
	void on_value_activate();
	/// Called when the user presses the "up" button
	void on_up_pressed();
	/// Called when the user releases the "up" button
	void on_up_released();
	/// Called when the user presses the "down" button
	void on_down_pressed();
	/// Called when the user releases the "down" button
	void on_down_released();
	/// Called when the GTK+ widgets are about to disappear
	void on_destroy();

	/// Called when the user begins scrolling by holding-down one of the arrow buttons
	void start_scrolling(const scroll_t ScrollType);
	/// Called when the user finishs scrolling by releasing an arrow button
	void cancel_scrolling();
	/// Called prior to making one-or-more changes to the underlying data source
	void push_editing();
	/// Called to increment the current value
	void increment();
	/// Called to decrement the current value
	void decrement();
	/// Called to modify the underlying data source
	void set_value(const double Value);
	/// Called once all changes to the underlying data source are complete
	void pop_editing();
	/// Called once all changes to the underlying data source are complete
	void pop_editing(const std::string NewValue);
	/// Formats a value and displays it in the edit control
	void format_edit_control(const double Value);
	/// Returns the contents of the edit control as a value
	const double read_edit_control();

	/// Stores the current scroll mode
	scroll_t m_scroll_mode;
	/// Stores the event that drives continuous scrolling
	sdpGtkEventTimeout* m_scrolling_event;
	/// Stores the event that gets scrolling started
	sdpGtkEventTimeout* m_start_scrolling_event;
	/// Stores the event that gets scrolling going faster
	sdpGtkEventTimeout* m_faster_scrolling_event;
	/// Stores the event that gets scrolling going at high speed
	sdpGtkEventTimeout* m_fastest_scrolling_event;

	/// Storeas a reference to the underlying data object
	std::auto_ptr<idata_proxy> m_data;
	/// Stores the increment used to modify the button's value using the spin controls
	double m_step_increment;
	/// Stores the number of digits of precision to display
	unsigned long m_precision;
	/// Stores the type of type of real-world-units to use for formatting & parsing
	const std::type_info* m_units;
};

} // namespace spin_button

} // namespace k3d

#endif // K3DUI_SPIN_BUTTON_H

