// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the options_dialog, which provides a UI for changing global user options
		\author Tim Shead (tshead@k-3d.com)
*/

#include "button.h"
#include "check_button.h"
#include "options_dialog.h"

#include <k3dsdk/application.h>
#include <k3dsdk/ioptions.h>
#include <k3dsdk/user_interface.h>

#include <sdpgtk/sdpgtkevents.h>

#include <fstream>

namespace
{

const std::string control_templatedocument = "templatedocument";
const std::string control_choosetemplatedocument = "choosetemplatedocument";
const std::string control_preferredlanguage = "preferredlanguage";
const std::string control_defaultengine = "defaultengine";
const std::string control_htmlviewer = "htmlviewer";
const std::string control_bitmapviewer = "bitmapviewer";
const std::string control_mpegviewer = "mpegviewer";
const std::string control_slpreprocessor = "slpreprocessor";
const std::string control_slparser = "slparser";
const std::string control_renderfarmpath = "renderfarmpath";

} // namespace

namespace k3d
{

/////////////////////////////////////////////////////////////////////////////
// options_dialog

options_dialog::options_dialog(k3d::icommand_node& Parent) :
	base(&Parent, "options", new k3d::options_window_geometry_store()),
	m_show_black_box_recorder(k3d::init_value(k3d::application().options().blackbox_recorder_at_startup())),
	m_show_tutorial_menu(k3d::init_value(k3d::application().options().tutorial_menu_at_startup())),
	m_restore_windows(k3d::init_value(k3d::application().options().restore_window_geometry()))
{
	// Load UI ...
	return_if_fail(LoadGTKMLTemplate("options.gtkml"));

	if(get_check_button("show_black_box_recorder"))
		get_check_button("show_black_box_recorder")->attach(k3d::check_button::proxy(m_show_black_box_recorder), 0, "");
	if(get_check_button("show_tutorials"))
		get_check_button("show_tutorials")->attach(k3d::check_button::proxy(m_show_tutorial_menu), 0, "");
	if(get_check_button("restore_windows"))
		get_check_button("restore_windows")->attach(k3d::check_button::proxy(m_restore_windows), 0, "");

	if(get_button("apply"))
		get_button("apply")->signal_activate().connect(SigC::slot(*this, &options_dialog::on_apply));
	if(get_button("cancel"))
		get_button("cancel")->signal_activate().connect(SigC::slot(*this, &options_dialog::OnClose));
			
	Entry(control_preferredlanguage).SetText(k3d::application().options().preferred_language().c_str());

	// Default render engine ...
	std::string defaultenginetype;
	std::string defaultengine;
	std::string defaultenginename;
	k3d::application().options().default_render_engine(defaultenginetype, defaultengine, defaultenginename);

	// Build a list of available engines ...
	std::vector<std::string> enginenames;
	const k3d::ioptions::render_engines_t engines = k3d::application().options().render_engines();
	for(k3d::ioptions::render_engines_t::const_iterator engine = engines.begin(); engine != engines.end(); ++engine)
		{
			if(engine->type != "ri")
				continue;
			enginenames.push_back(engine->name);
		}

	Combo(control_defaultengine).SetList(enginenames);
	Combo(control_defaultengine).SetEntryText(defaultenginename);

	// Helper applications ...
	Entry(control_htmlviewer).SetText(k3d::application().options().html_viewer().c_str());
	Entry(control_bitmapviewer).SetText(k3d::application().options().bitmap_viewer().c_str());
	Entry(control_slpreprocessor).SetText(k3d::application().options().sl_preprocessor().c_str());

	// Paths ...
	Entry(control_renderfarmpath).SetText(k3d::application().options().render_farm_path().native_file_string().c_str());

	// Show ourselves ...
	Show();
}

void options_dialog::on_apply()
{
	k3d::application().options().set_blackbox_recorder_at_startup(m_show_black_box_recorder.value());
	k3d::application().options().set_tutorial_menu_at_startup(m_show_tutorial_menu.value());
	k3d::application().options().set_restore_window_geometry(m_restore_windows.value());

	k3d::application().options().set_preferred_language(Entry(control_preferredlanguage).GetText());
	k3d::application().options().set_html_viewer(Entry(control_htmlviewer).GetText());
	k3d::application().options().set_bitmap_viewer(Entry(control_bitmapviewer).GetText());
	k3d::application().options().set_sl_preprocessor(Entry(control_slpreprocessor).GetText());

	// Get the selected engine name ...
	const std::string defaultenginename = Combo(control_defaultengine).GetEntryText();
	assert_warning(defaultenginename.size());

	// Get a list of available engines ...
	const k3d::ioptions::render_engines_t engines = k3d::application().options().render_engines();
	for(k3d::ioptions::render_engines_t::const_iterator engine = engines.begin(); engine != engines.end(); ++engine)
		{
			// Look for the default engine by name ...
			if(defaultenginename == engine->name)
				{
					k3d::application().options().set_default_render_engine(engine->type, engine->engine);
					break;
				}
		}

	k3d::application().options().set_render_farm_path(boost::filesystem::path(Entry(control_renderfarmpath).GetText(), boost::filesystem::native));

	k3d::application().options().commit_options();
	delete this;
}

} // namespace k3d

