// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the clipboard class, which acts as a data source for the primary X selection
		\author Tim Shead (tshead@k-3d.com)
*/

#include "clipboard.h"
#include "gtkml.h"

#include <sdpgtk/sdpgtkevents.h>

#ifdef	SDPWIN32
#include <windows.h>
#endif

namespace k3d
{

/////////////////////////////////////////////////////////////////////////////
// clipboard

clipboard::clipboard() :
	m_have_selection(false)
{
	// Create and load our UI template ...
	std::istringstream uitemplate(
		"<gtkml>"
			"<window type=\"toplevel\" show=\"false\">"
				"<event signal=\"selection-get\" name=\"selectionget\"/>"
				"<event signal=\"selection-clear-event\" name=\"selectionclear\"/>"
			"</window>"
		"</gtkml>\n");

	return_if_fail(load_gtkml(uitemplate, "clipboard builtin template", *this));

	// We want to handle X selections ...
	gtk_selection_add_target(RootWidget(), gdk_atom_intern("CLIPBOARD", false), GDK_SELECTION_TYPE_STRING, 0);
}

clipboard::~clipboard()
{
	RootWidget().Destroy();
}

void clipboard::set_text(const std::string& Text)
{
	m_text = Text;
}

bool clipboard::grab_selection()
{
#ifdef SDPWIN32

	m_have_selection = false;

	return_val_if_fail(OpenClipboard(0), false);
	return_val_if_fail(EmptyClipboard(), false);

	HGLOBAL clipbuffer = GlobalAlloc(GMEM_DDESHARE, m_text.size() + 1);
	return_val_if_fail(clipbuffer, false);

	char* buffer = static_cast<char*>(GlobalLock(clipbuffer));
	return_val_if_fail(buffer, false);

	strcpy(buffer, m_text.c_str());
	GlobalUnlock(clipbuffer);

	return_val_if_fail(SetClipboardData(CF_TEXT, buffer), false);
	return_val_if_fail(CloseClipboard(), false);

	m_have_selection = true;

#else // SDPWIN32

	m_have_selection = gtk_selection_owner_set(RootWidget(), gdk_atom_intern("CLIPBOARD", false), GDK_CURRENT_TIME) ? true : false;

#endif // !SDPWIN32

	return m_have_selection;
}

bool clipboard::have_selection() const
{
	return m_have_selection;
}

void clipboard::OnEvent(::sdpGtkEvent* Event)
{
	assert_warning(Event);
	
	if(Event->Name() == "selectionget")
		on_selection_get(Event);
	else if(Event->Name() == "selectionclear")
		on_selection_clear();
	else
		base::OnEvent(Event);
}

void clipboard::on_selection_get(::sdpGtkEvent* Event)
{
	sdpGtkEventWidgetSelectionGet* event = static_cast<sdpGtkEventWidgetSelectionGet*>(Event);
	gtk_selection_data_set(event->Selection(), GDK_SELECTION_TYPE_STRING, 8, reinterpret_cast<const guchar*>(m_text.c_str()), m_text.size());
}

void clipboard::on_selection_clear()
{
	m_have_selection = false;
}

} // namespace k3d

