#ifndef K3DSDK_OBJECTS_H
#define K3DSDK_OBJECTS_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "uuid.h"
#include "iobject_collection.h"
#include "iobject.h"

#include "utility.h"

#include <vector>

#ifdef	interface
#undef interface
#endif // inteface

namespace k3d
{

// Forward declarations
class idocument;
class iproperty;

/// Defines a collection of objects
typedef iobject_collection::objects_t objects_t;

/// Defines a unary function object that "filters" based on an interface type
template<typename interface_t, typename functor_t>
struct interface_filter_t
{
	explicit interface_filter_t(functor_t Functor) :
		functor(Functor)
	{
	}

	void operator()(iobject* Object)
	{
		interface_t* const interface = dynamic_cast<interface_t*>(Object);
		if(interface)
			functor(interface);
	}

	functor_t functor;
};

/// Utility function for creating interface_filter_t objects
template<typename interface_t, typename functor_t>
struct interface_filter_t<interface_t, functor_t> interface_filter(functor_t Functor)
{
	return interface_filter_t<interface_t, functor_t>(Functor);
}

/*
/// Defines a unary function object that "filters" based on object state
template<typename functor_t>
struct flag_set_filter_t
{
	explicit flag_set_filter_t(const unsigned long Flags, functor_t Functor) :
		flags(Flags),
		functor(Functor)
	{
	}

	void operator()(iobject* Object)
	{
		if(Object->state().QueryState(flags))
			functor(Object);
	}

	unsigned long flags;
	functor_t functor;
};

/// Utility function for creating flag_set_filter_t objects
template<typename functor_t>
struct flag_set_filter_t<functor_t> flag_set_filter(const unsigned long State, functor_t Functor)
{
	return flag_set_filter_t<functor_t>(State, Functor);
}
*/

/// Returns the object that matches the given id (could return NULL)
iobject* find_object(iobject_collection& Objects, const iobject::id_type ObjectID);

/// Returns the object that owns the given property (could return NULL)
iobject* find_object(iobject_collection& Objects, iproperty& Property);

/// Returns the set of objects that match a specific uuid (could return empty set!)
const objects_t find_objects(iobject_collection& Objects, const uuid ClassID);

/// Returns the set of objects that match the given name (be prepared to handle zero, one, or many results)
const objects_t find_objects(iobject_collection& Objects, const std::string ObjectName);

/// Returns the set of objects that implement a specific interface type (could return empty set!)
template<typename interface_t>
const objects_t find_objects(iobject_collection& Objects)
{
	objects_t objects;
	interface_t* interface;
	const objects_t::const_iterator end(Objects.collection().end());
	for(objects_t::const_iterator object = Objects.collection().begin(); object != end; ++object)
		{
			interface = dynamic_cast<interface_t*>(*object);
			if(interface)
				objects.insert(objects.end(), *object);
		}

	return objects;
}

/// Returns a unique object name based on the one supplied
const std::string unique_name(iobject_collection& Objects, const std::string& Name);

/// Deletes a collection of objects, cleaning-up all references and resources (this operation is undo-able, if state change recording is in effect)
void delete_objects(idocument& Document, const objects_t& Objects);

} // namespace k3d

#endif // !K3DSDK_OBJECTS_H


