#ifndef K3DSDK_ISTATE_CHANGE_SET_H
#define K3DSDK_ISTATE_CHANGE_SET_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
	\author Tim Shead (tshead@k-3d.com)
*/

#include "signal_system.h"

namespace k3d
{

// Forward declarations
class istate_container;

/// Abstract interface for an object that stores an atomic set of state changes that can be undone / redone
class istate_change_set
{
public:
	virtual ~istate_change_set() {}

	/// Records an original state that should be restored as part of an "undo" operation
	virtual void record_old_state(k3d::istate_container* const OldState) = 0;
	/// Records a new state that should be restored as part of a "redo" operation
	virtual void record_new_state(k3d::istate_container* const NewState) = 0;

	/// Defines a signal that will be emitted when recording for this change set is complete
	typedef SigC::Signal0<void> recording_done_signal_t;
	virtual recording_done_signal_t& recording_done_signal() = 0;
	/// Defines a signal that will be emitted if this change set is undone
	typedef SigC::Signal0<void> undo_signal_t;
	virtual undo_signal_t& undo_signal() = 0;
	/// Defines a signal that will be emitted if this change set is redone
	typedef SigC::Signal0<void> redo_signal_t;
	virtual redo_signal_t& redo_signal() = 0;

	/// Restores original states and emits the undo_signal
	virtual void undo() = 0;
	/// Restores new states and emits the redo signal
	virtual void redo() = 0;

protected:
	istate_change_set() {}
	istate_change_set(const istate_change_set& RHS) {}
	istate_change_set& operator = (const istate_change_set& RHS) { return *this; }
};

} // namespace k3d

#endif // K3DSDK_ISTATE_CHANGE_SET_H

